﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gpio/detail/gpio_Log.h>

#include "gpio_RegAccessor-soc.tegra.h"
#include "gpio_RegAccessorPinmux-soc.tegra.h"
#include "gpio_DdUtil.h"

namespace nn{
namespace gpio{
namespace driver{
namespace detail{

namespace
{

// PINMUX レジスタの Field Name の列挙体
enum PinmuxFieldName
{
    PinmuxFieldName_PINMUX0  = 0,
    PinmuxFieldName_PINMUX1,
    PinmuxFieldName_PUPD2,
    PinmuxFieldName_PUPD3,
    PinmuxFieldName_TRISTATE,
    PinmuxFieldName_E_INPUT,
    PinmuxFieldName_E_OD,
    PinmuxFieldName_LOCK,
    PinmuxFieldName_IO_RESET,
    PinmuxFieldName_RCV_SEL,
};

enum PinmuxFieldBitFlag
{
    PinmuxFieldBitFlag_PINMUX   = (1 << PinmuxFieldName_PINMUX1) | (1 << PinmuxFieldName_PINMUX0),
    PinmuxFieldBitFlag_PUPD     = (1 << PinmuxFieldName_PUPD3)   | (1 << PinmuxFieldName_PUPD2),
    PinmuxFieldBitFlag_TRISTATE = (1 << PinmuxFieldName_TRISTATE),
    PinmuxFieldBitFlag_E_INPUT  = (1 << PinmuxFieldName_E_INPUT),
    PinmuxFieldBitFlag_E_OD     = (1 << PinmuxFieldName_E_OD),
    PinmuxFieldBitFlag_LOCK     = (1 << PinmuxFieldName_LOCK),
    PinmuxFieldBitFlag_IO_RESET = (1 << PinmuxFieldName_IO_RESET),
    PinmuxFieldBitFlag_RCV_SEL  = (1 << PinmuxFieldName_RCV_SEL),
};


}

// GPIO ライブラリで使用するために Pinmux を設定する関数
void PinmuxRegAccessor::SetGpioPinmux(InternalGpioPadNumber pad, Direction direction)
{
    nn::Bit32* address = nullptr;

    // TORIAEZU: Pinmux 関連はどこかで一括設定をするようになったら削除する。
    // それまでは内部でサポートしているパッドの分だけを列挙しておく。
    switch(pad)
    {
    case InternalGpioPadNumber_GPIO_PR0:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x32bc);
        break;

    case InternalGpioPadNumber_GPIO_PR2:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x32c4);
        break;

    case InternalGpioPadNumber_GPIO_PU0:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3184);
        break;

    case InternalGpioPadNumber_GPIO_PU1:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3188);
        break;

    case InternalGpioPadNumber_GPIO_PU2:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x318c);
        break;

    case InternalGpioPadNumber_GPIO_PU3:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3190);
        break;

    case InternalGpioPadNumber_GPIO_PU4:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3194);
        break;

    case InternalGpioPadNumber_GPIO_PU5:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3198);
        break;

    case InternalGpioPadNumber_GPIO_PU6:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x319c);
        break;

    default:
        // TORIAEZU: システム全体で Pinmux がセットされるまではこのログを出しておく。
         NN_DETAIL_GPIO_INFO("Pinmux for Pad(%d) is not set by GPIO Driver", pad);
         address = nullptr;
    }

    // Pull-up / Pull-down を OFF にして、TRISTATE も OFF に。(GPIO_OE でコントロールするため)
    //(Tegra_K1_TRM_DP06905001v02p.pdf p.268 Table 28 参照)
    if(address != nullptr)
    {
        WriteMasked32( address, 0x00, PinmuxFieldBitFlag_TRISTATE | PinmuxFieldBitFlag_PUPD);

        if(direction == Direction_Output)
        {
            // E_INPUT = 0 (input receiver disabled)
            SetBit<Bit32>(address, 0, PinmuxFieldName_E_INPUT);
        }
        else
        {
            // E_INPUT = 1 (input receiver enabled)
            SetBit<Bit32>(address, 1, PinmuxFieldName_E_INPUT);
        }
        DummyRead(address);
    }
}

} // detail
} // driver
} // gpio
} // nn
