﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/gpio/detail/gpio_Log.h>

#include "gpio_DdUtil.h"
#include "gpio_Interrupt-soc.tegra.h"
#include "gpio_EventHolder-soc.tegra.h"

namespace {

// 割り込みハンドラスレッドに与えるスタックリソース
const size_t InterruptThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_Stack[InterruptThreadStackSize];

}

namespace nn {
namespace gpio {
namespace driver {
namespace detail {

void InterruptNotifier::Initialize() NN_NOEXCEPT
{
    // 二回目からは何もしない。
    if(m_IsInitialized)
    {
        return;
    }

    nn::os::InitializeMultiWait(&m_Waiter);

    // GPIO コントローラーすべての割り込みを Initialize し、MultiWaitHolder に登録する。
    for (int i=0; i<GpioController_NumOfGpioController; i++)
    {
        m_InterruptEvent[i].Initialize(InterruptNameTable[i], true);
        nn::os::InitializeMultiWaitHolder(&m_InterruptEventHolder[i], &m_InterruptEvent[i].GetInstance());
        nn::os::LinkMultiWaitHolder(&m_Waiter, &m_InterruptEventHolder[i]);
        nn::os::SetMultiWaitHolderUserData(&m_InterruptEventHolder[i], static_cast<uintptr_t>(i));
    }

    // 終了処理用内部イベントの設定
    nn::os::InitializeMultiWaitHolder(&m_FinalizeRequestEventHolder, m_FinalizeRequestEvent.GetBase());
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_FinalizeRequestEventHolder);

    // イベント通知スレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&m_WorkThread, InterruptThread, this,
                             s_Stack, InterruptThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(gpio, InterruptHandler))
        );
    nn::os::SetThreadNamePointer(&m_WorkThread, NN_SYSTEM_THREAD_NAME(gpio, InterruptHandler));

    m_IsInitialized = true;
    m_BoundEventCount = 0;

    // 初期化が完了したら、スレッドを走らせる。
    StartThread();
}

void InterruptNotifier::Finalize() NN_NOEXCEPT
{
    // 初期化されていないなら、何もしない。
    if(!m_IsInitialized)
    {
        return;
    }

    // スレッドを停止＆破棄する。
    StopThread();

    for (int i=0; i<GpioController_NumOfGpioController; i++)
    {
        m_InterruptEvent[i].Clear();
        nn::os::UnlinkMultiWaitHolder(&m_InterruptEventHolder[i]);
        nn::os::FinalizeMultiWaitHolder(&m_InterruptEventHolder[i]);
        m_InterruptEvent[i].Finalize();
    }

    nn::os::UnlinkMultiWaitHolder(&m_FinalizeRequestEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_FinalizeRequestEventHolder);

    nn::os::FinalizeMultiWait(&m_Waiter);

    m_IsInitialized = false;
}

nn::Result InterruptNotifier::AddEventHolderToBoundList(PadEventHolder* pEventHolder) NN_NOEXCEPT
{
    // List に EventHolder を追加する
    m_BoundPadEventHolderList.push_back(*pEventHolder);

    return nn::ResultSuccess();
}

void InterruptNotifier::DeleteEventHolderFromBoundList(PadEventHolder* pEventHolder) NN_NOEXCEPT
{
    // 登録された Session を検索する
    for (auto itr = m_BoundPadEventHolderList.begin(); itr != m_BoundPadEventHolderList.end(); ++itr)
    {
        if (itr->GetPadNumber() == pEventHolder->GetPadNumber())
        {
            // 番号が一致しているのに登録されたイベントが異なっている
            NN_SDK_ASSERT(itr->GetSystemEvent() == pEventHolder->GetSystemEvent(), "異なるイベントが登録されています。");

            m_BoundPadEventHolderList.erase(itr);
            return;
        }
    }

    NN_SDK_ASSERT(false, "入力されたイベントが紐付けられたパッドはありません。");
}

void InterruptNotifier::StartThread() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    nn::os::StartThread(&m_WorkThread);
}

void InterruptNotifier::StopThread() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    // 終了処理用内部イベントにシグナルを送りスレッドを終了させる
    m_FinalizeRequestEvent.Signal();
    nn::os::WaitThread(&m_WorkThread);
    nn::os::DestroyThread(&m_WorkThread);
    m_FinalizeRequestEvent.Clear();
}

// サポート済みのパッドの中から通知が来たパッドに対応するイベントを探す関数
nn::os::SystemEventType* InterruptNotifier::SearchInterrupedPadEvent(int interruptIdx) NN_NOEXCEPT
{
    // 登録されたイベントの中から通知が来たパッドに対応するイベントを探す
    for (auto&& e : m_BoundPadEventHolderList)
    {
        // コントローラーが一致しているか
        if (interruptIdx == ConvertInternalGpioPadNumberToControllerNumber(static_cast<InternalGpioPadNumber>(e.GetPadNumber())))
        {
            nn::Bit32* staAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_STA, static_cast<InternalGpioPadNumber>(e.GetPadNumber()), m_GpioVirtualAddress);
            nn::Bit32* enbAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_ENB, static_cast<InternalGpioPadNumber>(e.GetPadNumber()), m_GpioVirtualAddress);
            int bitPosition = ConvertInternalGpioPadNumberToBitPosition(static_cast<InternalGpioPadNumber>(e.GetPadNumber()));

            // INT_STA が立っており、イベントが登録済み、かつ割り込みが enable
            if (GetBit(staAddress, bitPosition) && e.IsBoundEvent() && GetBit(enbAddress, bitPosition))
            {
                // 該当パッドの割り込みを禁止にし、対応するイベントを返す
                nn::Bit32*   address = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_ENB, static_cast<InternalGpioPadNumber>(e.GetPadNumber()), m_GpioVirtualAddress);
                SetBitForTegraMaskedWrite(0, bitPosition, address);
                DummyRead(address);

                e.SetInternalInterruptFlag(false);

                return e.GetSystemEvent();
            }
        }
    }

    // 全パッドを探し終えても見つからない場合、カーネルからのイベントが入った後に Unbind or 割り込み disable されたはずなので、
    // nullptr を返す
    NN_DETAIL_GPIO_WARN("Warning : Cannot find Interrupt Pad ( interrupt block = %d)\n", interruptIdx);
    return nullptr;
}


// CreateThread に渡すメイン関数（要 static）
// InterruptNotifier のメンバに触りたいので中でメンバ関数を呼び出す
void InterruptNotifier::InterruptThread(void* arg) NN_NOEXCEPT
{
    // メンバ変数を触るために処理本体はメンバ関数で
    InterruptNotifier* notifier = reinterpret_cast<InterruptNotifier*>(arg);
    notifier->InterruptThreadBody();
}

void InterruptNotifier::InterruptThreadBody() NN_NOEXCEPT
{
    while (1)
    {
        // 登録されたイベントの発生またはスレッド終了シグナル待ち
        nn::os::MultiWaitHolderType* pEventHolder = nn::os::WaitAny(&m_Waiter);

        // Finalize 中だったら while を抜けて終了
        if (pEventHolder == &m_FinalizeRequestEventHolder)
        {
            break;
        }

        // スレッド終了シグナルでなければすべて割り込みイベントとして対応
        auto interruptIdx = static_cast<int>(nn::os::GetMultiWaitHolderUserData(pEventHolder));
        auto& interruptEvent = m_InterruptEvent[interruptIdx];

        if(!interruptEvent.TryWait())
        {
            continue;
        }

        // GPIO の interrupt は GPIO コントローラーごとに飛んでくる。
        // 対象コントローラーに入っているかつ、INT_STA が立っているパッドを検索し、シグナルする。
        {
            std::lock_guard<nn::os::Mutex> lock(m_InterruptEventMutex);

            // 1. 対象のピンを探し、割込みを禁止にする
            auto padEvent = SearchInterrupedPadEvent(interruptIdx);

            // 2. interruptEvent をクリアする
            interruptEvent.Clear();

            if (padEvent != nullptr)
            {
                // 3. Bind されているイベントをシグナルする
                nn::os::SignalSystemEvent(padEvent);
            }
        }
    }
}

} // detail
} // driver
} // gpio
} // nn
