﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/friends_Setting.h>
#include <nn/friends/detail/friends_ImplSetting.h>
#include <nn/friends/detail/friends_ApiDetail.h>
#include <nn/friends/detail/friends_Async.h>
#include <nn/friends/detail/friends_ShimLibraryGlobal.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

#define GET_USER_IMPL() NN_DETAIL_FRIENDS_GET_IMPL(UserSetting)
#define CONST_GET_USER_IMPL() NN_DETAIL_FRIENDS_CONST_GET_IMPL(UserSetting)

#define GET_FRIEND_IMPL() NN_DETAIL_FRIENDS_GET_IMPL(FriendSetting)
#define CONST_GET_FRIEND_IMPL() NN_DETAIL_FRIENDS_CONST_GET_IMPL(FriendSetting)

namespace nn { namespace friends {

UserSetting::UserSetting() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_CLEAR_IMPL();
}

nn::Result UserSetting::Initialize(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->LoadUserSetting(reinterpret_cast<detail::UserSettingImpl*>(this),
        uid));

    NN_RESULT_SUCCESS;
}

PresencePermission UserSetting::GetPresencePermission() const NN_NOEXCEPT
{
    return static_cast<PresencePermission>(CONST_GET_USER_IMPL()->presencePermission);
}

nn::Result UserSetting::ChangePresencePermission(AsyncContext* outAsync, PresencePermission newValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_USER_IMPL()->uid));

    struct AsyncParam
    {
        detail::UserSettingImpl* outSetting;
        nn::account::Uid uid;
        PresencePermission newValue;
    }
    ap = {reinterpret_cast<detail::UserSettingImpl*>(this), CONST_GET_USER_IMPL()->uid, newValue};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ChangePresencePermission",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ChangePresencePermission(
                ap->uid,
                ap->newValue));

            NN_RESULT_DO(session->LoadUserSetting(ap->outSetting, ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

bool UserSetting::GetFriendRequestReception() const NN_NOEXCEPT
{
    return CONST_GET_USER_IMPL()->friendRequestReception;
}

nn::Result UserSetting::ChangeFriendRequestReception(AsyncContext* outAsync, bool newValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_USER_IMPL()->uid));

    struct AsyncParam
    {
        detail::UserSettingImpl* outSetting;
        nn::account::Uid uid;
        bool newValue;
    }
    ap = {reinterpret_cast<detail::UserSettingImpl*>(this), CONST_GET_USER_IMPL()->uid, newValue};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ChangeFriendRequestReception",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ChangeFriendRequestReception(
                ap->uid,
                ap->newValue));

            NN_RESULT_DO(session->LoadUserSetting(ap->outSetting, ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

const FriendCode& UserSetting::GetFriendCode() NN_NOEXCEPT
{
    return CONST_GET_USER_IMPL()->friendCode;
}

const nn::time::PosixTime& UserSetting::GetFriendCodeNextIssuableTime() NN_NOEXCEPT
{
    return CONST_GET_USER_IMPL()->friendCodeNextIssuableTime;
}

nn::Result UserSetting::IssueFriendCode(AsyncContext* outAsync) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_USER_IMPL()->uid));

    struct AsyncParam
    {
        detail::UserSettingImpl* outSetting;
        nn::account::Uid uid;
    }
    ap = {reinterpret_cast<detail::UserSettingImpl*>(this), CONST_GET_USER_IMPL()->uid};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::IssueFriendCode",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->IssueFriendCode(ap->uid));

            NN_RESULT_DO(session->LoadUserSetting(ap->outSetting, ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

const ProfileExtra& UserSetting::GetExtra() const NN_NOEXCEPT
{
    return reinterpret_cast<const ProfileExtra&>(CONST_GET_USER_IMPL()->profileExtra);
}

ExtraPermission UserSetting::GetPlayLogPermission() const NN_NOEXCEPT
{
    return static_cast<ExtraPermission>(CONST_GET_USER_IMPL()->playLogPermission);
}

nn::Result UserSetting::ChangePlayLogPermission(AsyncContext* outAsync, ExtraPermission newValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_USER_IMPL()->uid));

    struct AsyncParam
    {
        detail::UserSettingImpl* outSetting;
        nn::account::Uid uid;
        ExtraPermission newValue;
    }
    ap = {reinterpret_cast<detail::UserSettingImpl*>(this), CONST_GET_USER_IMPL()->uid, newValue};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ChangePlayLogPermission",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ChangePlayLogPermission(
                ap->uid,
                ap->newValue));

            NN_RESULT_DO(session->LoadUserSetting(ap->outSetting, ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result UserSetting::ClearPlayLog(AsyncContext* outAsync) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_USER_IMPL()->uid));

    struct AsyncParam
    {
        detail::UserSettingImpl* outSetting;
        nn::account::Uid uid;
    }
    ap = {reinterpret_cast<detail::UserSettingImpl*>(this), CONST_GET_USER_IMPL()->uid};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ClearPlayLog",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ClearPlayLog(ap->uid));

            NN_RESULT_DO(session->LoadUserSetting(ap->outSetting, ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

FriendSetting::FriendSetting() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_CLEAR_IMPL();
}

nn::Result FriendSetting::Initialize(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->LoadFriendSetting(reinterpret_cast<detail::FriendSettingImpl*>(this),
        uid, accountId));

    NN_RESULT_SUCCESS;
}

bool FriendSetting::GetNewlyFlag() const NN_NOEXCEPT
{
    return CONST_GET_FRIEND_IMPL()->newly;
}

nn::Result FriendSetting::DropNewlyFlag() NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_FRIEND_IMPL()->uid));

    NN_RESULT_DO(session->DropFriendNewlyFlag(
        CONST_GET_FRIEND_IMPL()->uid,
        CONST_GET_FRIEND_IMPL()->accountId));

    NN_RESULT_DO(session->LoadFriendSetting(reinterpret_cast<detail::FriendSettingImpl*>(this),
        CONST_GET_FRIEND_IMPL()->uid, CONST_GET_FRIEND_IMPL()->accountId));

    NN_RESULT_SUCCESS;
}

bool FriendSetting::GetFavoriteFlag() const NN_NOEXCEPT
{
    return CONST_GET_FRIEND_IMPL()->favorite;
}

nn::Result FriendSetting::ChangeFavoriteFlag(AsyncContext* outAsync, bool newValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_FRIEND_IMPL()->uid));

    struct AsyncParam
    {
        detail::FriendSettingImpl* outSetting;
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        bool newValue;
    }
    ap = {reinterpret_cast<detail::FriendSettingImpl*>(this),
        CONST_GET_FRIEND_IMPL()->uid, CONST_GET_FRIEND_IMPL()->accountId, newValue};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ChangeFriendFavoriteFlag",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ChangeFriendFavoriteFlag(ap->uid, ap->accountId, ap->newValue));

            NN_RESULT_DO(session->LoadFriendSetting(ap->outSetting, ap->uid, ap->accountId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

bool FriendSetting::GetOnlineNotificationFlag() const NN_NOEXCEPT
{
    return CONST_GET_FRIEND_IMPL()->onlineNotification;
}

nn::Result FriendSetting::ChangeOnlineNotificationFlag(AsyncContext* outAsync, bool newValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_FRIEND_IMPL()->uid));

    struct AsyncParam
    {
        detail::FriendSettingImpl* outSetting;
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        bool newValue;
    }
    ap = {reinterpret_cast<detail::FriendSettingImpl*>(this),
        CONST_GET_FRIEND_IMPL()->uid, CONST_GET_FRIEND_IMPL()->accountId, newValue};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ChangeFriendOnlineNotificationFlag",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ChangeFriendOnlineNotificationFlag(ap->uid, ap->accountId, ap->newValue));

            NN_RESULT_DO(session->LoadFriendSetting(ap->outSetting, ap->uid, ap->accountId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

}}
