﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/friends_AsyncContext.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/detail/friends_AsyncContextInternal.h>
#include <mutex>

namespace nn { namespace friends {

AsyncContext::AsyncContext() NN_NOEXCEPT :
    m_Mutex(true),
    m_pInternal(nullptr)
{
}

AsyncContext::~AsyncContext() NN_NOEXCEPT
{
    if (m_pInternal)
    {
        detail::AsyncContextInternal::DeleteInstance(m_pInternal);
    }
}

nn::Result AsyncContext::GetSystemEvent(nn::os::SystemEvent *outEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!m_pInternal)
    {
        NN_RESULT_THROW(ResultNotCalled());
    }

    m_pInternal->AttachCompletionEventReadableHandle(outEvent);

    NN_RESULT_SUCCESS;
}

nn::Result AsyncContext::Cancel() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!m_pInternal)
    {
        NN_RESULT_THROW(ResultNotCalled());
    }

    m_pInternal->Cancel();

    NN_RESULT_SUCCESS;
}

nn::Result AsyncContext::HasDone(bool* out) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!m_pInternal)
    {
        NN_RESULT_THROW(ResultNotCalled());
    }

    *out = m_pInternal->IsCompleted();

    NN_RESULT_SUCCESS;
}

nn::Result AsyncContext::GetResult() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!m_pInternal)
    {
        NN_RESULT_THROW(ResultNotCalled());
    }

    return m_pInternal->GetResult();
}

void AsyncContext::SetInternal(detail::AsyncContextInternal* pInternal) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInternal);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_pInternal)
    {
        detail::AsyncContextInternal::DeleteInstance(m_pInternal);
    }

    m_pInternal = pInternal;
}

}}
