﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/friends_ApiFriendRequest.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/detail/friends_ApiDetail.h>
#include <nn/friends/detail/friends_Async.h>
#include <nn/friends/detail/friends_ShimLibraryGlobal.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace friends {

nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId,
    const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        InAppScreenName inAppScreenName;
        InAppScreenName myInAppScreenName;
    }
    ap = {uid, accountId, inAppScreenName, myInAppScreenName};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SendFriendRequestForApplication",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            InAppScreenName inAppScreenNameNormalized;
            NN_RESULT_DO(detail::NormalizeInAppScreenName(&inAppScreenNameNormalized, ap->inAppScreenName));

            InAppScreenName myInAppScreenNameNormalized;
            NN_RESULT_DO(detail::NormalizeInAppScreenName(&myInAppScreenNameNormalized, ap->myInAppScreenName));

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SendFriendRequestForApplication(
                ap->uid,
                ap->accountId,
                inAppScreenNameNormalized,
                myInAppScreenNameNormalized, 0));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetFacedFriendRequestRegistrationKey(FacedFriendRequestRegistrationKey* outKey,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outKey);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->GetFacedFriendRequestRegistrationKey(outKey, uid));

    NN_RESULT_SUCCESS;
}

nn::Result AddFacedFriendRequest(const nn::account::Uid& uid,
    const FacedFriendRequestRegistrationKey& key,
    const nn::account::Nickname& nickname, const void* image, size_t imageSize,
    const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(image);
    NN_SDK_REQUIRES_MINMAX(imageSize, 1u, ProfileImageSizeMax);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    InAppScreenName inAppScreenNameNormalized;
    NN_RESULT_DO(detail::NormalizeInAppScreenName(&inAppScreenNameNormalized, inAppScreenName));

    InAppScreenName myInAppScreenNameNormalized;
    NN_RESULT_DO(detail::NormalizeInAppScreenName(&myInAppScreenNameNormalized, myInAppScreenName));

    NN_RESULT_DO(session->AddFacedFriendRequestForApplication(
        uid,
        key,
        nickname,
        nn::sf::InBuffer(static_cast<const char*>(image), imageSize),
        inAppScreenNameNormalized,
        myInAppScreenNameNormalized, 0));

    NN_RESULT_SUCCESS;
}

}}
