﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/json/friends_JsonInputStream.h>
#include <nn/friends/friends_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

JsonInputStream::JsonInputStream() NN_NOEXCEPT :
    m_String(nullptr),
    m_StringBufferSize(0),
    m_Stream(nullptr),
    m_StreamBufferSize(0),
    m_StringSize(0),
    m_Read(0),
    m_ReadTotal(0),
    m_PositionInStream(0),
    m_LastError(nn::ResultSuccess()),
    m_Cancelable(nullptr)
{
}

JsonInputStream::~JsonInputStream() NN_NOEXCEPT
{
}

void JsonInputStream::SetBuffer(Ch* stringBuffer, size_t stringBufferSize, Ch* streamBuffer, size_t streamBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stringBuffer);
    NN_SDK_REQUIRES(stringBufferSize >= 8);
    NN_SDK_REQUIRES_NOT_NULL(streamBuffer);
    NN_SDK_REQUIRES(streamBufferSize >= 8);

    m_String = stringBuffer;
    m_StringBufferSize = stringBufferSize;
    m_Stream = streamBuffer;
    m_StreamBufferSize = streamBufferSize;
}

void JsonInputStream::SetCancelable(const detail::service::util::Cancelable* cancelable) NN_NOEXCEPT
{
    m_Cancelable = cancelable;
}

const std::atomic<bool>* JsonInputStream::GetOverflowDetector() NN_NOEXCEPT
{
    return &m_OverflowDetector;
}

nn::Result JsonInputStream::GetLastError() const NN_NOEXCEPT
{
    return m_LastError;
}

JsonInputStream::Ch JsonInputStream::Peek() NN_NOEXCEPT
{
    if (m_PositionInStream >= m_Read)
    {
        NN_SDK_ASSERT(m_PositionInStream == m_Read);

        if (!Fill())
        {
            return '\0';
        }
    }

    return m_Stream[m_PositionInStream];
}

JsonInputStream::Ch JsonInputStream::Take() NN_NOEXCEPT
{
    Ch c = Peek();
    m_PositionInStream++;

    return c;
}

size_t JsonInputStream::Tell() const NN_NOEXCEPT
{
    return (m_ReadTotal - m_Read) + m_PositionInStream;
}

JsonInputStream::Ch* JsonInputStream::PutBegin() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_String);

    m_StringSize = 0;
    m_OverflowDetector = false;

    return m_String;
}

void JsonInputStream::Put(Ch c) NN_NOEXCEPT
{
    if (m_StringSize >= m_StringBufferSize)
    {
        return;
    }

    m_String[m_StringSize++] = c;
}

size_t JsonInputStream::PutEnd(Ch* begin) NN_NOEXCEPT
{
    NN_SDK_ASSERT(begin == m_String);
    NN_UNUSED(begin);

    if (m_String[m_StringSize - 1] != '\0')
    {
        m_OverflowDetector = true;
        m_String[m_StringSize - 1] = '\0';
    }

    return m_StringSize;
}

bool JsonInputStream::IsCanceled() const NN_NOEXCEPT
{
    if (m_Cancelable)
    {
        return m_Cancelable->IsCanceled();
    }

    return false;
}

bool JsonInputStream::Fill() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Stream);

    m_PositionInStream = 0;

    if (m_LastError.IsSuccess())
    {
        m_LastError = FillImpl(&m_Read, m_Stream, m_StreamBufferSize);
    }

    if (m_LastError.IsFailure())
    {
        m_Read = 0;
    }

    m_ReadTotal += m_Read;

    return (m_Read > 0);
}

JsonMemoryInputStream::JsonMemoryInputStream() NN_NOEXCEPT :
    m_Memory(nullptr),
    m_Size(0),
    m_IsOpened(false),
    m_Position(0)
{
}

JsonMemoryInputStream::~JsonMemoryInputStream() NN_NOEXCEPT
{
}

nn::Result JsonMemoryInputStream::Open(const void* memory, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(memory);
    NN_SDK_REQUIRES(size > 0);

    NN_SDK_ASSERT(!m_IsOpened);

    m_Memory = memory;
    m_Size = size;
    m_Position = 0;

    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

void JsonMemoryInputStream::Close() NN_NOEXCEPT
{
    m_IsOpened = false;
}

nn::Result JsonMemoryInputStream::FillImpl(size_t* outRead, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpened);

    if (IsCanceled())
    {
        NN_RESULT_THROW(ResultCanceled());
    }

    size_t remainSize = m_Size - m_Position;

    if (remainSize == 0)
    {
        *outRead = 0;
        NN_RESULT_SUCCESS;
    }

    size_t copySize = (bufferSize < remainSize ? bufferSize : remainSize);

    const Bit8* memory = reinterpret_cast<const Bit8*>(m_Memory);

    std::memcpy(buffer, &memory[m_Position], copySize);
    m_Position += copySize;

    *outRead = copySize;

    NN_RESULT_SUCCESS;
}

}}}}}
