﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskGetProfileList.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerProfileResourceList.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

GetProfileListTask::GetProfileListTask() NN_NOEXCEPT :
    m_IsParameterSet(false),
    m_IsParameterSetWithFriendCode(false),
    m_OutProfiles(nullptr),
    m_OutProfileExtras(nullptr)
{
}

nn::Result GetProfileListTask::SetParameter(ProfileImpl* outProfiles,
    const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(count > 0 && count <= GetProfileCountMax, ResultInvalidArgument());

    m_OutProfiles = outProfiles;
    m_AccountIds = accountIds;
    m_Count = count;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileListTask::SetParameter(ProfileExtraImpl* outProfiles,
    const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(count > 0 && count <= GetProfileCountMax, ResultInvalidArgument());

    m_OutProfileExtras = outProfiles;
    m_AccountIds = accountIds;
    m_Count = count;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileListTask::SetParameter(ProfileExtraImpl* outProfile,
    const FriendCode& friendCode) NN_NOEXCEPT
{
    const int FriendCodeSize = static_cast<int>(sizeof (friendCode.value));

    // TODO: 検証
    NN_RESULT_THROW_UNLESS(friendCode.value[0] != '\0' && nn::util::Strnlen(friendCode.value, FriendCodeSize) < FriendCodeSize,
        ResultInvalidArgument());

    m_OutProfileExtras = outProfile;
    m_Count = 1;
    m_FriendCode = friendCode;

    m_IsParameterSetWithFriendCode = true;

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileListTask::Main() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsParameterSet || m_IsParameterSetWithFriendCode);

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    if (m_IsParameterSetWithFriendCode)
    {
        NN_RESULT_DO(StepGetProfileExtraWithFriendCode());
    }
    else if (m_OutProfiles)
    {
        NN_RESULT_DO(StepGetProfile());
    }
    else if (m_OutProfileExtras)
    {
        NN_RESULT_DO(StepGetProfileExtra());
    }
    else
    {
        NN_ABORT("");
    }

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileListTask::StepGetProfile() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] GetProfileListTask::StepGetProfile ...\n");

    nn::account::NetworkServiceAccountId accountIds[GetProfileCountMax] = {};
    int actualCount;

    GetUniqueArray(&actualCount, accountIds, m_AccountIds, m_Count);

    detail::service::json::JsonHttpInputStream stream;

    char url[2048] = {};

    int length = nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users?filter.id.$in=", WebApi::GetFqdn());

    for (int i = 0; i < actualCount; i++)
    {
        length += nn::util::SNPrintf(&url[length], sizeof (url) - length, "%016llx", accountIds[i].id);

        if (i != actualCount - 1)
        {
            url[length++] = ',';
        }
    }

    NN_RESULT_DO(stream.Open("GET", url));

    ProfileResourceListHandler handler;
    handler.Initialize(m_OutProfiles, m_AccountIds, m_Count);

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileListTask::StepGetProfileExtra() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] GetProfileListTask::StepGetProfileExtra ...\n");

    nn::account::NetworkServiceAccountId accountIds[GetProfileCountMax] = {};
    int actualCount;

    GetUniqueArray(&actualCount, accountIds, m_AccountIds, m_Count);

    detail::service::json::JsonHttpInputStream stream;

    char url[2048] = {};

    int length = nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users?filter.id.$in=", WebApi::GetFqdn());

    for (int i = 0; i < actualCount; i++)
    {
        length += nn::util::SNPrintf(&url[length], sizeof (url) - length, "%016llx", accountIds[i].id);

        if (i != actualCount - 1)
        {
            url[length++] = ',';
        }
    }

    NN_RESULT_DO(stream.Open("GET", url));

    ProfileResourceListHandler handler;
    handler.Initialize(m_OutProfileExtras, m_AccountIds, m_Count);

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileListTask::StepGetProfileExtraWithFriendCode() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] GetProfileListTask::StepGetProfileExtraWithFriendCode ...\n");

    detail::service::json::JsonHttpInputStream stream;

    char url[256] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users?filter.links.friendCode.id.$in=%s", WebApi::GetFqdn(), m_FriendCode.value);

    NN_RESULT_DO(stream.Open("GET", url));

    ProfileResourceListHandler handler;
    handler.Initialize(m_OutProfileExtras);

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_THROW_UNLESS(m_OutProfileExtras[0].data.isValid, ResultNetworkServiceAccountNotExistsGeneral());

    NN_RESULT_SUCCESS;
}

void GetProfileListTask::GetUniqueArray(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
    const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    int actualCount = 0;

    for (int i = 0; i < count; i++)
    {
        bool isFound = false;

        for (int n = 0; n < actualCount; n++)
        {
            if (accountIds[i] == outAccountIds[n])
            {
                isFound = true;
                break;
            }
        }

        if (!isFound)
        {
            outAccountIds[actualCount++] = accountIds[i];
        }
    }

    *outCount = actualCount;
}

}}}}}
