﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerFriendResourceListForPresence.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPresence.h>
#include <nn/friends/detail/service/core/friends_FriendPresenceManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

FriendResourceListForPresenceHandler::FriendResourceListForPresenceHandler() NN_NOEXCEPT :
    m_Uid(nn::account::InvalidUid)
{
}

FriendResourceListForPresenceHandler::~FriendResourceListForPresenceHandler() NN_NOEXCEPT
{
}

void FriendResourceListForPresenceHandler::Initialize(const nn::account::Uid& uid) NN_NOEXCEPT
{
    m_Uid = uid;
}

bool FriendResourceListForPresenceHandler::OnStartObject(const detail::service::json::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.items[*]"))
    {
        std::memset(&m_Resource, 0, sizeof (FriendResource));
        std::memset(&m_Bits, 0, sizeof (m_Bits));
    }

    return true;
}

bool FriendResourceListForPresenceHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Match("$.items[*]"))
    {
        // オフライン状態の時はログアウト時刻を最終更新時刻扱いにする。
        if (m_Bits.presenceLogoutTime && m_Resource.presence.data.status == PresenceStatus_Offline)
        {
            m_Resource.presence.data.lastUpdateTime = m_Resource.logoutTime;
        }
        if (m_Bits.accountId)
        {
            FriendPresenceManager::GetInstance().UpdatePresence(m_Uid, m_Resource);
        }
    }

    return true;
}

bool FriendResourceListForPresenceHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.presenceLastUpdateTime &&
        jsonPath.Match("$.items[*].friend.presence.updatedAt"))
    {
        m_Resource.presence.data.lastUpdateTime.value = value.s;
        m_Bits.presenceLastUpdateTime = true;
    }
    else if (!m_Bits.presenceLogoutTime &&
        jsonPath.Match("$.items[*].friend.presence.logoutAt"))
    {
        m_Resource.logoutTime.value = value.s;
        m_Bits.presenceLogoutTime = true;
    }

    return true;
}

bool FriendResourceListForPresenceHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    if (!m_Bits.accountId &&
        jsonPath.Match("$.items[*].friendId"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_Resource.accountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.presenceStatus &&
        jsonPath.Match("$.items[*].friend.presence.state"))
    {
        m_Resource.presence.data.status = ParameterConverter::ConvertPresenceStatus(value);
        m_Bits.presenceStatus = true;
    }
    else if (!m_Bits.presenceAppId &&
        jsonPath.Match("$.items[*].friend.presence.extras.friends.appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Resource.presence.data.lastPlayedApp.appId = appId;
            m_Bits.presenceAppId = true;
        }
    }
    else if (!m_Bits.presencePresenceGroupId &&
        jsonPath.Match("$.items[*].friend.presence.extras.friends.appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Resource.presence.data.lastPlayedApp.presenceGroupId = presenceGroupId;
            m_Bits.presencePresenceGroupId = true;
        }
    }
    else if (!m_Bits.presenceAppField &&
        jsonPath.Match("$.items[*].friend.presence.extras.friends.appField"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] The presence string is overflowed.\n");
        }

        m_Bits.presenceAppField = ParseFriendPresenceAppField(value, length);
    }

    return true;
}

bool FriendResourceListForPresenceHandler::ParseFriendPresenceAppField(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '{' || json[length - 1] != '}')
    {
        return false;
    }

    PresenceHandler handler;
    handler.Initialize(m_Resource.presence.data.appField, PresenceAppFieldSize);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[PresenceAppFieldSize];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

}}}}}
