﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerFriendCandidateResourceList.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

FriendCandidateResourceListHandler::FriendCandidateResourceListHandler() NN_NOEXCEPT :
    m_OutCount(0),
    m_OutCandidates(nullptr),
    m_Count(0),
    m_Index(0)
{
}

void FriendCandidateResourceListHandler::Initialize(int* outCount, FriendCandidateImpl* outCandidates,
    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outCandidates);
    NN_SDK_REQUIRES(count > 0);

    m_OutCount = outCount;
    m_OutCandidates = outCandidates;

    m_Count = count;

    std::memset(m_OutCandidates, 0, sizeof (FriendCandidateImpl) * m_Count);
}

bool FriendCandidateResourceListHandler::OnStartObject(const detail::service::json::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.items[*]"))
    {
        std::memset(&m_Candidate, 0, sizeof (m_Candidate));
        std::memset(&m_Bits, 0, sizeof (m_Bits));
    }

    return true;
}

bool FriendCandidateResourceListHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Match("$.items[*]"))
    {
        // アプリケーション ID、カタログ ID は必須ではない。
        if (m_Bits.accountId && m_Bits.playRecordName && m_Bits.playRecordName && m_Bits.playRecordTime)
        {
            if (m_Bits.catalogId)
            {
                m_Candidate.data.type = FriendCandidateType_ExternalApplication;
            }

            m_Candidate.data.isValid = true;

            if (m_Index < m_Count)
            {
                std::memcpy(&m_OutCandidates[m_Index], &m_Candidate, sizeof (FriendCandidateImpl));
                m_Index++;
            }
        }
    }

    return true;
}

bool FriendCandidateResourceListHandler::OnEndArray(const detail::service::json::JsonPath& jsonPath, size_t numElements) NN_NOEXCEPT
{
    NN_UNUSED(numElements);

    if (jsonPath.Compare("$.items[]"))
    {
        *m_OutCount = m_Index;
    }

    return true;
}

bool FriendCandidateResourceListHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.playRecordTime &&
        jsonPath.Match("$.items[*].createdAt"))
    {
        m_Candidate.data.playRecord.time.value = value.s;
        m_Bits.playRecordTime = true;
    }

    return true;
}

bool FriendCandidateResourceListHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(isOverflowed);

    if (!m_Bits.accountId &&
        jsonPath.Match("$.items[*].requestedClientFriendId"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_Candidate.data.accountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.catalogId &&
        jsonPath.Match("$.items[*].client.catalogId"))
    {
        if (ParameterConverter::ConvertExternalApplicationCatalogId(&m_Candidate.data.catalogId, value))
        {
            m_Bits.catalogId = true;
        }
    }
    else if (!m_Bits.playRecordAppId &&
        jsonPath.Match("$.items[*].client.applicationId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Candidate.data.playRecord.appInfo.appId = appId;
            m_Bits.playRecordAppId = true;
        }
    }
    else if (!m_Bits.playRecordName &&
        jsonPath.Match("$.items[*].clientFriend.nickname"))
    {
        nn::util::CopyUtf8String(m_Candidate.data.playRecord.name.name,
            sizeof (m_Candidate.data.playRecord.name.name), value, length, InAppScreenNameLengthMax);
        m_Bits.playRecordName = true;
    }
    else if (!m_Bits.playRecordMyName &&
        jsonPath.Match("$.items[*].clientUser.nickname"))
    {
        nn::util::CopyUtf8String(m_Candidate.data.playRecord.myName.name,
            sizeof (m_Candidate.data.playRecord.myName.name), value, length, InAppScreenNameLengthMax);
        m_Bits.playRecordMyName = true;
    }

    return true;
}

}}}}}
