﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerExternalApplicationCatalog.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    struct LanguagePair
    {
        const char* siglo;
        const char* na;
    };

    const LanguagePair LanguageConvertTable[] =
    {
        {"ja",      "ja-JP"},
        {"en-US",   "en-US"},
        {"fr",      "fr-FR"},
        {"de",      "de-DE"},
        {"it",      "it-IT"},
        {"es",      "es-ES"},
        {"zh-CN",   "zh-CN"},
        {"ko",      "ko-KR"},
        {"nl",      "nl-NL"},
        {"pt",      "pt-PT"},
        {"ru",      "ru-RU"},
        {"zh-TW",   "zh-TW"},
        {"en-GB",   "en-GB"},
        {"fr-CA",   "fr-CA"},
        {"es-419",  "es-MX"},
        {"zh-Hans", "zh-Hans"},
        {"zh-Hant", "zh-Hant"},
        {"", ""}
    };

    // MEMO: ビルドターゲットによってファイル名を変えること。
    const char* FileName = "nx";
}

ExternalApplicationCatalogHandler::ExternalApplicationCatalogHandler() NN_NOEXCEPT :
    m_OutCatalog(nullptr),
    m_Bits(),
    m_LanguageNaLocale(nullptr)
{
}

void ExternalApplicationCatalogHandler::Initialize(ExternalApplicationCatalog* outCatalog, const nn::settings::LanguageCode& language) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCatalog);

    m_OutCatalog = outCatalog;
    m_Language = language;

    std::memset(m_OutCatalog, 0, sizeof (ExternalApplicationCatalog));

    for (int i = 0; LanguageConvertTable[i].siglo[0] != '\0'; i++)
    {
        if (nn::util::Strncmp(m_Language.string, LanguageConvertTable[i].siglo, sizeof (m_Language.string)) == 0)
        {
            m_LanguageNaLocale = LanguageConvertTable[i].na;
            break;
        }
    }
}

bool ExternalApplicationCatalogHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Compare("$"))
    {
        if (m_Bits.catalogId && m_Bits.icon)
        {
            nn::util::SNPrintf(m_OutCatalog->iconImageUrl.value, sizeof (m_OutCatalog->iconImageUrl.value),
                "https://c-%%.accounts.nintendo.com/v1/apps/%016llx%016llx/%s.jpg",
                m_OutCatalog->id.value[0], m_OutCatalog->id.value[1], FileName);
        }
    }

    return true;
}

bool ExternalApplicationCatalogHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(isOverflowed);

    if (!m_Bits.catalogId &&
        jsonPath.Compare("$.catalog_id"))
    {
        if (ParameterConverter::ConvertExternalApplicationCatalogId(&m_OutCatalog->id, value))
        {
            m_Bits.catalogId = true;
        }
    }
    else if (!m_Bits.name &&
        jsonPath.Match("$.application_names.*"))
    {
        if (m_LanguageNaLocale &&
            nn::util::Strncmp(m_LanguageNaLocale, jsonPath.ToString() + sizeof ("$.application_names.") - 1, 8) == 0)
        {
            nn::util::CopyUtf8String(m_OutCatalog->name.name, sizeof (m_OutCatalog->name.name), value, length, ExternalApplicationNameLengthMax);
            m_OutCatalog->name.language = m_Language;
            m_Bits.name = true;
        }
    }
    else if (!m_Bits.icon &&
        jsonPath.Match("$.enable_icons[*]"))
    {
        if (nn::util::Strncmp(value, FileName, static_cast<int>(length) + 1) == 0)
        {
            m_Bits.icon = true;
        }
    }

    return true;
}

}}}}}
