﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_BlockedUserListManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* FileName = "block.cache";
}

BlockedUserListManager::BlockedUserListManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentUid(nn::account::InvalidUid),
    m_Count(0)
{
}

nn::Result BlockedUserListManager::GetBlockedUserList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outAccountIds);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outAccountIds, 0, sizeof (nn::account::NetworkServiceAccountId) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = offset; i < m_Count; i++)
    {
        outAccountIds[actualCount++] = m_Records[i].resource.accountId;

        if (actualCount == count)
        {
            break;
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result BlockedUserListManager::GetBlockedUserList(int* outCount, BlockedUserImpl* outUsers,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outUsers);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outUsers, 0, sizeof (BlockedUserImpl) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = offset; i < m_Count; i++)
    {
        BlockedUserImpl& impl = outUsers[actualCount++];

        impl.data.uid = uid;
        impl.data.accountId = m_Records[i].resource.accountId;
        impl.data.nickname = m_Records[i].resource.nickname;
        impl.data.profileImageUrl = m_Records[i].resource.profileImageUrl;
        impl.data.blockReason = m_Records[i].resource.blockReason;
        impl.data.routeInfo = m_Records[i].resource.routeInfo;
        impl.data.isValid = true;

        if (actualCount == count)
        {
            break;
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result BlockedUserListManager::UpdateBlockedUserList(const nn::account::Uid& uid, const BlockedUserResource* resources, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(resources);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    for (int i = 0; i < count; i++)
    {
        Record record = {};

        record.resource = resources[i];

        std::memcpy(&m_Records[i], &record, sizeof (Record));
    }

    m_Count = count;

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result BlockedUserListManager::DeleteUser(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchRecord(accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotBlocked());

    m_Count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, index);

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

void BlockedUserListManager::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_CurrentUid = nn::account::InvalidUid;
}

nn::Result BlockedUserListManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            m_Count = 0;
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result BlockedUserListManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. ((size % sizeof (Record)) != 0)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    int count = static_cast<int>(size / sizeof (Record));

    if (count > BlockedUserCountMax)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (count > BlockedUserCountMax)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (Record) * count));
    m_Count = count;

    NN_RESULT_SUCCESS;
}

nn::Result BlockedUserListManager::Save() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(m_CurrentUid);

    // 新規にデータを作成する際、ストレージの容量不足にならないよう、不要なファイルを削除する。
    NN_RESULT_DO(AccountStorageManager::GetInstance().DeleteUnmanagedNetworkServiceAccountDirectory());
    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    if (m_Count == 0)
    {
        NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());
        NN_RESULT_SUCCESS;
    }

    {
        NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (Record) * m_Count));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (Record) * m_Count,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    NN_RESULT_SUCCESS;
}

int BlockedUserListManager::SearchRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (m_Records[i].resource.accountId == accountId)
        {
            return i;
        }
    }

    return -1;
}

}}}}}
