﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_AccountStorageManager.h>
#include <nn/util/util_Endian.h>

#define ENABLE_HOSTFS 0 // NOLINT(preprocessor/const)

#if ENABLE_HOSTFS == 1

#define HOSTFS_ROOT "C:\\siglo\\friends\\"

#endif

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    nn::util::Uuid& ConvertId(nn::util::Uuid* out, const nn::account::Uid& in) NN_NOEXCEPT
    {
        nn::account::Uid id = {};
        nn::util::StoreBigEndian(&id._data[0], in._data[0]);
        nn::util::StoreBigEndian(&id._data[1], in._data[1]);

        std::memcpy(out, &id, sizeof (nn::util::Uuid));

        return *out;
    }

    nn::account::Uid& ConvertId(nn::account::Uid* out, const nn::util::Uuid& in) NN_NOEXCEPT
    {
        nn::account::Uid id = {};
        std::memcpy(&id, &in, sizeof (nn::account::Uid));

        out->_data[0] = nn::util::LoadBigEndian(&id._data[0]);
        out->_data[1] = nn::util::LoadBigEndian(&id._data[1]);

        return *out;
    }

#if ENABLE_HOSTFS == 1

    void MakeHostFsRoot(char* buffer, size_t size, const nn::account::Uid& uid) NN_NOEXCEPT
    {
        nn::util::Uuid uuid;
        char str[nn::util::Uuid::StringSize];

        NN_ABORT_UNLESS(nn::util::SNPrintf(buffer, size, HOSTFS_ROOT "%s",
            ConvertId(&uuid, uid).ToString(str, sizeof (str))) < static_cast<int>(size));
    }

    void MakeRoot(char* buffer, size_t size, const nn::account::Uid& uid) NN_NOEXCEPT
    {
        nn::util::Uuid uuid;
        char str[nn::util::Uuid::StringSize];

        NN_ABORT_UNLESS(nn::util::SNPrintf(buffer, size, "friends:/%s",
            ConvertId(&uuid, uid).ToString(str, sizeof (str))) < static_cast<int>(size));
    }

#endif

    void Make(char* buffer, size_t size, const char* name) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(nn::util::SNPrintf(buffer, size, "friends:/%s", name) < static_cast<int>(size));
    }

    void Make(char* buffer, size_t size, nn::account::NetworkServiceAccountId accountId, const char* name) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(nn::util::SNPrintf(buffer, size, "friends:/%016llx/%s", accountId.id, name) < static_cast<int>(size));
    }
}

AccountStorageManager::AccountStorageManager() NN_NOEXCEPT :
    m_Mutex(false),
    m_Uid(nn::account::InvalidUid),
    m_IsNetworkServiceAccountAvailable(false),
    m_SystemSaveDataId(0x8000000000000080)
{
}

void AccountStorageManager::SetSystemSaveDataId(nn::fs::SystemSaveDataId id) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_SystemSaveDataId = id;
}

nn::Result AccountStorageManager::Mount(const nn::account::Uid& uid, bool createIfNotExists) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&manager, uid));

    nn::account::NetworkServiceAccountId accountId = {};

    bool isNetworkServiceAccountAvailable = false;
    NN_RESULT_DO(manager.IsNetworkServiceAccountAvailable(&isNetworkServiceAccountAvailable));

    if (isNetworkServiceAccountAvailable)
    {
        NN_RESULT_DO(manager.GetNetworkServiceAccountId(&accountId));
    }

    m_Mutex.Lock();

    bool isSuccess = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            m_Mutex.Unlock();
        }
    };

#if ENABLE_HOSTFS == 1

    char root[128] = {};
    MakeHostFsRoot(root, sizeof (root), uid);

    NN_RESULT_TRY(nn::fs::MountHost("friends", root))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            if (createIfNotExists)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("friends", HOSTFS_ROOT));

                MakeRoot(root, sizeof (root), uid);
                NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::CreateDirectoryRecursively(root));

                nn::fs::Unmount("friends");

                MakeHostFsRoot(root, sizeof (root), uid);
                NN_RESULT_DO(nn::fs::MountHost("friends", root));
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY;

#else

    NN_RESULT_TRY(nn::fs::MountSystemSaveData("friends", m_SystemSaveDataId, uid))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            if (createIfNotExists)
            {
                NN_RESULT_DO(nn::fs::CreateSystemSaveData(m_SystemSaveDataId, uid, StorageSize, StorageJournalSize, 0));
                NN_RESULT_DO(nn::fs::MountSystemSaveData("friends", m_SystemSaveDataId, uid));
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY;

#endif

    m_Uid = uid;
    m_AccountId = accountId;
    m_IsNetworkServiceAccountAvailable = isNetworkServiceAccountAvailable;

    isSuccess = true;

    NN_RESULT_SUCCESS;
}

void AccountStorageManager::Unmount() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    nn::fs::Unmount("friends");

    m_Mutex.Unlock();
}

nn::Result AccountStorageManager::Commit() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

#if ENABLE_HOSTFS == 0

    NN_RESULT_DO(nn::fs::CommitSaveData("friends"));

#endif

    NN_RESULT_SUCCESS;
}

nn::Result AccountStorageManager::DeleteUnmanagedNetworkServiceAccountDirectory() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    while (NN_STATIC_CONDITION(true))
    {
        nn::fs::DirectoryHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&handle, "friends:/", nn::fs::OpenDirectoryMode_Directory));

        bool isOpened = true;

        NN_UTIL_SCOPE_EXIT
        {
            if (isOpened)
            {
                nn::fs::CloseDirectory(handle);
            }
        };

        nn::fs::DirectoryEntry entry;
        int64_t count = 0;

        if (m_IsNetworkServiceAccountAvailable)
        {
            do
            {
                NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, handle, 1));

                if (count == 0)
                {
                    break;
                }

                char name[17] = {};
                nn::util::SNPrintf(name, sizeof (name), "%016llx", m_AccountId);

                if (nn::util::Strncmp(name, entry.name, sizeof (name)) == 0)
                {
                    // 有効なディレクトリなので、スキップして次のディレクトリを検査する。
                    // NN_DETAIL_FRIENDS_INFO("[friends] Managed directory (%s)\n", entry.name);
                }
                else
                {
                    nn::fs::CloseDirectory(handle);
                    isOpened = false;

                    char path[128] = {};
                    Make(path, sizeof (path), entry.name);

                    NN_DETAIL_FRIENDS_INFO("[friends] Unmanaged directory (%s). Delete...\n", entry.name);

                    NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path));
                    NN_RESULT_DO(Commit());
                    break;
                }
            }
            while (NN_STATIC_CONDITION(true));

            if (count == 0)
            {
                break;
            }
        }
        else
        {
            NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, handle, 1));

            if (count == 0)
            {
                break;
            }

            nn::fs::CloseDirectory(handle);
            isOpened = false;

            char path[128] = {};
            Make(path, sizeof (path), entry.name);

            NN_DETAIL_FRIENDS_INFO("[friends] Unmanaged directory (%s). Delete...\n", entry.name);

            NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path));
            NN_RESULT_DO(Commit());
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result AccountStorageManager::DeleteNetworkServiceAccountDirectory() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    // 念のため管理外のディレクトリも存在すれば削除する。

    while (NN_STATIC_CONDITION(true))
    {
        nn::fs::DirectoryHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&handle, "friends:/", nn::fs::OpenDirectoryMode_Directory));

        bool isOpened = true;

        NN_UTIL_SCOPE_EXIT
        {
            if (isOpened)
            {
                nn::fs::CloseDirectory(handle);
            }
        };

        nn::fs::DirectoryEntry entry;
        int64_t count = 0;

        NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, handle, 1));

        if (count == 0)
        {
            break;
        }

        nn::fs::CloseDirectory(handle);
        isOpened = false;

        char path[128] = {};
        Make(path, sizeof (path), entry.name);

        NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path));
        NN_RESULT_DO(Commit());
    }

    NN_RESULT_SUCCESS;
}

void AccountStorageManager::MakePath(char* path, size_t size, const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES(size > 0);
    NN_SDK_REQUIRES_NOT_NULL(name);

    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    Make(path, size, name);
}

nn::Result AccountStorageManager::MakePathWithNetworkServiceAccountDirectory(char* path, size_t size, const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES(size > 0);
    NN_SDK_REQUIRES_NOT_NULL(name);

    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    NN_RESULT_THROW_UNLESS(m_IsNetworkServiceAccountAvailable, ResultNetworkServiceAccountNotLinked());

    Make(path, size, m_AccountId, name);

    NN_RESULT_SUCCESS;
}

void AccountStorageManager::MakeFacedFriendRequestProfileImageUrl(char* url, size_t size, const nn::account::Uid& uid,
    nn::account::NetworkServiceAccountId myAccountId, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(url);
    NN_SDK_REQUIRES(size > 0);

    nn::util::Uuid uuid;
    char str[nn::util::Uuid::StringSize];

    NN_ABORT_UNLESS(nn::util::SNPrintf(url, size, "friends-faced-image:/%s/%016llx/%016llx.jpg",
        ConvertId(&uuid, uid).ToString(str, sizeof (str)), myAccountId.id, accountId.id) < static_cast<int>(size));
}

bool AccountStorageManager::ResolveFacedFriendRequestProfileImageUrl(nn::account::Uid* outUid,
    char* path, size_t size, const char* url) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outUid);
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES(size > 0);
    NN_SDK_REQUIRES_NOT_NULL(url);

    nn::util::Uuid uuid;
    char str[nn::util::Uuid::StringSize];

    unsigned long long a1;
    unsigned long long a2;

    int num = std::sscanf(url, "friends-faced-image:/%36s/%016llx/%016llx.jpg", str, &a1, &a2);

    if (num != 3)
    {
        return false;
    }

    nn::account::NetworkServiceAccountId myAccountId;
    nn::account::NetworkServiceAccountId accountId;

    myAccountId.id = a1;
    accountId.id = a2;

    uuid.FromString(str);

    if (!ConvertId(outUid, uuid))
    {
        return false;
    }

    NN_ABORT_UNLESS(nn::util::SNPrintf(path, size, "friends:/%016llx/faced/%016llx.jpg",
        myAccountId.id, accountId.id) < static_cast<int>(size));

    return true;
}

}}}}}
