﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/ipc/friends_IpcService.h>
#include <nn/friends/detail/ipc/friends_ServiceConfig.h>
#include <nn/friends/detail/ipc/friends_ServiceName.h>
#include <nn/friends/detail/ipc/friends_IServiceCreator.sfdl.h>
#include <nn/friends/detail/ipc/friends_IFriendService.sfdl.h>
#include <nn/friends/detail/ipc/friends_INotificationService.sfdl.h>
#include <nn/friends/detail/ipc/friends_IDaemonSuspendSessionService.sfdl.h>
#include <nn/friends/friends_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sm/sm_Result.h>
#include <nn/sf/sf_ShimLibraryUtility.h>

#if defined (NN_BUILD_CONFIG_OS_WIN)
#include <nn/friends/detail/service/friends_ServiceCreator.generic.h>
#endif

namespace nn { namespace friends { namespace detail { namespace ipc {

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

#if defined (NN_BUILD_CONFIG_OS_WIN)

    detail::service::ServiceCreatorForGeneric g_Creator;

#else

    nn::sf::ShimLibraryObjectHolder<IServiceCreator>
        g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    nn::sf::SimpleAllInOneHipcSubDomainClientManager<ObjectCountPerProcess>
        g_Domain = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;

#endif
}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

#if !defined (NN_BUILD_CONFIG_OS_WIN)

    const char* serviceNames[] =
    {
        ServiceNameForGeneric,
        ServiceNameForViewer,
        ServiceNameForManager,
        ServiceNameForSystem,
        ServiceNameForAdministrator
    };

    bool isServiceFound = false;

    // 1 アプリに割り当てる権限は 1 つのみ。
    // 1 アプリが複数の権限を所持していた場合、最初に発見したもののみを利用する。
    for (auto name : serviceNames)
    {
        nn::Result result = g_Domain.InitializeShimLibraryHolder(&g_Holder, name);

        if (result.IsSuccess())
        {
            isServiceFound = true;
            break;
        }
        else if (!nn::sm::ResultNotPermitted::Includes(result))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }

    NN_ABORT_UNLESS(isServiceFound, "[friends] Needs API permission.");

    g_Domain.GetClientSessionManager().SetSessionCount(SessionCountPerProcess);

#endif
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

#if !defined (NN_BUILD_CONFIG_OS_WIN)

    g_Holder.FinalizeHolder();

#endif
}

nn::Result CreateFriendService(IFriendService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<IFriendService> service = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateFriendService(&service));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateFriendService(&service));

#endif

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result CreateNotificationService(INotificationService** outService, const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<INotificationService> service = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateNotificationService(&service, uid));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateNotificationService(&service, uid));

#endif

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result CreateDaemonSuspendSessionService(IDaemonSuspendSessionService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<IDaemonSuspendSessionService> service = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateDaemonSuspendSessionService(&service));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateDaemonSuspendSessionService(&service));

#endif

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

}}}}
