﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/friends_AsyncContextInternal.h>
#include <nn/friends/friends_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <new>
#include <mutex>

#define ROUNDUP_ALIGNMENT(ptr, align) \
    reinterpret_cast<nn::Bit8*>((reinterpret_cast<uintptr_t>(ptr) + ((align) - 1)) & ~((align) - 1))

namespace nn { namespace friends { namespace detail {

nn::Result AsyncContextInternal::CreateInstance(AsyncContextInternal** outInstance,
    detail::ipc::IFriendService* session, nn::AllocateFunction allocateFunction, nn::DeallocateFunction freeFunction) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInstance);
    NN_SDK_REQUIRES_NOT_NULL(session);
    NN_SDK_REQUIRES_NOT_NULL(allocateFunction);
    NN_SDK_REQUIRES_NOT_NULL(freeFunction);

    nn::Bit8* buffer = static_cast<nn::Bit8*>(allocateFunction(AllocateBufferSize));

    if (!buffer)
    {
        NN_RESULT_THROW(ResultOutOfMemory());
    }

    nn::Bit8* stack = ROUNDUP_ALIGNMENT(buffer, nn::os::ThreadStackAlignment);

    // 前：バッファの先頭とスタックの先頭の差分領域
    size_t headSpace = static_cast<size_t>(stack - buffer);
    // 後：スタックの終端とバッファの終端の差分領域
    size_t tailSpace = AllocateBufferSize - (headSpace +  ThreadStackSize);

    // スタックの位置がどこであろうとも、前後どちらかに ObjectBufferSize 以上のサイズのバッファが存在する。
    NN_SDK_ASSERT(headSpace >= ObjectBufferSize || tailSpace >= ObjectBufferSize);

    void* obj = nullptr;

    if (headSpace >= tailSpace)
    {
        obj = ROUNDUP_ALIGNMENT(buffer, sizeof (std::max_align_t));
    }
    else
    {
        // nn::os::ThreadStackAlignment でアラインメントされているのでサイズ調整不要。
        obj = stack + ThreadStackSize;
    }

    AsyncContextInternal* internal = new (obj) AsyncContextInternal;

    nn::sf::NativeHandle handle;

    // Generic 権限の API なので必ず成功するはず。
    NN_ABORT_UNLESS_RESULT_SUCCESS(session->GetCompletionEvent(&handle));

    nn::os::AttachReadableHandleToSystemEvent(&internal->m_CompletionEvent,
        handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
    internal->m_IsCompletionEventCreated = true;

    handle.Detach();

    internal->m_Session = session;

    internal->m_Stack = stack;
    internal->m_AllocatedBuffer = buffer;

    internal->m_FreeFunction = freeFunction;

    *outInstance = internal;

    NN_RESULT_SUCCESS;
}

void AsyncContextInternal::DeleteInstance(AsyncContextInternal* instance) NN_NOEXCEPT
{
    instance->~AsyncContextInternal();
}

AsyncContextInternal::AsyncContextInternal() NN_NOEXCEPT :
    m_IsThreadCreated(false),
    m_IsCompletionEventCreated(false),
    m_Result(ResultNotCalled()),
    m_Session(nullptr),
    m_AllocatedBuffer(nullptr)
{
    m_Mutex.Initialize();
}

AsyncContextInternal::~AsyncContextInternal() NN_NOEXCEPT
{
    if (m_Session)
    {
        m_Session->Cancel();
    }
    if (m_IsThreadCreated)
    {
        nn::os::DestroyThread(&m_Thread);
    }
    if (m_Session)
    {
        nn::sf::ReleaseSharedObject(m_Session);
    }
    if (m_IsCompletionEventCreated)
    {
        nn::os::DestroySystemEvent(&m_CompletionEvent);
    }

    if (m_AllocatedBuffer)
    {
        // これ以降、AsyncContextInternal オブジェクトに一切アクセスしてはいけない。
        // デストラクタが呼ばれるオブジェクトには注意すること。
        m_FreeFunction(m_AllocatedBuffer, AllocateBufferSize);
    }
}

void AsyncContextInternal::CallAsync(const char* threadName, SyncCall function, void* param, size_t paramSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(paramSize <= sizeof (m_Param));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_Function = function;
    std::memcpy(&m_Param, param, paramSize);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, FunctionCaller, this,
        m_Stack, ThreadStackSize,
        nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()), nn::os::GetCurrentCoreNumber()));
    m_IsThreadCreated = true;

    nn::os::SetThreadNamePointer(&m_Thread, threadName);
    nn::os::StartThread(&m_Thread);
}

void AsyncContextInternal::CallAsync(const char* threadName,
    SyncCall function, DeepCopyCallback callback, const void* sourceParam) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_Function = function;
    callback(sourceParam, &m_Param, sizeof (m_Param));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, FunctionCaller, this,
        m_Stack, ThreadStackSize,
        nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()), nn::os::GetCurrentCoreNumber()));
    m_IsThreadCreated = true;

    nn::os::SetThreadNamePointer(&m_Thread, threadName);
    nn::os::StartThread(&m_Thread);
}

void AsyncContextInternal::Cancel() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_Session)
    {
        m_Session->Cancel();
    }
}

bool AsyncContextInternal::IsCompleted() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_IsCompletionEventCreated)
    {
        return nn::os::TimedWaitSystemEvent(&m_CompletionEvent, nn::TimeSpan());
    }

    return false;
}

nn::Result AsyncContextInternal::GetResult() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return m_Result;
}

void AsyncContextInternal::AttachCompletionEventReadableHandle(nn::os::SystemEvent *outEvent) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsCompletionEventCreated);

    outEvent->AttachReadableHandle(nn::os::GetReadableHandleOfSystemEvent(&m_CompletionEvent),
        false, nn::os::EventClearMode_ManualClear);
}

void AsyncContextInternal::FunctionCaller(void* arg) NN_NOEXCEPT
{
    AsyncContextInternal* p = static_cast<AsyncContextInternal*>(arg);

    p->m_Result = ResultCallInProgress();
    p->m_Result = p->m_Function(p->m_Session, &p->m_Param);

    {
        std::lock_guard<decltype (p->m_Mutex)> lock(p->m_Mutex);

        // Session の破棄によって m_CompletionEvent は Signal される。
        nn::sf::ReleaseSharedObject(p->m_Session);
        p->m_Session = nullptr;
    }
}

}}}
