﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if defined(NN_FONTLL2_BUILD_FONTLL2)
#include <nn/fontll2/fontll2_ScalableFontEngine.h>
#else
#include <nn/fontll/fontll_ScalableFontEngine.h>
#endif

#include <nn/util/util_Endian.h>
#include <algorithm>

#if defined(NN_FONTLL2_BUILD_FONTLL2)
namespace nn { namespace fontll2 {
#else
namespace nn { namespace fontll {
#endif

namespace detail
{

// 2バイトのビッグエンディアンの値を取得する
uint16_t AcquireUint16Value(const uint8_t** ptr)
{
    uint16_t result = nn::util::LoadBigEndian<uint16_t>(reinterpret_cast<const uint16_t*>(*ptr));
    *ptr += sizeof(uint16_t);
    return result;
}

// FeatureTableOffsetの値を抜き出す
uint16_t PickFeatureTableOffset(const void* pGposTable, uint16_t offsetScriptList, uint16_t offsetFeatureList, const char* tableName)
{
    // ScriptListテーブル情報に移動
    const uint8_t* ptr = static_cast<const uint8_t*>(pGposTable) + offsetScriptList;
    uint16_t scriptCount = AcquireUint16Value(&ptr);

    // カーニング情報が存在するFeatureTableのオフセットを取得する
    for (uint16_t i = 0; i < scriptCount; i++)
    {
        // ScriptListテーブルから特定のタグのものを探す
        bool found = false;
        if ((ptr[0] == 'D' && ptr[1] == 'F' && ptr[2] == 'L' && ptr[3] == 'T') ||
            (ptr[0] == 'k' && ptr[1] == 'a' && ptr[2] == 'n' && ptr[3] == 'a') ||
            (ptr[0] == 'l' && ptr[1] == 'a' && ptr[2] == 't' && ptr[3] == 'n'))
        {
            found = true;
        }
        ptr += 4;
        if (found)
        {
            // ScriptTableの読み込み
            uint16_t offset = AcquireUint16Value(&ptr);
            const uint8_t* ptr2 = static_cast<const uint8_t*>(pGposTable) + offsetScriptList + offset;
            uint16_t defaultLangSysOffset = AcquireUint16Value(&ptr2);
            uint16_t langSysCount = AcquireUint16Value(&ptr2);

            // LangSysTableの位置の取得
            if (defaultLangSysOffset == 0)
            {
                // DefaultLangSysTableが定義されていない場合はLangSysTableのリストを読み込む
                if (langSysCount == 0)
                {
                    continue; // 存在しなければ次へ
                }
                ptr2 += 4; // tag
                uint16_t langSysTableOffset = AcquireUint16Value(&ptr2);
                ptr2 = static_cast<const uint8_t*>(pGposTable) + offsetScriptList + offset + langSysTableOffset;
            }
            else
            {
                ptr2 = static_cast<const uint8_t*>(pGposTable) + offsetScriptList + offset + defaultLangSysOffset;
            }

            // LangSysTableの読み込み
            ptr2 += 2; // LookupOrder
            ptr2 += 2; // ReqFeatureIndex
            uint16_t featureCount = AcquireUint16Value(&ptr2);
            for (uint16_t j = 0; j < featureCount; j++)
            {
                uint16_t featureIndex = AcquireUint16Value(&ptr2);

                // FeatureTableの位置の取得
                const uint8_t* ptr3 = static_cast<const uint8_t*>(pGposTable) + offsetFeatureList;
                ptr3 += 2; // FeatureCount
                for (uint16_t k = 0; k < featureIndex; k++)
                {
                    ptr3 += 4; // Tag
                    ptr3 += 2; // Offset
                }
                if (ptr3[0] == tableName[0] && ptr3[1] == tableName[1] && ptr3[2] == tableName[2] && ptr3[3] == tableName[3])
                {
                    ptr3 += 4; // Tag
                    uint16_t featureTableOffset = AcquireUint16Value(&ptr3);
                    return featureTableOffset;
                }
            }
            break;
        }
        else
        {
            ptr += sizeof(uint16_t); // offset
        }
    }
    return 0;
}

// Coverageを読み込む
uint16_t ReadCoverage(const void* pGposTable, uint16_t offset, uint16_t index)
{
    // index番目のCoverageを取得する
    const uint8_t* ptr = static_cast<const uint8_t*>(pGposTable) + offset;
    uint16_t format = AcquireUint16Value(&ptr);
    if (format == 1)
    {
        uint16_t glyphCount = AcquireUint16Value(&ptr);
        for (uint16_t i = 0; i < glyphCount; i++)
        {
            uint16_t coverage = AcquireUint16Value(&ptr);
            if (index == 0)
            {
                return coverage;
            }
            index--;
        }
    }
    else
    {
        uint16_t rangeCount = AcquireUint16Value(&ptr);
        for (uint16_t i = 0; i < rangeCount; i++)
        {
            uint16_t startGlyphId = AcquireUint16Value(&ptr);
            uint16_t endGlyphId = AcquireUint16Value(&ptr);
            ptr += 2; // startCoverageIndex
            for (uint16_t j = startGlyphId; j <= endGlyphId; j++)
            {
                uint16_t coverage = j;
                if (index == 0)
                {
                    return coverage;
                }
                index--;
            }
        }
    }
    NN_SDK_ASSERT(false); // 必ず見つかるのでここには来ないはず
    return 0;
}

// XAdvanceとXPlacementを読み込む
void ReadXValues(uint16_t* pXAdvance, uint16_t* pXPlacement, const uint8_t** ptr, uint16_t format)
{
    *pXAdvance = 0;
    *pXPlacement = 0;
    if ((format & 0x01/*XPlacement*/) > 0)
    {
        *pXPlacement = AcquireUint16Value(ptr);
    }
    if ((format & 0x02/*YPlacement*/) > 0)
    {
        *ptr += 2; // YPlacement
    }
    if ((format & 0x04/*XAdvance*/) > 0)
    {
        *pXAdvance = AcquireUint16Value(ptr);
    }
    if ((format & 0x08/*YAdvance*/) > 0)
    {
        *ptr += 2; // YAdvance
    }
    if ((format & 0x10/*XPlaDevice*/) > 0)
    {
        *ptr += 2; // XPlaDevice
    }
    if ((format & 0x20/*YPlaDevice*/) > 0)
    {
        *ptr += 2; // YPlaDevice
    }
    if ((format & 0x40/*XAdvDevice*/) > 0)
    {
        *ptr += 2; // XAdvDevice
    }
    if ((format & 0x80/*YAdvDevice*/) > 0)
    {
        *ptr += 2; // YAdvDevice
    }
}

// XAdvanceとXPlacementを読み飛ばす
void SkipXValues(const uint8_t** ptr, uint16_t format)
{
    if ((format & 0x01/*XPlacement*/) > 0)
    {
        *ptr += 2; // XPlacement
    }
    if ((format & 0x02/*YPlacement*/) > 0)
    {
        *ptr += 2; // YPlacement
    }
    if ((format & 0x04/*XAdvance*/) > 0)
    {
        *ptr += 2; // XAdvance
    }
    if ((format & 0x08/*YAdvance*/) > 0)
    {
        *ptr += 2; // YAdvance
    }
    if ((format & 0x10/*XPlaDevice*/) > 0)
    {
        *ptr += 2; // XPlaDevice
    }
    if ((format & 0x20/*YPlaDevice*/) > 0)
    {
        *ptr += 2; // YPlaDevice
    }
    if ((format & 0x40/*XAdvDevice*/) > 0)
    {
        *ptr += 2; // XAdvDevice
    }
    if ((format & 0x80/*YAdvDevice*/) > 0)
    {
        *ptr += 2; // YAdvDevice
    }
}

// XAdvanceとXPlacementの値をカーニングテーブルに追加する
void AddXValuesToOtfKerningTable(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, uint16_t c1, uint16_t c2, uint16_t xAdvance, uint16_t xPlacement)
{
    if (xAdvance == 0 && xPlacement == 0)
    {
        // カーニング情報には xAdvance と xPlacement 以外の用途で含まれているものがあるが、
        // 使用しないのでカーニングテーブルには追加しない。
        return;
    }

    // カーニングテーブルが NULL であれば、要素数だけをインクリメントして実際の追加処理は飛ばす
    if (pOtfKerningTable != NULL)
    {
        OtfKerningTable::KerningPair* pKerningPair = &pOtfKerningTable->pKerningPairs[*pKerningPairCount];
        pKerningPair->first = c1;
        pKerningPair->second = c2;
        pKerningPair->advance = static_cast<int16_t>(xAdvance);
        pKerningPair->placement = static_cast<int16_t>(xPlacement);
    }
    (*pKerningPairCount)++;
}

// 2つ目のClass内の全文字を読み込む
void ReadClass2(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t key1, uint16_t offset2, uint16_t classId2, uint16_t xAdvance, uint16_t xPlacement)
{
    const uint8_t* ptr = static_cast<const uint8_t*>(pGposTable) + offset2;
    uint16_t format = AcquireUint16Value(&ptr);
    if (format == 1)
    {
        uint16_t glyphIndex = AcquireUint16Value(&ptr);
        uint16_t glyphCount = AcquireUint16Value(&ptr);
        for (uint16_t i = 0; i < glyphCount; i++)
        {
            uint16_t value = AcquireUint16Value(&ptr);
            if (classId2 == value)
            {
                uint16_t key2 = glyphIndex + i;
                AddXValuesToOtfKerningTable(pKerningPairCount, pOtfKerningTable, key1, key2, xAdvance, xPlacement);
            }
        }
    }
    else
    {
        uint16_t rangeCount = AcquireUint16Value(&ptr);
        for (uint16_t i = 0; i < rangeCount; i++)
        {
            uint16_t startGlyphId = AcquireUint16Value(&ptr);
            uint16_t endGlyphId = AcquireUint16Value(&ptr);
            uint16_t classValue = AcquireUint16Value(&ptr);
            uint16_t value = classValue;
            if (classId2 == value)
            {
                for (uint16_t j = startGlyphId; j <= endGlyphId; j++)
                {
                    uint16_t key2 = j;
                    AddXValuesToOtfKerningTable(pKerningPairCount, pOtfKerningTable, key1, key2, xAdvance, xPlacement);
                }
            }
        }
    }
}

// 1つ目のClass内の全文字を読み込む
void ReadClass1(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t offset1, uint16_t offset2, uint16_t classId1, uint16_t classId2, uint16_t xAdvance, uint16_t xPlacement)
{
    const uint8_t* ptr = static_cast<const uint8_t*>(pGposTable) + offset1;
    uint16_t format = AcquireUint16Value(&ptr);
    if (format == 1)
    {
        uint16_t glyphIndex = AcquireUint16Value(&ptr);
        uint16_t glyphCount = AcquireUint16Value(&ptr);
        for (uint16_t i = 0; i < glyphCount; i++)
        {
            uint16_t value = AcquireUint16Value(&ptr);
            if (classId1 == value)
            {
                uint16_t key1 = glyphIndex + i;
                ReadClass2(pKerningPairCount, pOtfKerningTable, pGposTable, key1, offset2, classId2, xAdvance, xPlacement);
            }
        }
    }
    else
    {
        uint16_t rangeCount = AcquireUint16Value(&ptr);
        for (uint16_t i = 0; i < rangeCount; i++)
        {
            uint16_t startGlyphId = AcquireUint16Value(&ptr);
            uint16_t endGlyphId = AcquireUint16Value(&ptr);
            uint16_t classValue = AcquireUint16Value(&ptr);
            uint16_t value = classValue;
            if (classId1 == value)
            {
                for (uint16_t j = startGlyphId; j <= endGlyphId; j++)
                {
                    uint16_t key1 = j;
                    ReadClass2(pKerningPairCount, pOtfKerningTable, pGposTable, key1, offset2, classId2, xAdvance, xPlacement);
                }
            }
        }
    }
}

// GlyphPairAdjustmentからXAdvanceとXPlacementの値を抜き出す
void PickKernXValuesFromGlyphPairAdjustment(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t offset, const uint8_t** ptr)
{
    uint16_t coverageOffset = AcquireUint16Value(ptr);
    uint16_t valueFormat1 = AcquireUint16Value(ptr);
    uint16_t valueFormat2 = AcquireUint16Value(ptr);
    uint16_t pairsetCount = AcquireUint16Value(ptr);
    for (uint16_t i = 0; i < pairsetCount; i++)
    {
        uint16_t pairsetOffset = AcquireUint16Value(ptr);
        uint16_t firstGlyphIndex = ReadCoverage(pGposTable, offset + coverageOffset, i);
        const uint8_t* ptr2 = static_cast<const uint8_t*>(pGposTable) + offset + pairsetOffset;
        uint16_t pairValueCount = AcquireUint16Value(&ptr2);
        for (uint16_t j = 0; j < pairValueCount; j++)
        {
            uint16_t secondGlyphIndex = AcquireUint16Value(&ptr2);
            uint16_t firstValueRecordAdvance;
            uint16_t firstValueRecordPlacement;
            ReadXValues(&firstValueRecordAdvance, &firstValueRecordPlacement, &ptr2, valueFormat1);
            AddXValuesToOtfKerningTable(pKerningPairCount, pOtfKerningTable, firstGlyphIndex, secondGlyphIndex, firstValueRecordAdvance, firstValueRecordPlacement);
            SkipXValues(&ptr2, valueFormat2); // secondValueRecordは無視する
        }
    }
}

// ClassPairAdjustmentからXAdvanceとXPlacementの値を抜き出す
void PickKernXValuesFromClassPairAdjustment(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t offset, const uint8_t** ptr)
{
    *ptr += 2; // coverageOffset
    uint16_t valueFormat1 = AcquireUint16Value(ptr);
    uint16_t valueFormat2 = AcquireUint16Value(ptr);
    uint16_t classDefOffset1 = AcquireUint16Value(ptr);
    uint16_t classDefOffset2 = AcquireUint16Value(ptr);
    uint16_t classCount1 = AcquireUint16Value(ptr);
    uint16_t classCount2 = AcquireUint16Value(ptr);
    for (uint16_t i = 0; i < classCount1; i++)
    {
        for (uint16_t j = 0; j < classCount2; j++)
        {
            uint16_t classValueRecordAdvance1;
            uint16_t classValueRecordPlacement1;
            ReadXValues(&classValueRecordAdvance1, &classValueRecordPlacement1, ptr, valueFormat1);
            SkipXValues(ptr, valueFormat2); // classValueRecord2
            ReadClass1(pKerningPairCount, pOtfKerningTable, pGposTable, offset + classDefOffset1, offset + classDefOffset2, i, j, classValueRecordAdvance1, classValueRecordPlacement1);
        }
    }
}

// kern テーブルの XAdvanceの値を抜き出す
void PickKernXValues(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t offsetFeatureList, uint16_t featureTableOffset, uint16_t offsetLookupList)
{
    const uint8_t* ptr = static_cast<const uint8_t*>(pGposTable) + offsetFeatureList + featureTableOffset;
    ptr += 2; // FeatureParams
    uint16_t lookupCount = AcquireUint16Value(&ptr);
    for (uint16_t i = 0; i < lookupCount; i++)
    {
        // LookupTableの読み込み
        uint16_t lookupListIndex = AcquireUint16Value(&ptr);
        const uint8_t* ptr2 = static_cast<const uint8_t*>(pGposTable) + offsetLookupList;
        ptr2 += 2; // LookupCount
        for (uint16_t j = 0; j < lookupListIndex; j++)
        {
            ptr2 += 2; // Offset
        }
        uint16_t offset = AcquireUint16Value(&ptr2);
        ptr2 = static_cast<const uint8_t*>(pGposTable) + offsetLookupList + offset;
        uint16_t lookupType = AcquireUint16Value(&ptr2);
        ptr2 += 2; // lookupFlag
        uint16_t subTableCount = AcquireUint16Value(&ptr2);

        // SubTableの読み込み
        for (int j = 0; j < subTableCount; j++)
        {
            uint16_t subTableOffset = AcquireUint16Value(&ptr2);
            if (lookupType == 2/*PairAdjustment*/)
            {
                const uint8_t* ptr3 = static_cast<const uint8_t*>(pGposTable) + offsetLookupList + offset + subTableOffset;
                uint16_t format = AcquireUint16Value(&ptr3);
                if (format == 1)
                {
                    PickKernXValuesFromGlyphPairAdjustment(pKerningPairCount, pOtfKerningTable, pGposTable, offsetLookupList + offset + subTableOffset, &ptr3);
                }
                else if (format == 2)
                {
                    PickKernXValuesFromClassPairAdjustment(pKerningPairCount, pOtfKerningTable, pGposTable, offsetLookupList + offset + subTableOffset, &ptr3);
                }
            }
        }
    }
}

// SinglePositioningValueからXAdvanceとXPlacementの値を抜き出す
void PickPaltXValuesFromSinglePositioningValue(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t offset, const uint8_t** ptr)
{
    const uint16_t paltCode = 0xffff;
    uint16_t coverageOffset = AcquireUint16Value(ptr);
    uint16_t valueFormat = AcquireUint16Value(ptr);
    uint16_t valueRecordAdvance;
    uint16_t valueRecordPlacement;
    ReadXValues(&valueRecordAdvance, &valueRecordPlacement, ptr, valueFormat);
    const uint8_t* ptr2 = static_cast<const uint8_t*>(pGposTable) + offset + coverageOffset;
    ptr2 += 2; // format
    uint16_t coverageCount = AcquireUint16Value(&ptr2);
    for (uint16_t i = 0; i < coverageCount; i++)
    {
        uint16_t glyphIndex = ReadCoverage(pGposTable, offset + coverageOffset, i);
        AddXValuesToOtfKerningTable(pKerningPairCount, pOtfKerningTable, glyphIndex, paltCode, valueRecordAdvance, valueRecordPlacement);
    }
}

// ArrayOfPositioningValuesからXAdvanceとXPlacementの値を抜き出す
void PickPaltXValuesFromArrayOfPositioningValues(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t offset, const uint8_t** ptr)
{
    const uint16_t paltCode = 0xffff;
    uint16_t coverageOffset = AcquireUint16Value(ptr);
    uint16_t valueFormat = AcquireUint16Value(ptr);
    *ptr += 2; // valueCount
    const uint8_t* ptr2 = static_cast<const uint8_t*>(pGposTable) + offset + coverageOffset;
    ptr2 += 2; // format
    uint16_t coverageCount = AcquireUint16Value(&ptr2);
    for (uint16_t i = 0; i < coverageCount; i++)
    {
        uint16_t glyphIndex = ReadCoverage(pGposTable, offset + coverageOffset, i);
        uint16_t valueRecordAdvance;
        uint16_t valueRecordPlacement;
        ReadXValues(&valueRecordAdvance, &valueRecordPlacement, ptr, valueFormat);
        AddXValuesToOtfKerningTable(pKerningPairCount, pOtfKerningTable, glyphIndex, paltCode, valueRecordAdvance, valueRecordPlacement);
    }
}

// palt テーブルの XAdvance と XPlacement の値を抜き出す
void PickPaltXValues(int* pKerningPairCount, OtfKerningTable* pOtfKerningTable, const void* pGposTable, uint16_t offsetFeatureList, uint16_t featureTableOffset, uint16_t offsetLookupList)
{
    const uint8_t* ptr = static_cast<const uint8_t*>(pGposTable) + offsetFeatureList + featureTableOffset;
    ptr += 2; // FeatureParams
    uint16_t lookupCount = AcquireUint16Value(&ptr);
    for (uint16_t i = 0; i < lookupCount; i++)
    {
        // LookupTableの読み込み
        uint16_t lookupListIndex = AcquireUint16Value(&ptr);
        const uint8_t* ptr2 = static_cast<const uint8_t*>(pGposTable) + offsetLookupList;
        ptr2 += 2; // LookupCount
        for (uint16_t j = 0; j < lookupListIndex; j++)
        {
            ptr2 += 2; // Offset
        }
        uint16_t offset = AcquireUint16Value(&ptr2);
        ptr2 = static_cast<const uint8_t*>(pGposTable) + offsetLookupList + offset;
        uint16_t lookupType = AcquireUint16Value(&ptr2);
        ptr2 += 2; // lookupFlag
        uint16_t subTableCount = AcquireUint16Value(&ptr2);

        // SubTableの読み込み
        for (int j = 0; j < subTableCount; j++)
        {
            uint16_t subTableOffset = AcquireUint16Value(&ptr2);
            if (lookupType == 1/*SingleAdjustment*/)
            {
                const uint8_t* ptr3 = static_cast<const uint8_t*>(pGposTable) + offsetLookupList + offset + subTableOffset;
                uint16_t format = AcquireUint16Value(&ptr3);
                if (format == 1)
                {
                    PickPaltXValuesFromSinglePositioningValue(pKerningPairCount, pOtfKerningTable, pGposTable, offsetLookupList + offset + subTableOffset, &ptr3);
                }
                else if (format == 2)
                {
                    PickPaltXValuesFromArrayOfPositioningValues(pKerningPairCount, pOtfKerningTable, pGposTable, offsetLookupList + offset + subTableOffset, &ptr3);
                }
            }
        }
    }
}

// カーニング情報を比較する
bool CompareKerningPair(const OtfKerningTable::KerningPair& a, const OtfKerningTable::KerningPair& b)
{
    if (a.first < b.first || (a.first == b.first && a.second < b.second))
    {
        return true;
    }
    return false;
}

// GPOSテーブルからカーニングの値を抜き出す
int PickKerningFromGposTable(OtfKerningTable* pOtfKerningTable, const void* pGposTable, bool ignorePalt)
{
    const uint8_t* ptr = static_cast<const uint8_t*>(pGposTable);

    // Commonテーブル情報の読み込み
    ptr += 4; // バージョン情報(2byte*2)
    uint16_t offsetScriptList = AcquireUint16Value(&ptr);
    uint16_t offsetFeatureList = AcquireUint16Value(&ptr);
    uint16_t offsetLookupList = AcquireUint16Value(&ptr);

    // FeatureTableOffsetの取得
    uint16_t featureTableOffsetOfKern = PickFeatureTableOffset(pGposTable, offsetScriptList, offsetFeatureList, "kern");
    uint16_t featureTableOffsetOfPalt;
    if (ignorePalt)
    {
        featureTableOffsetOfPalt = 0;
    }
    else
    {
        featureTableOffsetOfPalt = PickFeatureTableOffset(pGposTable, offsetScriptList, offsetFeatureList, "palt");
    }

    int kerningPairCount = 0;

    // FeatureTableからカーニングの情報を走査する
    if (featureTableOffsetOfKern != 0)
    {
        PickKernXValues(&kerningPairCount, pOtfKerningTable, pGposTable, offsetFeatureList, featureTableOffsetOfKern, offsetLookupList);
    }


    // FeatureTableから palt の情報を走査する
    if (featureTableOffsetOfPalt != 0)
    {
        PickPaltXValues(&kerningPairCount, pOtfKerningTable, pGposTable, offsetFeatureList, featureTableOffsetOfPalt, offsetLookupList);
    }

    // 二分探索のためにカーニングの情報をソートする
    if (pOtfKerningTable != NULL)
    {
        std::sort(pOtfKerningTable->pKerningPairs, pOtfKerningTable->pKerningPairs + pOtfKerningTable->kerningPairCount, CompareKerningPair);
    }

    return kerningPairCount;
}

} // namespace nn::font::detail

int ScalableFontEngine::PickKerningFromGposTable(OtfKerningTable* pOtfKerningTable, const void* pGposTable, bool ignorePalt)
{
    return detail::PickKerningFromGposTable(pOtfKerningTable, pGposTable, ignorePalt);
}

}}
