﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include "fgm_Debugger.h"

namespace nn     {
namespace fgm    {
namespace server {
namespace        {

nn::os::Mutex g_DebuggerListLock(false);
nn::util::IntrusiveList<Debugger, nn::util::IntrusiveListBaseNodeTraits<Debugger>> g_DebuggerList;

} // namespace

nn::Result Debugger::Initialize(
    nn::sf::Out<nn::sf::NativeHandle> handleOut,
    nn::sf::NativeHandle handleIn,
    uint64_t memorySize)
NN_NOEXCEPT
{
    void*      pMemory;
    nn::Result result;

    if (m_Initialized)
    {
        return nn::fgm::ResultAlreadyInitialized();
    }

    nn::os::AttachTransferMemory(&m_TransferMemory, memorySize, handleIn.GetOsHandle(), handleIn.IsManaged());

    handleIn.Detach();

    if ((result = nn::os::MapTransferMemory(&pMemory, &m_TransferMemory, nn::os::MemoryPermission_None)).IsFailure())
    {
        nn::os::DestroyTransferMemory(&m_TransferMemory);
        return result;
    }

    if ((result = Log::Initialize(pMemory, memorySize)).IsFailure())
    {
        nn::os::UnmapTransferMemory(&m_TransferMemory);
        nn::os::DestroyTransferMemory(&m_TransferMemory);
        return result;
    }

    std::lock_guard<nn::os::Mutex> lock(g_DebuggerListLock);

    g_DebuggerList.push_front(*this);

    handleOut.Set(nn::sf::NativeHandle(m_SystemEvent.GetReadableHandle(), false));

    m_Initialized = true;

    return result;
}

void Debugger::Write(DebugEntry* pEntryIn)
NN_NOEXCEPT
{
    if (Log::Write(pEntryIn).IsSuccess())
    {
        Signal();
    }
}

nn::Result Debugger::Read(
    nn::sf::OutBuffer bufferOut,
    nn::sf::Out<uint32_t> numberReadOut,
    nn::sf::Out<uint32_t> numberLostOut,
    nn::sf::Out<uint32_t> numberRemainingOut)
NN_NOEXCEPT
{
    nn::Result  result;
    DebugEntry* pEntries = reinterpret_cast<DebugEntry*>(bufferOut.GetPointerUnsafe());
    uint32_t    toRead  = bufferOut.GetSize() / sizeof(DebugEntry);
    uint32_t    read;
    uint32_t    lost;
    uint32_t    remaining;

    if (!m_Initialized)
    {
        return nn::fgm::ResultNotInitialized();
    }

    if ((result = Log::Read(pEntries, toRead, &read, &lost, &remaining)).IsSuccess())
    {
        numberReadOut.Set(read);
        numberLostOut.Set(lost);
        numberRemainingOut.Set(remaining);
    }

    return result;
}

nn::Result Debugger::Signal()
NN_NOEXCEPT
{
    m_SystemEvent.Signal();
    return ResultSuccess();
}

nn::Result Debugger::Cancel()
NN_NOEXCEPT
{
    if (!m_Initialized)
    {
        return nn::fgm::ResultNotInitialized();
    }

    m_SystemEvent.Signal();
    return ResultSuccess();
}

Debugger::Debugger()
NN_NOEXCEPT :
    m_SystemEvent(nn::os::EventClearMode_AutoClear, true),
    m_Initialized(false)
{

}

Debugger::~Debugger()
NN_NOEXCEPT
{
    if (m_Initialized)
    {
        std::lock_guard<nn::os::Mutex> lock(g_DebuggerListLock);
        g_DebuggerList.erase(g_DebuggerList.iterator_to(*this));
        Log::Finalize();
        nn::os::UnmapTransferMemory(&m_TransferMemory);
        nn::os::DestroyTransferMemory(&m_TransferMemory);
        m_Initialized = false;
    }
}

void LogEntry(DebugEntry* pEntryIn)
NN_NOEXCEPT
{
    pEntryIn->timestamp = nn::os::GetSystemTick();

    std::lock_guard<nn::os::Mutex> lock(g_DebuggerListLock);

    for (auto itr  = g_DebuggerList.begin();
              itr != g_DebuggerList.end();
              itr++)
    {
        itr->Write(pEntryIn);
    }
}

}}}
