﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fatal/detail/fatal_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include "fatalsrv_Cpu.h"
#include "fatalsrv_Environment.h"
#include "fatalsrv_ErrorReport.h"
#include "fatalsrv_Memory.h"
#include "fatalsrv_PowerControl.h"
#include "fatalsrv_Screen.h"
#include "fatalsrv_Sound.h"
#include "fatalsrv_Task.h"
#include "fatalsrv_ThreadContextAccessor.h"

namespace nn { namespace fatalsrv {
    namespace
    {
        class WorkerThread
        {
            NN_DISALLOW_COPY(WorkerThread);
            NN_DISALLOW_MOVE(WorkerThread);

        public:
            WorkerThread() NN_NOEXCEPT
            {}

            Result Run(ITask* task, void* threadStack, size_t threadStackSize) NN_NOEXCEPT
            {
                NN_RESULT_DO(
                    nn::os::CreateThread(
                        &m_Thread,
                        WorkerThreadFunction,
                        task,
                        threadStack,
                        threadStackSize,
                        NN_SYSTEM_THREAD_PRIORITY(fatalsrv, FatalTaskThread),
                        3)
                );

                nn::os::SetThreadName(&m_Thread, NN_SYSTEM_THREAD_NAME(fatalsrv, FatalTaskThread));
                nn::os::StartThread(&m_Thread);

                NN_RESULT_SUCCESS;
            }
        private:
            static void WorkerThreadFunction(void* arg) NN_NOEXCEPT
            {
                GetDefaultThreadContextAccessor()->SetTaskThreadFlag(true);

                auto task = reinterpret_cast<ITask*>(arg);
                NN_DETAIL_FATAL_TRACE("'%s' task started.\n", task->GetTaskName());
                auto taskResult = task->Run();
                if (taskResult.IsSuccess())
                {
                    NN_DETAIL_FATAL_TRACE("'%s' task SUCCESS.\n", task->GetTaskName());
                }
                else
                {
                    NN_DETAIL_FATAL_ERROR("'%s' task FAILED. result = %08x\n", task->GetTaskName(), taskResult.GetInnerValueForDebug());
                }
            }

            nn::os::ThreadType m_Thread;
        };

        template<int MaxTaskCount>
        class TaskManager
        {
        public:
            TaskManager() NN_NOEXCEPT : m_TaskCount(0)
            {}


            void InvokeTask(ITask* task) NN_NOEXCEPT
            {
                NN_SDK_ASSERT(m_TaskCount < MaxTaskCount);
                auto& worker = m_WorkerThreads[m_TaskCount++];

                auto threadStackSize = task->GetRequiredStackSize();
                auto threadStack = Allocate(threadStackSize, nn::os::ThreadStackAlignment, nullptr);
                NN_ABORT_UNLESS(threadStack, "Failed thread stack allocation. size = %zu, name = %s\n", threadStackSize, task->GetTaskName());

                auto result = worker.Run(task, threadStack, threadStackSize);
                if (result.IsFailure())
                {
                    NN_DETAIL_FATAL_TRACE(
                        "'%s' failed to start, result = %08x\n", task->GetTaskName(), result.GetInnerValueForDebug());
                }
            }

        private:
            int m_TaskCount;
            WorkerThread m_WorkerThreads[MaxTaskCount];
        };

        TaskManager<8> g_TaskManager;
    }

    void InvokeTasks(const Service::Context& context,
        Bit64 callerProgramId,
        bool needsErrorReport,
        os::Event* errorReportWrittenEvent,
        os::Event* batteryCheckedEvent) NN_NOEXCEPT
    {
        g_TaskManager.InvokeTask(InitializeAndGetErrorReportTask(context, callerProgramId, needsErrorReport, errorReportWrittenEvent));
        g_TaskManager.InvokeTask(InitializeAndGetPowerControlTask(errorReportWrittenEvent, batteryCheckedEvent));
        g_TaskManager.InvokeTask(InitializeAndGetFatalScreenTask(context, callerProgramId, batteryCheckedEvent));
        g_TaskManager.InvokeTask(InitializeAndGetSoundTask());
        g_TaskManager.InvokeTask(InitializeAndGetBacklightControlTask());
        g_TaskManager.InvokeTask(InitializeAndGetAdjustClockTask());
        g_TaskManager.InvokeTask(InitializeAndGetPowerButtonTask(errorReportWrittenEvent));
        g_TaskManager.InvokeTask(InitializeAndGetStateTransitionStopTask());
    }

}} // namespace nn::fatalsrv
