﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fatal/detail/fatal_Log.h>
#include <nn/fatal/fatal_Result.h>
#include <nn/fatalsrv/fatalsrv_Service.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_Event.h>
#include <nn/pm/pm_InformationApi.h>

#include "fatalsrv_ErrorReport.h"
#include "fatalsrv_Environment.h"
#include "fatalsrv_Task.h"

namespace nn { namespace fatalsrv {
    namespace {

        Result GetProgramIdFromProcessId(nn::ncm::ProgramId* out, const nn::os::ProcessId& processId) NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
            NN_FUNCTION_LOCAL_STATIC(bool, isPmInitialized, = false);
            if (!isPmInitialized)
            {
                NN_RESULT_DO(nn::pm::InitializeForInformation());
                // INFO: Finalize は実装されていないので呼び出さない
                isPmInitialized = true;
            }

            NN_RESULT_DO(nn::pm::GetProgramId(out, processId));
#else
            NN_UNUSED(processId);
            *out = { 0 };
#endif
            NN_RESULT_SUCCESS;
        }

        Result PerformFatalProcess(
            const Service::Context& context,
            nn::Bit64 callerProgramId,
            bool needsErrorReport,
            os::Event* errorReportWrittenEvent,
            os::Event* batteryCheckedEvent) NN_NOEXCEPT
        {
            if (GetEnvironmentInfo().transitionToFatal)
            {
                InvokeTasks(context, callerProgramId, needsErrorReport, errorReportWrittenEvent, batteryCheckedEvent);
            }
            else
            {
                NN_DETAIL_FATAL_TRACE("Skip fatal process because transition flag is false.\n");
            }
            NN_RESULT_SUCCESS;
        }

    } // namespace

    Service::Service() NN_NOEXCEPT : m_AlreadyThrown(false), m_Context({}), m_ErrorReportWrittenEvent(os::EventClearMode_ManualClear), m_BatteryCheckedEvent(os::EventClearMode_ManualClear)
    {
    }

    Result Service::ThrowFatal(nn::fatalsrv::FatalContext context, nn::Bit64 processId) NN_NOEXCEPT
    {
        return ThrowFatalWithCpuContext(context, processId, nn::fatal::FatalPolicy::FatalPolicy_Default, {});
    }

    Result Service::ThrowFatalWithPolicy(nn::fatalsrv::FatalContext context, nn::Bit64 processId, nn::fatal::FatalPolicy policy) NN_NOEXCEPT
    {
        return ThrowFatalWithCpuContext(context, processId, policy, {});
    }

    Result Service::ThrowFatalWithCpuContext(nn::fatalsrv::FatalContext fatalContext, nn::Bit64 processId, nn::fatal::FatalPolicy policy, const nn::fatalsrv::CpuContext& cpuContext) NN_NOEXCEPT
    {
        NN_DETAIL_FATAL_TRACE("ThrowFatal() is called.\n");

        nn::ncm::ProgramId callerProgramId = {};
        auto result = GetProgramIdFromProcessId(&callerProgramId, { processId });
        if (result.IsSuccess())
        {
            NN_DETAIL_FATAL_TRACE("Caller ProgramId: 0x%016llx\n", callerProgramId.value);
        }
        else
        {
            NN_DETAIL_FATAL_TRACE("Failed to specify caller program. result = 0x%08x\n", result.GetInnerValueForDebug());
        }

        NN_DETAIL_FATAL_TRACE("FatalContext: LastResult = 0x%08x\n", fatalContext.lastResult.GetInnerValueForDebug());
        NN_DETAIL_FATAL_TRACE("FatalPolicy: %d\n", policy);

        switch (policy)
        {
            case fatal::FatalPolicy_OnlyErrorReport:
                {
                    NN_RESULT_DO(WriteErrorReport(fatalContext, cpuContext, callerProgramId.value));
                }
                break;
            case fatal::FatalPolicy_Default:
            case fatal::FatalPolicy_WithoutErrorReport:
                {
                    NN_RESULT_DO(CheckAndUpdateAlreadyThrownFlag());
                    // IPC から返った後も Fatal の各タスクから非同期に参照されうる値なのでインスタンスのメンバにコピーしておく
                    SaveContext(fatalContext, cpuContext);

                    m_FatalEventManager.Signal();
                    auto needsErrorReport = (policy == fatal::FatalPolicy_Default) ? true : false;
                    NN_RESULT_DO(PerformFatalProcess(GetContext(), callerProgramId.value, needsErrorReport, &m_ErrorReportWrittenEvent, &m_BatteryCheckedEvent));
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        NN_RESULT_SUCCESS;
    }

    Result Service::GetFatalEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
    {
        os::SystemEvent* event;
        NN_RESULT_DO(m_FatalEventManager.GetEvent(&event));
        outHandle.Set(nn::sf::NativeHandle(event->GetReadableHandle(), false));
        NN_RESULT_SUCCESS;
    }
}} // namespace nn::fatalsrv
