﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/vi.h>
#include <nn/vi.private.h>
#include <nn/vi/buffer/vi_Buffer.h>
#include <nn/vi/vi_Content.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/manager/vi_Manager.h>

#include <nn/lbl/lbl.h>
#include <nn/lbl/lbl_Switch.h>

#include <nn/nn_Abort.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fatal/detail/fatal_Log.h>

#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/settings/system/settings_ProductModel.h>
#include <nn/settings/system/settings_FirmwareVersion.h>

#include <nn/mem.h>
#include <nn/init.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>

#include <nn/fatal/fatal_Result.h>

#include "fatalsrv_Environment.h"
#include "fatalsrv_ErrorReport.h"
#include "fatalsrv_Screen.h"
#include "fatalsrv_Font.h"
#include "fatalsrv_Memory.h"
#include "fatalsrv_MessageManager.h"
#include "fatalsrv_Task.h"

#define NN_FATALSRV_GENERATE_STRING(NAME, NUMBER) #NAME,
#define LOCAL_TO_LOGICAL(x) ((x) * 3 / 2)

namespace nn { namespace fatalsrv {
    namespace {
        const int FontBufferSize = 128 * 1024;
        const size_t GraphicsSystemMemorySize = 256 * 1024;
        const size_t DefaultGraphicsAlign = 4096;

        // INFO:
        // http://spdlybra.nintendo.co.jp/jira/browse/SWHUID-520
        void DrawFatalScreen(
            TextRenderer& renderer,
            void* buffer,
            int width, int height,
            int stride,
            const err::ErrorCode& errorCode,
            float scaleX, float scaleY,
            const CpuContext& cpuContext) NN_NOEXCEPT
        {
            Color gray = { { 190, 190, 190, 255 } };
            Color white = { { 255, 255, 255, 255 } };
            renderer.SetScaleX(scaleX);
            renderer.SetScaleY(scaleY);

            auto envInfo = GetEnvironmentInfo();

            renderer.SetFontSize(16);
            renderer.SetColor(gray);
            renderer.PutString(0, 0, GetErrorCodeLabel(envInfo.languageCode),
                errorCode.category,
                errorCode.number);


            renderer.SetFontSize(16);
            renderer.SetColor(gray);
            renderer.SetNewLineHeight(26);
            const char InfoMessage[] =
                "(X1) %s\n"
                "(X2) %d.%d.%d\n";
                renderer.PutString(0, 352, InfoMessage,
                envInfo.serialNumber,
                envInfo.firmwareVersion.major, envInfo.firmwareVersion.minor, envInfo.firmwareVersion.micro
            );

            renderer.SetFontSize(20);
            renderer.SetColor(white);
            renderer.SetNewLineHeight(36);

            renderer.PutString(0, 38, GetMessage(envInfo.languageCode, errorCode));

            const int baseX = static_cast<int>(0 * scaleX);
            const int baseY = static_cast<int>(330 * scaleY);
            const int displayWidth = static_cast<int>(width * scaleX);

            // 下部の横線
            Color gray2 = { { 125, 125, 125, 255 } };
            for (int i = 0; i < 2; ++i)
            {
                for (int j = 0; j < displayWidth; ++j)
                {
                    reinterpret_cast<PixelType*>(buffer)
                        [(baseY + i) * stride + baseX + j] = GetPixelValue(gray2);

                }
            }
        }

        class IWriter
        {
        public:
            virtual Result Draw(void* bufferAddr, int stride) NN_NOEXCEPT = 0;
        };

        class FatalScreenWriter : public IWriter
        {
        public:
            FatalScreenWriter(
                const FatalContext& fatalContext,
                int layerWidth,
                int layerHeight,
                const CpuContext& cpuContext,
                nn::Bit64 callerProgramId
            ) NN_NOEXCEPT
                : FatalContext(fatalContext), LayerWidth(layerWidth), LayerHeight(layerHeight),
                    m_CpuContext(cpuContext), m_CallerProgramId(callerProgramId)
            {
            }

            virtual Result Draw(void* bufferAddr, int stride) NN_NOEXCEPT
            {
                // 黒で塗りつぶし
                Color black = { {0, 0, 0, 255} };
                for (int i = 0; i < LayerHeight; ++i)
                {
                    for (int j = 0; j < LayerWidth; ++j)
                    {
                        reinterpret_cast<PixelType*>(bufferAddr)[i * stride + j] = GetPixelValue(black);
                    }
                }

                // 画面描画
                auto errorCode = CreateErrorCodeFromResult(FatalContext.lastResult);

                void* pFontBuffer = Allocate(FontBufferSize, DefaultGraphicsAlign, nullptr);
                TextRenderer renderer;
                NN_ABORT_UNLESS_RESULT_SUCCESS(renderer.Initialize(
                    pFontBuffer,
                    FontBufferSize,
                    stride));
                renderer.SetDisplayBuffer(bufferAddr, LayerWidth, LayerHeight);


                auto envInfo = GetEnvironmentInfo();
                DrawFatalScreen(renderer, bufferAddr, LayerWidth, LayerHeight, stride, errorCode, 1.0f, 1.0f, m_CpuContext);

                if (envInfo.showExtraInfo)
                {
                    DrawDebugScreen(renderer, FatalContext, m_CpuContext, m_CallerProgramId);
                }

                NN_RESULT_SUCCESS;
            }

        private:
            const FatalContext& FatalContext;
            const int LayerWidth, LayerHeight;
            const CpuContext& m_CpuContext;
            nn::Bit64 m_CallerProgramId;
        };

        class LayerManager {
        public:
            LayerManager() NN_NOEXCEPT
                : m_pDisplay(nullptr), m_pLayer(nullptr),
                m_BufferInfo({}), m_BufferQueue({})
            {}

            Result Initialize(
                vi::Display* pDisplay,
                int x,
                int y,
                int z,
                int layerWidth,
                int layerHeight,
                int bufferWidth,
                int bufferHeight,
                vi::LayerSettings settings) NN_NOEXCEPT
            {
                NN_SDK_ASSERT_NOT_NULL(pDisplay);

                const nn::vi::PixelFormat PixelFormat = nn::vi::PixelFormat_Rgba4444;

                m_pDisplay = pDisplay;

                m_BufferInfo.width = bufferWidth;
                m_BufferInfo.height = bufferHeight;
                m_BufferInfo.format = PixelFormat;
                m_BufferInfo.bufferCount = 1;

DumpHeapState("Before create layer.");
                NN_RESULT_DO(
                    nn::vi::CreateLayer(&m_pLayer, m_pDisplay, settings)
                );
DumpHeapState("After create layer.");
                NN_RESULT_DO(nn::vi::SetLayerSize(m_pLayer, layerWidth, layerHeight));
                NN_RESULT_DO(nn::vi::SetLayerZ(m_pLayer, z));

                NN_RESULT_DO(
                    nn::vi::SetLayerPosition(m_pLayer,
                        static_cast<float>(x),
                        static_cast<float>(y))
                );

                NN_RESULT_DO(
                    m_BufferQueue.Initialize(m_pLayer, m_BufferInfo)
                );

                size_t size = m_BufferQueue.GetRequiredMemorySize(m_BufferInfo);
                size_t alignment = m_BufferQueue.GetRequiredMemoryAlignment(m_BufferInfo);

                NN_DETAIL_FATAL_TRACE("layerBuffer: Size = %lld, Alignment = %lld\n", size, alignment);
                void* pMemory = Allocate(size, alignment, nullptr);
                NN_RESULT_THROW_UNLESS(pMemory, nn::fatal::ResultAllocationFailed());

                NN_RESULT_DO(
                    m_BufferQueue.SetScanBuffer(0, pMemory, size)
                );

                NN_DETAIL_FATAL_TRACE("scan buffer: %08x - %08x\n", pMemory, size + reinterpret_cast<uintptr_t>(pMemory));

                NN_RESULT_SUCCESS;
            }

            Result WriteBuffer(IWriter& writer) NN_NOEXCEPT
            {
                void* bufferAddr;
                nn::vi::buffer::BufferQueueHandle* buffer;

DumpHeapState("Before dequeue buffer.");
                NN_RESULT_DO(
                    m_BufferQueue.DequeueBuffer(&buffer)
                );
DumpHeapState("After dequeue buffer.");

                NN_RESULT_DO(
                    m_BufferQueue.GetScanBufferAddress(&bufferAddr, buffer)
                );
                NN_RESULT_THROW_UNLESS(bufferAddr, fatal::ResultNullScanBuffer());

                NN_DETAIL_FATAL_TRACE("actual scan buffer: %08x\n", bufferAddr);

                NN_RESULT_DO(writer.Draw(bufferAddr, m_BufferQueue.GetStride(m_BufferInfo)));

                NN_RESULT_DO(
                    m_BufferQueue.QueueBuffer(buffer)
                );

                NN_RESULT_SUCCESS;
            }
        private:
            vi::Display* m_pDisplay;
            vi::Layer* m_pLayer;
            vi::buffer::BufferInfo m_BufferInfo;
            vi::buffer::BufferQueue m_BufferQueue;
        };

        void HideCurrentContents() NN_NOEXCEPT
        {
            nn::vi::SetContentVisibility(false);
        }

        void InitializeGraphics() NN_NOEXCEPT
        {
            nv::SetGraphicsAllocator(Allocate, Free, Reallocate, nullptr);
            // Initialize nv with system process port name
            nv::SetGraphicsServiceName("nvdrv:s");
            nv::InitializeGraphics(
                Allocate(GraphicsSystemMemorySize, DefaultGraphicsAlign, nullptr),
                GraphicsSystemMemorySize);

            nn::vi::Initialize();
        }

        Result RestoreDisplayAlpha(nn::vi::Display* pDisplay) NN_NOEXCEPT
        {
            NN_RESULT_DO(nn::vi::SetDisplayAlpha(pDisplay, 1.0f));
            NN_DETAIL_FATAL_TRACE("Display alpha restored\n");

            NN_RESULT_SUCCESS;
        }

        Result PrepareInternalDisplay() NN_NOEXCEPT
        {
            nn::vi::Display* pDisplay;

            // 内部ディスプレイ取得
            NN_RESULT_TRY(nn::vi::OpenDisplay(&pDisplay, "Internal"))
                NN_RESULT_CATCH(vi::ResultNotFound)
                {
                    NN_DETAIL_FATAL_WARN("Internal display not found.\n");
                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY;

            NN_UTIL_SCOPE_EXIT{ nn::vi::CloseDisplay(pDisplay); };

            // LCD ON
            NN_RESULT_DO(
                nn::vi::SetDisplayPowerState(pDisplay, nn::vi::PowerState::PowerState_On)
            );

            // アルファ復元
            NN_RESULT_DO(RestoreDisplayAlpha(pDisplay));

            NN_RESULT_SUCCESS;
        }

        Result PrepareExternalDisplay() NN_NOEXCEPT
        {
            nn::vi::Display* pDisplay;

            // 外部ディスプレイ取得
            NN_RESULT_TRY(nn::vi::OpenDisplay(&pDisplay, "External"))
                NN_RESULT_CATCH(vi::ResultNotFound)
            {
                NN_DETAIL_FATAL_WARN("External display not found.\n");
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_RETHROW;
            }
            NN_RESULT_END_TRY;

            NN_UTIL_SCOPE_EXIT{ nn::vi::CloseDisplay(pDisplay); };

            // アルファ復元
            NN_RESULT_DO(RestoreDisplayAlpha(pDisplay));

            NN_RESULT_SUCCESS;
        }

        class BacklightControlTask : public ITask
        {
        public:
            virtual Result Run() NN_NOEXCEPT
            {
                SwitchBacklightOn();
                NN_RESULT_SUCCESS;
            }
            virtual const char* GetTaskName() const
            {
                return "BacklightControlTask";
            }
        };

        BacklightControlTask g_BacklightTask;

        class ShowFatalScreenTask : public ITask
        {
        public:
            ShowFatalScreenTask() NN_NOEXCEPT : m_Context(nullptr), m_CallerProgramId(0), m_BatteryCheckedEvent(nullptr) {}

            void Initialize(const Service::Context& context, nn::Bit64 callerProgramId, nn::os::Event* batteryCheckedEvent) NN_NOEXCEPT
            {
                m_Context = &context;
                m_CallerProgramId = callerProgramId;
                m_BatteryCheckedEvent = batteryCheckedEvent;
            }

            virtual Result Run() NN_NOEXCEPT
            {
                // バッテリーの電圧が一定以上であることを確認してから LCD を触り始める。
                m_BatteryCheckedEvent->Wait();

                NN_RESULT_DO(ShowFatalScreen(m_Context->fatalContext, m_Context->cpuContext, m_CallerProgramId));
                NN_RESULT_SUCCESS;
            }
            virtual const char* GetTaskName() const
            {
                return "ShowFatal";
            }
            virtual size_t GetRequiredStackSize() const
            {
                return 20 * 1024;
            }
        private:
            const Service::Context* m_Context;
            nn::Bit64 m_CallerProgramId;
            nn::os::Event* m_BatteryCheckedEvent;
        };

        ShowFatalScreenTask g_FatalScreenTask;
    } // namespace

    void SwitchBacklightOn() NN_NOEXCEPT
    {
        // バックライト ON
        nn::lbl::Initialize();
        NN_UTIL_SCOPE_EXIT{ nn::lbl::Finalize(); };
        nn::lbl::SwitchBacklightOn(0);
    }

    PixelType GetPixelValue(Color& c) NN_NOEXCEPT
    {
        uint16_t v =
            ((c.bits.a / 16) << 12) |
            ((c.bits.b / 16) << 8) |
            ((c.bits.g / 16) << 4) |
            ((c.bits.r / 16) << 0);
        return v;
    }

    Result ShowFatalScreen(const FatalContext& context, const CpuContext& cpuContext, nn::Bit64 callerProgramId) NN_NOEXCEPT
    {

        InitializeGraphics();
        HideCurrentContents();

        NN_RESULT_DO(PrepareInternalDisplay());
        NN_RESULT_DO(PrepareExternalDisplay());

        nn::vi::Display* pDisplay;
        NN_RESULT_DO(
            nn::vi::OpenDefaultDisplay(&pDisplay)
        );

        // INFO: 文字を描画するレイヤを作成
        NN_FUNCTION_LOCAL_STATIC(LayerManager, primaryLayer);
        int displayWidth;
        int displayHeight;

        const int LayerWidth = LOCAL_TO_LOGICAL(770);
        const int LayerHeight = LOCAL_TO_LOGICAL(400);
        const int BufferWidth = 770;
        const int BufferHeight = 400;
        const int Z = 100;
        NN_RESULT_DO(
            nn::vi::GetDisplayLogicalResolution(&displayWidth, &displayHeight, pDisplay)
        );

        // Reset magnification to default
        NN_RESULT_DO(
            nn::vi::SetDisplayMagnification(pDisplay, 0, 0, displayWidth, displayHeight)
        );

        nn::vi::LayerSettings settings;
        nn::vi::SetLayerSettingsDefaults(&settings);
        settings.Reset<nn::vi::LayerFlags::Fullscreen>();

        NN_RESULT_DO(
            primaryLayer.Initialize(
                pDisplay,
                (displayWidth - LayerWidth) / 2,
                (displayHeight - LayerHeight) / 2,
                Z,
                LayerWidth,
                LayerHeight,
                BufferWidth,
                BufferHeight,
                settings
            )
        );

        // INFO: 文字を表示するレイヤに対して描画処理
        FatalScreenWriter fatalScreenDrawer(context, BufferWidth, BufferHeight, cpuContext, callerProgramId);
        NN_RESULT_DO(
            primaryLayer.WriteBuffer(fatalScreenDrawer)
        );

        NN_RESULT_SUCCESS;
    }

    ITask* InitializeAndGetBacklightControlTask() NN_NOEXCEPT
    {
        return &g_BacklightTask;
    }

    ITask* InitializeAndGetFatalScreenTask(const Service::Context& context, nn::Bit64 callerProgramId, nn::os::Event* batteryCheckedEvent) NN_NOEXCEPT
    {
        g_FatalScreenTask.Initialize(context, callerProgramId, batteryCheckedEvent);
        return &g_FatalScreenTask;
    }
}} // namespace nn::fatalsrv
