﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fatal/detail/fatal_Log.h>
#include <nn/fatalsrv/fatalsrv_FatalContext.h>
#include <nn/fatalsrv/fatalsrv_CpuContext.h>

#include <nn/mem.h>
#include <nn/init.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>

#include <nn/fatal/fatal_Result.h>

#include "fatalsrv_Font.h"
#include "fatalsrv_Memory.h"

#include <algorithm>

#define NN_FATALSRV_GENERATE_STRING(NAME, NUMBER) #NAME,

namespace nn { namespace fatalsrv {

    const char* fatalsrv::Aarch64Context::ValueNameStrings[] =
    {
        NN_FATALSRV_AARCH64_VALUES(NN_FATALSRV_GENERATE_STRING)
    };

    const char* fatalsrv::Aarch32Context::ValueNameStrings[] =
    {
        NN_FATALSRV_AARCH32_VALUES(NN_FATALSRV_GENERATE_STRING)
    };

    namespace {
        bool CallFromExceptionHandler(const CpuContext& context) NN_NOEXCEPT
        {
            return !(context.exceptionType == 0);
        }


        template<typename ContextType, typename PtrType>
        void DumpBacktrace(const ContextType& context, const char* format, TextRenderer& renderer) NN_NOEXCEPT
        {
            Color Green = { { 0, 255, 0, 255 } };
            const int FontSize = 20;
            const int X = 450;
            const int Y = 10 * FontSize;
            const int MaxDisplayCount = 8;


            renderer.SetFontSize(FontSize);
            renderer.SetColor(Green);

            renderer.PutString(X, Y, "Backtrace:");
            auto displayCount = std::min(MaxDisplayCount, context.backtraceCount);
            for (int i = 0; i < displayCount; ++i)
            {
                renderer.PutString(X, Y + (i + 1) * FontSize,
                    format, i, context.backtrace[i] );
            }
        }

        template<typename ContextType, typename PtrType>
        void DumpRegisters(const ContextType& context, const char* format, TextRenderer& renderer) NN_NOEXCEPT
        {
            Color White = { { 255, 255, 255, 255 } };
            const int FontSize = 12;
            const int X = 0;
            const int Y = FontSize * 8;

            renderer.SetFontSize(FontSize);

            const int MaxCountOfColumn = 24;

            for (int i = 0; i < ContextType::ValueNames::ValueNameCount; ++i)
            {
                auto dx = i / MaxCountOfColumn;
                auto dy = i % MaxCountOfColumn;
                if (context.setFlag.Test(i))
                {
                    renderer.SetColor(White);
                    renderer.PutString(X + dx * 200, Y + dy * FontSize, format, ContextType::ValueNameStrings[i], context.r[i]);
                }
            }
        }

        void DrawBasicInfo(TextRenderer& renderer, const FatalContext& fatalContext, const CpuContext& cpuContext, nn::Bit64 callerProramId) NN_NOEXCEPT
        {
            const int FontSize = 20;
            const int X = 450;
            const int Y = 0;

            Color Green = { {0, 255, 0, 255} };
            renderer.SetNewLineHeight(FontSize + 2);
            renderer.SetColor(Green);
            renderer.PutString(X, Y,
                "Program:\n%016llx\n", callerProramId);


            // INFO:
            // 例外ハンドラから呼ばれていない場合（NN_ABORT から呼ばれている場合）は
            // 追加情報を表示する
            if (!CallFromExceptionHandler(cpuContext))
            {
                if (cpuContext.archType == CpuContext::ArchType::Aarch64)
                {
                    renderer.PutString(X, Y + 5 * FontSize,
                        "Start address:\n%016llx", cpuContext.aarch64.programMappedAddr);
                }
                else if (cpuContext.archType == CpuContext::ArchType::Aarch32)
                {
                    renderer.PutString(X, Y + 5 * FontSize,
                        "Start address:\n%08x", cpuContext.aarch32.programMappedAddr);
                }
            }
        }

        void DrawExtraInfo(TextRenderer& renderer, const FatalContext& fatalContext, const CpuContext& cpuContext, nn::Bit64 callerProgramId) NN_NOEXCEPT
        {
            DrawBasicInfo(renderer, fatalContext, cpuContext, callerProgramId);

            // INFO:
            // 例外ハンドラから呼ばれていない場合（NN_ABORT から呼ばれている場合）は
            // 追加情報を表示する
            if (!CallFromExceptionHandler(cpuContext))
            {
                if (cpuContext.archType == CpuContext::ArchType::Aarch64)
                {
                    DumpBacktrace<fatalsrv::Aarch64Context, nn::Bit64>(
                        cpuContext.aarch64, "[%2d] %016llx\n", renderer);
                    DumpRegisters<fatalsrv::Aarch64Context, nn::Bit64>(
                        cpuContext.aarch64, "%s:%016llx", renderer);
                }
                else if (cpuContext.archType == CpuContext::ArchType::Aarch32)
                {
                    DumpBacktrace<fatalsrv::Aarch32Context, nn::Bit32>(
                        cpuContext.aarch32, "[%2d] %08x\n", renderer);
                    DumpRegisters<fatalsrv::Aarch32Context, nn::Bit32>(
                        cpuContext.aarch32, "%s:%08x", renderer);
                }
            }
        }

    } // namespace

    Result DrawDebugScreen(TextRenderer& renderer, const FatalContext& fatalContext, const CpuContext& cpuContext, nn::Bit64 callerProgramId) NN_NOEXCEPT
    {
        DrawExtraInfo(renderer, fatalContext, cpuContext, callerProgramId);
        NN_RESULT_SUCCESS;
    }

}} // namespace nn::fatalsrv
