﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/apm/apm_Api.h>
#include <nn/apm/server/apm_Configuration.h>
#include <nn/fatal/detail/fatal_Log.h>
#include <nn/pcv/pcv.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "fatalsrv_Cpu.h"
#include "fatalsrv_Task.h"

namespace nn { namespace fatalsrv {
    namespace
    {
        const nn::apm::server::ConfigurationEntry& GetPerformanceConfigurationEntry(nn::apm::PerformanceConfiguration configuration) NN_NOEXCEPT
        {
            auto count = sizeof(nn::apm::server::ConfigurationTable) / sizeof(nn::apm::server::ConfigurationTable[0]);
            for (int i = 0; i < count; ++i)
            {
                auto& entry = nn::apm::server::ConfigurationTable[i];
                if (entry.config == configuration)
                {
                    return entry;
                }
            }
            // INFO: 通常到達しないはず
            NN_ABORT("Failed to find performance configuration entry.");
        }

        Result AdjustClock() NN_NOEXCEPT
        {
            NN_DETAIL_FATAL_TRACE("AdjustClock\n");

            nn::pcv::Initialize();
            NN_UTIL_SCOPE_EXIT{ nn::pcv::Finalize(); };

            auto& entry = GetPerformanceConfigurationEntry(nn::apm::server::DefaultPerformanceConfigurationForNormalMode);

            NN_RESULT_DO(
                nn::pcv::SetClockRate(nn::pcv::Module::Module_Gpu, entry.gpuMin));
            NN_RESULT_DO(
                nn::pcv::SetClockRate(nn::pcv::Module::Module_Cpu, entry.cpuMin));
            NN_RESULT_DO(
                nn::pcv::SetClockRate(nn::pcv::Module::Module_Emc, entry.emcMin));

            NN_RESULT_SUCCESS;
        }

        class AdjustClockTask : public ITask
        {
        public:
            virtual Result Run() NN_NOEXCEPT
            {
                NN_RESULT_DO(AdjustClock());
                NN_RESULT_SUCCESS;
            }
            virtual const char* GetTaskName() const
            {
                return "AdjustClockTask";
            }
        };

        AdjustClockTask g_AdjustClockTask;
    }

    ITask* InitializeAndGetAdjustClockTask() NN_NOEXCEPT
    {
        return &g_AdjustClockTask;
    }
}} // namespace nn::fatalsrv
