﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/fan/detail/fan_Log.h>
#include <nn/fan/impl/fan_Lib.h>
#include <nn/fan/impl/fan_Control.h>
#include <nn/gpio/gpio.h>
#include <nn/pwm/pwm.h>

#include "fan_RotationSpeedLevel-hardware.nx.h"
#include "fan_DeviceAccessorGpio-hardware.nx.h"

namespace nn {
namespace fan {
namespace impl {
namespace detail {

DeviceAccessorGpio::DeviceAccessorGpio(FanName name) NN_NOEXCEPT
    : m_Enabled(true),
      m_Mutex(false)
{
    NN_UNUSED(name);

    nn::gpio::Initialize();

    nn::gpio::OpenSession(&m_GpioPadSessionFanEn, nn::gpio::GpioPadName::GpioPadName_FanEn);
    nn::gpio::SetDirection(&m_GpioPadSessionFanEn, nn::gpio::Direction::Direction_Output);

    nn::gpio::OpenSession(&m_GpioPadSessionPwmFan, nn::gpio::GpioPadName::GpioPadName_PwmFan);
    nn::gpio::SetDirection(&m_GpioPadSessionPwmFan, nn::gpio::Direction::Direction_Output);

    UpdateGpio();
}

void DeviceAccessorGpio::SetTargetRotationRate(int rotationRate) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_RotationRate = rotationRate;
    UpdateGpio();
}

void DeviceAccessorGpio::SetFanEnabled(bool enabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_Enabled = enabled;
    UpdateGpio();
}

void DeviceAccessorGpio::SetFanEnabledFromServer(bool enabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_Enabled = enabled;
    UpdateGpio();
}

void DeviceAccessorGpio::UpdateGpio() NN_NOEXCEPT
{
    // m_GpioPadSessionFanEn の設定
    if (m_Enabled && m_RotationRate > 0)
    {
        nn::gpio::SetValue(&m_GpioPadSessionFanEn, nn::gpio::GpioValue::GpioValue_High);
    }
    else
    {
        nn::gpio::SetValue(&m_GpioPadSessionFanEn, nn::gpio::GpioValue::GpioValue_Low);
    }

    // m_GpioPadSessionPwmFan の設定
    if (m_RotationRate >= GetMaxRotationRate())
    {
        nn::gpio::SetValue(&m_GpioPadSessionPwmFan, nn::gpio::GpioValue::GpioValue_High);
    }
    else
    {
        nn::gpio::SetValue(&m_GpioPadSessionPwmFan, nn::gpio::GpioValue::GpioValue_Low);
    }
}

}}}} // nn::fan::impl::detail
