﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/dauth/dauth_Api.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include <nn/eupld/eupld_Result.h>
#include <nn/eupld/server/eupld_Result.http.h>

#include "eupldsrv_Transport.h"
#include "eupldsrv_Request.h"
#include "eupldsrv_ReportStream.h"
#include "eupldsrv_CurlErrorHandler.h"
#include "eupldsrv_HttpStatusCodeHandler.h"

namespace nn    {
namespace eupld {
namespace srv   {

namespace {

const uint64_t ClientId = 0x16e96f76850156d1ull;

size_t ReadCallback(void* pBuffer, size_t size, size_t nmemb, void* arg)
NN_NOEXCEPT
{
    uint32_t      readCount = static_cast<uint32_t>(size * nmemb);
    ReportStream* pStream   = reinterpret_cast<ReportStream*>(arg);
    nn::Result    result    = pStream->Read(&readCount, static_cast<uint8_t*>(pBuffer), readCount);

    return result.IsFailure() ?
           static_cast<size_t>(-1) :
           readCount;
}

int ProgressCallback(void* pContext, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow)
NN_NOEXCEPT
{
    NN_UNUSED(pContext);
    NN_UNUSED(dlTotal);
    NN_UNUSED(dlNow);
    NN_UNUSED(ulTotal);
    NN_UNUSED(ulNow);

    if (!Transport::IsTransportEnabled())
    {
        NN_SDK_LOG("[eupld]: ProgressCallback : Transport is canceled.\n");
        return 1;
    }
    return 0;
}

const char* BuildString(const char* pString1, const char* pString2)
NN_NOEXCEPT
{
    static char tmpBuffer[1024];
    util::SNPrintf(tmpBuffer, sizeof(tmpBuffer), "%s%s", pString1, pString2);
    return tmpBuffer;
}

} // ~nn::eupld::srv::<anonymous>

bool         Transport::m_TransportEnabled                  = true;
bool         Transport::m_UrlSet                            = false;
char         Transport::m_Url[MaxUrlLength]                 = "UrlIsNotSet";
char         Transport::m_Environment[MaxEnvironmentLength] = "dd1";
nn::socket::SystemConfigLightDefaultWithMemory<TcpSocketCount, 0>
             Transport::m_SocketConfiguration(1);

nn::Result Transport::UploadReports(Request* pRequest)
NN_NOEXCEPT
{
    CURL*        pCurl;
    curl_slist*  pHeader = nullptr;
    ReportStream reportStream(pRequest->GetUploadList(), pRequest->IsAuto());
    long         replyCode = 0;
    const long   ExpectedServerReplyCode = 200;

    if (reportStream.GetStreamSize() == 0)
    {
        return ResultSuccess();
    }

    nn::TimeSpan tokenExpireTime;
    int tokenSize;
    char token[dauth::RequiredBufferSizeForDeviceAuthenticationToken];
    NN_RESULT_DO(dauth::AcquireDeviceAuthenticationToken(&tokenExpireTime, &tokenSize, token, sizeof(token), ClientId, false, nullptr));

    NN_RESULT_DO(HandleCurlError(curl_global_init(CURL_GLOBAL_DEFAULT)));

    if ((pCurl = curl_easy_init()) == nullptr)
    {
        curl_global_cleanup();
        return nn::eupld::srv::ResultCurlEasyInitFailure();
    }

    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(pHeader);
        curl_easy_cleanup(pCurl);
        curl_global_cleanup();
    };

    NN_RESULT_DO(SetupCurlHandle(pCurl));

    pHeader = curl_slist_append(pHeader, "Content-Type: application/x-msgpack");
    pHeader = curl_slist_append(pHeader, BuildString("X-Nintendo-Device-Authentication-Token:", token));
    pHeader = curl_slist_append(pHeader, BuildString("X-Nintendo-Error-Report-Send-Type:", (pRequest->IsAuto() ? "auto" : "manual")));
    pHeader = curl_slist_append(pHeader, BuildString("X-Nintendo-Environment:", m_Environment));

    curl_easy_setopt(pCurl, CURLOPT_URL,            m_Url);
    curl_easy_setopt(pCurl, CURLOPT_VERBOSE,        1L);
    curl_easy_setopt(pCurl, CURLOPT_CONNECTTIMEOUT, 15L);
    curl_easy_setopt(pCurl, CURLOPT_POST,           1);
    curl_easy_setopt(pCurl, CURLOPT_READFUNCTION,   ReadCallback);
    curl_easy_setopt(pCurl, CURLOPT_READDATA,       reinterpret_cast<void*>(&reportStream));
    curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE,  reportStream.GetStreamSize());
    curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER,     pHeader);
    curl_easy_setopt(pCurl, CURLOPT_LOW_SPEED_LIMIT,1L);
    curl_easy_setopt(pCurl, CURLOPT_LOW_SPEED_TIME, 30L);
    curl_easy_setopt(pCurl, CURLOPT_TIMEOUT,        60L);
    curl_easy_setopt(pCurl, CURLOPT_XFERINFOFUNCTION, ProgressCallback);
    curl_easy_setopt(pCurl, CURLOPT_NOPROGRESS, 0L);

    NN_RESULT_DO(HandleCurlError(curl_easy_perform(pCurl)));
    NN_RESULT_DO(HandleCurlError(curl_easy_getinfo(pCurl, CURLINFO_RESPONSE_CODE, &replyCode)));

    if (replyCode != ExpectedServerReplyCode)
    {
        return HandleHttpStatusCode(replyCode);
    }

    reportStream.MarkAsTransmitted();

    return ResultSuccess();
}

nn::Result Transport::Upload(Request* pRequest)
NN_NOEXCEPT
{
    ReportUploadList* pList = pRequest->GetUploadList();
    nn::Result        result;

    if (!m_UrlSet)
    {
        for (uint32_t i = 0; i < pList->reportCount; i++)
        {
            pList->Report[i].result = nn::eupld::ResultUrlNotSet();
        }

        #if defined(NN_SDK_BUILD_RELEASE)
        return nn::eupld::ResultUrlNotSet();
        #else
        // Temporarily, to allow development return Success.
        // By default, upload URL is not set. If you set it,
        // you better make sure to import SSL certificates,
        // and request access to the server.
        return ResultSuccess();
        #endif
    }

    if ((result = nn::nifm::InitializeSystem()).IsSuccess() &&
        (result = nn::socket::Initialize(m_SocketConfiguration)).IsSuccess() &&
        (result = nn::ssl::Initialize()).IsSuccess())
    {
        if(pRequest->IsAuto())
        {
            nn::nifm::NetworkConnection networkConnection;

            nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess);
            networkConnection.SubmitRequest();

            while (networkConnection.IsRequestOnHold() && m_TransportEnabled)
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            result = (m_TransportEnabled && networkConnection.IsAvailable()) ?
                      UploadReports(pRequest) :
                      nn::eupld::ResultNetworkNotAvailable();

            networkConnection.CancelRequest();
        }
        else
        {
            // In the manual upload case, the caller needs to submit the network connection request.
            result = UploadReports(pRequest);
        }

    }

    nn::ssl::Finalize();
    nn::socket::Finalize();

    return result;
}

nn::Result Transport::SetUrl(const char* pUrlString, uint32_t urlLength)
NN_NOEXCEPT
{
    if (urlLength > MaxUrlLength)
    {
        return nn::eupld::ResultInvalidArgument();
    }

    std::memcpy(m_Url, pUrlString, urlLength);
    m_UrlSet = true;

    return ResultSuccess();
}

nn::Result Transport::SetEnvironmentId()
NN_NOEXCEPT
{
    nn::nsd::EnvironmentIdentifier environmentIdentifier;

    nn::nsd::GetEnvironmentIdentifier(&environmentIdentifier);

    NN_STATIC_ASSERT(sizeof(environmentIdentifier.value) <= MaxEnvironmentLength);

    std::memcpy(m_Environment, environmentIdentifier.value, sizeof(environmentIdentifier.value));

    return ResultSuccess();
}

void Transport::TransportControl(bool enable)
NN_NOEXCEPT
{
    m_TransportEnabled = enable;
}

bool Transport::IsTransportEnabled()
NN_NOEXCEPT
{
    return m_TransportEnabled;
}

}}}
