﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/eupld/eupld_Result.h>

#include "eupldsrv_IRequest.h"
#include "eupldsrv_Request.h"

namespace nn    {
namespace eupld {
namespace srv   {

nn::Result IRequest::UploadAll()
NN_NOEXCEPT
{
    nn::Result           result;
    nn::erpt::ReportList reportList;
    nn::erpt::Manager    reportManager;

    if ((result = reportManager.Initialize()).IsFailure() ||
        (result = reportManager.GetReportList(reportList)).IsFailure())
    {
        return result;
    }

    return m_pRequestEntry->Initialize(reportList, false);
}

nn::Result IRequest::UploadSelected(const nn::sf::InBuffer& reportIdArray)
NN_NOEXCEPT
{
    const nn::erpt::ReportId* pReportIds  = reinterpret_cast<const nn::erpt::ReportId*>(reportIdArray.GetPointerUnsafe());
    uint32_t                  reportCount = static_cast<uint32_t>(reportIdArray.GetSize() / sizeof(nn::erpt::ReportId));

    if (reportCount == 0 || reportCount > nn::erpt::NumberOfReports)
    {
        return nn::eupld::ResultInvalidArgument();
    }

    return m_pRequestEntry->Initialize(pReportIds, reportCount, false);
}

nn::Result IRequest::GetUploadStatus(const nn::sf::OutBuffer& status)
NN_NOEXCEPT
{
    ReportUploadList* pList = reinterpret_cast<ReportUploadList*>(status.GetPointerUnsafe());

    if (status.GetSize() != sizeof(ReportUploadList))
    {
        return nn::eupld::ResultInvalidArgument();
    }

    *pList = *(m_pRequestEntry->GetUploadList());

    return ResultSuccess();
}

nn::Result IRequest::CancelUpload()
NN_NOEXCEPT
{
    return m_pRequestEntry->Cancel();
}

nn::Result IRequest::GetResult()
NN_NOEXCEPT
{
    return m_pRequestEntry->GetResult();
}

nn::Result IRequest::Initialize(nn::sf::Out<nn::sf::NativeHandle> handleOut)
NN_NOEXCEPT
{
    if (m_pRequestEntry != nullptr)
    {
        return nn::eupld::ResultAlreadyInitialized();
    }
    if ((m_pRequestEntry = new Request(&m_SystemEvent)) == nullptr)
    {
        return nn::eupld::ResultOutOfMemory();
    }

    m_pRequestEntry->AddReference();
    handleOut.Set(nn::sf::NativeHandle(m_SystemEvent.GetReadableHandle(), false));

    return ResultSuccess();
}

IRequest::IRequest()
NN_NOEXCEPT :
    m_SystemEvent(nn::os::EventClearMode_AutoClear, true),
    m_pRequestEntry(nullptr)
{

}

IRequest::~IRequest()
NN_NOEXCEPT
{
    if (m_pRequestEntry != nullptr)
    {
        m_pRequestEntry->Finalize();

        if (m_pRequestEntry->RemoveReference())
        {
            delete m_pRequestEntry;
        }
    }
};

}}}
