﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdint>
#include <mutex>

#include <nn/fs/fs_RightsId.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/util/util_ScopedTransaction.h>
#include <nn/util/util_Span.h>
#include <nn/es/es_Configuration.h>
#include <nn/es/es_ServiceLog.h>
#include <nn/es/es_TypesForInner.h>
#include "es_Clock.h"
#include "es_RightsAvailabilityMaster.h"
#include "es_ActiveRightsContextManager.h"

namespace nn { namespace es {

RightsAvailabilityMaster& GetRightsAvailabilityMaster() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(RightsAvailabilityMaster, g_RightsAvailabilityMaster);
    return g_RightsAvailabilityMaster;
}

namespace
{
    bool IsELicenseStillValid(const ELicenseInfoForSystem& eLicense) NN_NOEXCEPT
    {
        if (eLicense.expireDate.IsPermanent())
        {
            return true;
        }

        time::PosixTime time = {};

        bool isGettingTimeSucceeded = true;
        NN_RESULT_ABORTING_BLOCK
        {
            NN_RESULT_TRY(Clock::GetCurrentTime(&time))
                NN_RESULT_CATCH(time::ResultClockInvalid)   // 時間の取得に失敗した場合
                {
                    isGettingTimeSucceeded = false;
                }
                NN_RESULT_CATCH_ALL                         // その他の失敗も同様の扱いにする
                {
                    isGettingTimeSucceeded = false;
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        };

        return isGettingTimeSucceeded && eLicense.expireDate.time >= time;
    }

    bool IsELicensePermanent(const ELicenseInfoForSystem& eLicense) NN_NOEXCEPT
    {
        return eLicense.expireDate.IsPermanent();
    }

    bool FindDeviceShareableRights(ELicenseInfoForSystem* pOutELicenseInfo, ELicenseList* pELicenseList, RightsId rightsId, bool onlyPermanentRights) NN_NOEXCEPT
    {
        ELicenseInfoForSystem eLicenseList[account::UserCountMax] = {};
        int eLicenseCount = pELicenseList->Find(eLicenseList, sizeof(eLicenseList) / sizeof(eLicenseList[0]), rightsId);

        auto end = eLicenseList + eLicenseCount;
        auto itr = std::find_if(eLicenseList, end,
            [&](const ELicenseInfoForSystem& eLicense) NN_NOEXCEPT
            {
                return true
                    && eLicense.scope == ELicenseScope::Device                              // デバイス共有である
                    && IsELicenseStillValid(eLicense)                                       // 有効期限内である
                    && (onlyPermanentRights ? IsELicensePermanent(eLicense) : true)         // permanent な権利のみを検索する場合、permanent な権利である
                ;
            }
        );

        if (itr == end)
        {
            return false;
        }

        *pOutELicenseInfo = *itr;
        return true;
    }

    bool FindAccountRestrictedRights(ELicenseInfoForSystem* pOutELicenseInfo, ELicenseList* pELicenseList, const ELicenseUserId userIdList[], size_t userIdCount, RightsId rightsId, bool onlyPermanentRights) NN_NOEXCEPT
    {
        if (userIdCount == 0)
        {
            return false;
        }

        ELicenseInfoForSystem eLicenseList[account::UserCountMax] = {};
        int eLicenseCount = pELicenseList->Find(eLicenseList, sizeof(eLicenseList) / sizeof(eLicenseList[0]), rightsId);

        auto userIdListEnd = userIdList + userIdCount;
        bool hasRights = std::all_of(userIdList, userIdListEnd,
            [&](const ELicenseUserId& user) NN_NOEXCEPT
            {
                auto end = eLicenseList + eLicenseCount;
                auto itr = std::find_if(eLicenseList, end,
                    [&](const ELicenseInfoForSystem& eLicense) NN_NOEXCEPT
                    {
                        return true
                            && eLicense.scope == ELicenseScope::Account                     // アカウント限定である
                            && user.id == eLicense.ownerNaId.id                             // 利用者と所有者が一致している
                            && IsELicenseStillValid(eLicense)                               // 有効期限内である
                            && (onlyPermanentRights ? IsELicensePermanent(eLicense) : true) // permanent な権利のみを検索する場合、permanent な権利である
                        ;
                    }
                );

                if (itr == end)
                {
                    return false;
                }

                // TORIAEZU: 最後の ELicenseId を返す
                // 複数の RestrictedRightsUser を登録しての権利の使用を正しく記録するためには修正が必要
                *pOutELicenseInfo = *itr;
                return true;
            }
        );

        return hasRights;
    }

    void SetRightsStatus(RightsStatus* outValue, RightsId rightsId, ELicenseId eLicenseId, ExpireDate expireDate, ELicenseOwnerId ownerId, bool isPermanent, bool recommendsServerInteraction, RightsStatus::Status status) NN_NOEXCEPT
    {
        outValue->rights.rightsId = rightsId;
        outValue->rights.eLicenseId = eLicenseId;
        outValue->rights.expireDate = expireDate;
        outValue->rights._eLicenseOwnerIdValue = ownerId.id;
        outValue->rights._flags.Reset();
        outValue->status = status;

        if (isPermanent)
        {
            outValue->rights._flags.Set(RightsStatusFlag_Permanent::Index);
        }

        if (recommendsServerInteraction)
        {
            outValue->rights._flags.Set(RightsStatusFlag_RecommendsServerInteraction::Index);
        }
    }

    template <typename RightsIdType, typename GetRightsIdFuncType>
    Result CheckRightsStatusImpl(
        RightsStatus outStatusList[], size_t statusCount,
        const RightsIdType rightsIdList[], size_t rightsIdCount,
        const ELicenseUserId userIdList[], size_t userIdCount,
        const TicketDatabase* pCommonDatabase, const TicketDatabase* pPersonalizedDatabase, const TicketMetaRecordDatabase* pTicketMetaRecordDatabase, ELicenseList* pELicenseList,
        bool onlyAllAccountRights,
        GetRightsIdFuncType func) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(statusCount == rightsIdCount);

        for (size_t i = 0; i < rightsIdCount; i++)
        {
            auto& status = outStatusList[i];
            const auto& rightsId = rightsIdList[i];

            SetRightsStatus(&status, func(rightsId), InvalidELicenseId, {{0}}, InvalidELicenseOwnerId, false, false, RightsStatus::Status::NotAvailable);

            // まずコモンチケットを確認する
            // コモンチケットを持っている場合は権利がある
            if (pCommonDatabase->CountTicket(rightsId) > 0)
            {
                SetRightsStatus(&status, func(rightsId), InvalidELicenseId, {{ExpireDateValueAsPermanent}}, InvalidELicenseOwnerId, true, false, RightsStatus::Status::Available);
                continue;
            }

            // eLicense 配下にいるか確認する
            // eLicense 配下の場合は eLicense を確認する
            if (pELicenseList->IsELicenseEnabled())
            {
                ELicenseInfoForSystem eLicenseList[account::UserCountMax] = {};
                int eLicenseCount = pELicenseList->Find(eLicenseList, sizeof(eLicenseList) / sizeof(eLicenseList[0]), func(rightsId));

                // eLicense がない場合は権利がない
                if (eLicenseCount <= 0)
                {
                    // RightsStatus::Status::NotAvailable は既に設定済み
                    continue;
                }

                // チケットがない場合は権利がない
                if (pPersonalizedDatabase->CountTicket(rightsId) <= 0)
                {
                    // RightsStatus::Status::NotAvailable は既に設定済み
                    continue;
                }

                ELicenseInfoForSystem tmpELicenseInfo;
                // まずデバイス共有の eLicense の永続的な権利があるかを確認する
                // 1つでもデバイス共有の権利を持っていれば権利がある
                if (FindDeviceShareableRights(&tmpELicenseInfo, pELicenseList, func(rightsId), true))
                {
                    SetRightsStatus(&status, func(rightsId), tmpELicenseInfo.eLicenseId, tmpELicenseInfo.expireDate, tmpELicenseInfo.ownerNaId, true, tmpELicenseInfo.IsServerInteractionRecommended(), RightsStatus::Status::Available);
                    continue;
                }

                if (!onlyAllAccountRights)
                {
                    // 次にアカウントに限定された eLicense の永続的な権利があるかを確認する
                    // 登録されている全てのアカウントが権利を持っていれば権利がある
                    if (FindAccountRestrictedRights(&tmpELicenseInfo, pELicenseList, userIdList, userIdCount, func(rightsId), true))
                    {
                        SetRightsStatus(&status, func(rightsId), tmpELicenseInfo.eLicenseId, tmpELicenseInfo.expireDate, tmpELicenseInfo.ownerNaId, true, tmpELicenseInfo.IsServerInteractionRecommended(), RightsStatus::Status::AvailableSinceAccountRestricted);
                        continue;
                    }
                }

                // 次にデバイス共有の eLicense の一時的な権利があるかを確認する
                // 1つでもデバイス共有の権利を持っていれば権利がある
                if (FindDeviceShareableRights(&tmpELicenseInfo, pELicenseList, func(rightsId), false))
                {
                    SetRightsStatus(&status, func(rightsId), tmpELicenseInfo.eLicenseId, tmpELicenseInfo.expireDate, tmpELicenseInfo.ownerNaId, false, tmpELicenseInfo.IsServerInteractionRecommended(), RightsStatus::Status::Available);
                    continue;
                }

                if (!onlyAllAccountRights)
                {
                    // 次にアカウントに限定された eLicense の一時的な権利があるかを確認する
                    // 登録されている全てのアカウントが権利を持っていれば権利がある
                    if (FindAccountRestrictedRights(&tmpELicenseInfo, pELicenseList, userIdList, userIdCount, func(rightsId), false))
                    {
                        SetRightsStatus(&status, func(rightsId), tmpELicenseInfo.eLicenseId, tmpELicenseInfo.expireDate, tmpELicenseInfo.ownerNaId, false, tmpELicenseInfo.IsServerInteractionRecommended(), RightsStatus::Status::AvailableSinceAccountRestricted);
                        continue;
                    }
                }

                for (const auto& eLicense : util::MakeSpan(eLicenseList, eLicenseCount))
                {
                    if (IsELicenseStillValid(eLicense))
                    {
                        SetRightsStatus(&status, func(rightsId), InvalidELicenseId, {{0}}, eLicense.ownerNaId, false, false, RightsStatus::Status::HasAccountRestrictedRights);
                        break;
                    }
                }
            }

            // そうでない場合はパーソナライズドチケットを確認する
            // 機器認証パーソナライズドチケットを持っている場合は権利がある
            else
            {
                if (pPersonalizedDatabase->HasDeviceLinkedTicket(func(rightsId)))
                {
                    SetRightsStatus(&status, func(rightsId), InvalidELicenseId, {{ExpireDateValueAsPermanent}}, InvalidELicenseOwnerId, true, false, RightsStatus::Status::Available);
                    continue;
                }
            }

            // 予約状態かを確認する
            if (pTicketMetaRecordDatabase->HasTicketMetaRecord(func(rightsId), TicketMetaRecordType_Prepurchase))
            {
                SetRightsStatus(&status, func(rightsId), InvalidELicenseId, {{0}}, InvalidELicenseOwnerId, false, false, RightsStatus::Status::Prepurchased);
                continue;
            }
        }

        NN_RESULT_SUCCESS;
    }
}

nn::Result RightsAvailabilityMaster::CheckRightsStatusIncludingKeyId(nn::es::RightsStatus outStatusList[], size_t statusCount, const nn::es::RightsIdIncludingKeyId rightsIdList[], size_t rightsIdCount, const nn::es::ELicenseUserId userIdList[], size_t userIdCount, bool onlyAllAccountRights) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckRightsStatusImpl(
        outStatusList, statusCount,
        rightsIdList, rightsIdCount,
        userIdList, userIdCount,
        m_pCommonDatabase, m_pPersonalizedDatabase, m_pTicketMetaRecordDatabase, m_pELicenseList,
        onlyAllAccountRights,
        [](const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
        {
            return rightsId.GetRightsId();
        }
    ));

    NN_RESULT_SUCCESS;
}

nn::Result RightsAvailabilityMaster::CheckRightsStatus(nn::es::RightsStatus outStatusList[], size_t statusCount, const nn::es::RightsId rightsIdList[], size_t rightsIdCount, const nn::es::ELicenseUserId userIdList[], size_t userIdCount, bool onlyAllAccountRights) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckRightsStatusImpl(
        outStatusList, statusCount,
        rightsIdList, rightsIdCount,
        userIdList, userIdCount,
        m_pCommonDatabase, m_pPersonalizedDatabase, m_pTicketMetaRecordDatabase, m_pELicenseList,
        onlyAllAccountRights,
        [](const es::RightsId& rightsId) NN_NOEXCEPT
        {
            return rightsId;
        }
    ));

    NN_RESULT_SUCCESS;
}

sf::SharedPointer<es::IActiveRightsContext> RightsAvailabilityMaster::CreateActiveRightsContext(es::ETicketServiceImpl* pImpl) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_MutexContextList)> lk(m_MutexContextList);

    for (auto&& pContext : m_pContextList)
    {
        if (pContext.expired())
        {
            // 一旦、前の権利コンテキストへのポインタを破棄
            pContext.reset();

            // 新たな権利コンテキストの作成
            auto p = std::make_shared<ActiveRightsContext>(pImpl);
            pContext = p;
            return sf::CreateSharedObject<es::IActiveRightsContext>(p);
        }
    }
    NN_ABORT();
}

Result RightsAvailabilityMaster::UpdateAllRightsIdListsValidity() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("RightsAvailabilityMaster::UpdateAllRightsIdListsValidity()\n");

    {
        std::lock_guard<decltype(m_MutexContextList)> lk(m_MutexContextList);

        // 全権利コンテキストの eLicense 情報を更新
        for (auto& context : m_pContextList)
        {
            auto p = context.lock();
            if (!(p && p->m_pUsingRightsArray))
            {
                continue;
            }
            p->CheckRightsIdListValidity();
        }
    }

    // 更新後の eLicense 情報に基づいて、必要に応じて外部鍵を再登録
    this->SetAlreadyRegisteredTitleKeys(false);
    NN_RESULT_SUCCESS;
}

nn::Result RightsAvailabilityMaster::GetTitleKey(nn::es::AesKey* outKey, const nn::es::RightsIdIncludingKeyId& rightsId, int keyGeneration) NN_NOEXCEPT
{
    TicketId ticketId;
    NN_RESULT_TRY(m_pCommonDatabase->FindTicket(&ticketId, rightsId))
        NN_RESULT_CATCH(ResultRightsIdNotFound)
        {
            NN_RESULT_DO(m_pPersonalizedDatabase->FindTicket(&ticketId, rightsId));
            NN_RESULT_DO(m_pPersonalizedDatabase->GetTitleKey(outKey, rightsId, ticketId, keyGeneration));

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    NN_RESULT_DO(m_pCommonDatabase->GetTitleKey(outKey, rightsId, ticketId, keyGeneration));

    NN_RESULT_SUCCESS;
}

Result RightsAvailabilityMaster::RegisterTitleKeys(
            const util::Span<const RightsIdIncludingKeyId> rightsIdList,
            const util::Span<const int32_t> keyGenerationList,
            const util::Span<const ELicenseUserId> userIdList
        )
{
    NN_SDK_ASSERT(rightsIdList.size() == keyGenerationList.size());

    std::lock_guard<decltype(m_MutexTitleKeyList)> lk(m_MutexTitleKeyList);

    // 受け取った鍵の数だけループ
    for (std::ptrdiff_t i = 0; i < rightsIdList.size(); ++i)
    {
        const auto& rightsId      = rightsIdList[i];
        const auto& keyGeneration = keyGenerationList[i];

        // 既に鍵リストに登録済みの場合は、登録をスキップ
        if ( std::find_if(
                m_TitleKeyList.begin(),
                m_TitleKeyList.end(),
                [&](const UsingTitleKey& usingTitleKey) NN_NOEXCEPT
                {
                    return usingTitleKey.rightsIdIncludingKeyId == rightsId;
                }
            ) != m_TitleKeyList.end() )
        {
            continue;
        }

        // 鍵リストに空きがあるか
        NN_RESULT_THROW_UNLESS(
            m_TitleKeyList.size() < m_TitleKeyList.max_size(),
            ResultUsingTitleKeyListFull()
        );

        // 外部鍵の登録は、通常は権利リストへの RightsId 登録の後に行われる。
        // 全権利コンテキスト中のいずれかの権利リストに登録済みかをチェックして
        // 権利リストに RightsId があるものは鍵リストの管理対象とする。
        //
        // ただし、NAND に install した nsp を DevMenuCommand application launch
        // でフローティング起動した場合は、権利コンテキスト作成前に外部鍵の
        // RegisterTitleKeys() が行われるため、開発機に限りこれを許容する。
        // この場合でも、oe::Initialize() でアプリが am と接続した後には、
        // 前記の通り権利コンテキスト作成、RightsId 登録、外部鍵登録が
        // 通常通り行なわれるため、アプリ起動後は特別な実装分岐は不要。
        //
        bool isFound = false;
        {
            std::lock_guard<decltype(m_MutexContextList)> lk2(m_MutexContextList);
            for (auto&& context : m_pContextList)
            {
                auto p = context.lock();
                if (p && p->IsValidRightsIdListed(rightsId.GetRightsId()))
                {
                    isFound = true;
                    break;
                }
            }
        }

        if (IsProdMode() && !isFound)
        {
            NN_ES_SERVICE_LOG_TRACE("RightsAvailabilityMaster::RegisterTitleKeys(rightsId=0x%016llx) Failed.\n", rightsId.GetRightsId());
            NN_RESULT_THROW( ResultRightsNotAvailable() );
        }

        // タイトル鍵を取得
        UsingTitleKey titleKey = {};
        {
            es::AesKey aesKey;
            NN_RESULT_DO(GetTitleKey(&aesKey, rightsId, keyGeneration));

            titleKey.rightsIdIncludingKeyId = rightsId;
            memcpy(titleKey.accessKey.value, aesKey._data, sizeof(aesKey._data));
            NN_RESULT_DO(fs::RegisterExternalKey(titleKey.fsRightsId, titleKey.accessKey));
        }

        // 鍵リストに追加
        m_TitleKeyList.push_back(titleKey);
    }

    NN_RESULT_SUCCESS;
}

Result RightsAvailabilityMaster::SetAlreadyRegisteredTitleKeys(bool forcibly) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_MutexTitleKeyList)> lk(m_MutexTitleKeyList);

    if (forcibly)
    {
        // 一度タイトル鍵リストに登録されたものを再度 fs に登録
        for (auto& titleKey : util::MakeSpan(m_TitleKeyList.begin(), m_TitleKeyList.end()))
        {
            NN_RESULT_DO(fs::RegisterExternalKey(titleKey.fsRightsId, titleKey.accessKey));
        }
        NN_RESULT_SUCCESS;
    }

    // 権利が有効なもののみ fs に再登録する
    for (auto& titleKey : util::MakeSpan(m_TitleKeyList.begin(), m_TitleKeyList.end()))
    {
        std::lock_guard<decltype(m_MutexContextList)> lk2(m_MutexContextList);
        for (auto&& context : m_pContextList)
        {
            auto p = context.lock();
            if (p && p->IsValidRightsIdListed(titleKey.rightsIdIncludingKeyId.GetRightsId()))
            {
                NN_RESULT_DO(fs::RegisterExternalKey(titleKey.fsRightsId, titleKey.accessKey));
                break;
            }
        }
    }
    NN_RESULT_SUCCESS;
}

Result RightsAvailabilityMaster::UnregisterAllTitleKeys() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_MutexTitleKeyList)> lk(m_MutexTitleKeyList);

    NN_RESULT_DO(fs::UnregisterAllExternalKey());
    m_TitleKeyList.clear();
    NN_RESULT_SUCCESS;
}

}} // namespace nn::es
