﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include "es_ELicenseList.h"

namespace nn { namespace es {

bool ELicenseList::Insert(ELicenseInfoForSystem license) NN_NOEXCEPT
{
    bool isSuccess = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT -> Result
        {
            if (record.state == ELicenseRecord::State::Empty)
            {
                record.license = license;
                record.state = ELicenseRecord::State::Inserted;

                isSuccess = true;
                *isEnd = true;
            }

            NN_RESULT_SUCCESS;
        }
    ));

    return isSuccess;
}

void ELicenseList::Commit(const ELicenseArchiveId& archiveId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT -> Result
        {
            if (record.state == ELicenseRecord::State::Inserted)
            {
                record.license.archiveId = archiveId;
                record.state = ELicenseRecord::State::Valid;
            }

            NN_RESULT_SUCCESS;
        }
    ));
}

void ELicenseList::Revert() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT-> Result
        {
            if (record.state == ELicenseRecord::State::Inserted)
            {
                record.state = ELicenseRecord::State::Empty;
            }

            NN_RESULT_SUCCESS;
        }
    ));
}

int ELicenseList::Count() NN_NOEXCEPT
{
    int count = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT-> Result
        {
            if (record.state == ELicenseRecord::State::Valid)
            {
                count++;
            }

            NN_RESULT_SUCCESS;
        }
    ));

    return count;
}

void ELicenseList::Delete(ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT-> Result
        {
            if (record.license.ownerNaId == ownerId)
            {
                record.state = ELicenseRecord::State::Empty;
            }

            NN_RESULT_SUCCESS;
        }
    ));
}

void ELicenseList::DeleteAll() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT->Result
        {
            record.state = ELicenseRecord::State::Empty;
            NN_RESULT_SUCCESS;
        }
    ));
}

int ELicenseList::List(ELicenseId outIdList[], int idCount) NN_NOEXCEPT
{
    int count = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT-> Result
        {
            if (record.state == ELicenseRecord::State::Valid)
            {
                outIdList[count] = record.license.eLicenseId;
                count++;

                if (count >= idCount)
                {
                    *isEnd = true;
                }
            }

            NN_RESULT_SUCCESS;
        }
    ));

    return count;
}

int ELicenseList::Find(ELicenseInfoForSystem outInfoList[], int infoCount, const ELicenseId idList[], int idCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(infoCount, idCount);

    int count = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT-> Result
        {
            if (record.state == ELicenseRecord::State::Valid)
            {
                auto idListEnd = idList + idCount;
                auto result = std::find(idList, idListEnd, record.license.eLicenseId);

                if (result != idListEnd)
                {
                    outInfoList[count] = record.license;
                    count++;
                }
            }

            NN_RESULT_SUCCESS;
        }
    ));

    return count;
}

int ELicenseList::Find(ELicenseInfoForSystem outInfoList[], int infoCount, RightsId rightsId) NN_NOEXCEPT
{
    int count = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(Traverse(
        [&](bool* isEnd, ELicenseRecord& record) NN_NOEXCEPT-> Result
        {
            if (record.state == ELicenseRecord::State::Valid && record.license.rightsId == rightsId)
            {
                outInfoList[count] = record.license;
                count++;

                if (count >= infoCount)
                {
                    *isEnd = true;
                }
            }

            NN_RESULT_SUCCESS;
        }
    ));

    return count;
}

}} // namespace nn::es
