﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es/es_Result.h>
#include <nn/util/util_FormatString.h>
#include "es_ELicenseArchiveAdaptor.h"

namespace nn { namespace es {

Result ELicenseArchiveAdaptor::UpdateImpl(const JsonPathType& path, const char* pValue, int length) NN_NOEXCEPT
{
    // eLicense が有効なときだけ解釈する
    if (m_ELicense)
    {
        if (path.Match(GetCurrentELicensePath("elicense_id")))
        {
            NN_RESULT_THROW_UNLESS(length == ELicenseId::StringSize && m_ELicense->eLicenseId.FromString(pValue), ResultELicenseArchiveELicenseIdParseError());
        }
        else if (path.Match(GetCurrentELicensePath("rights_id")))
        {
            char* pEnd;
            m_ELicense->rightsId = std::strtoull(pValue, &pEnd, 16);
            NN_RESULT_THROW_UNLESS(*pEnd == '\0', ResultELicenseArchiveRightsIdParseError());
        }
        else if (path.Match(GetCurrentELicensePath("ticket_owner_na_id")))
        {
        }
        else if (path.Match(GetCurrentELicensePath("scope")))
        {
            if (util::Strncmp(pValue, "account", length) == 0)
            {
                m_ELicense->scope = ELicenseScope::Account;
            }
            else if (util::Strncmp(pValue, "device", length) == 0)
            {
                m_ELicense->scope = ELicenseScope::Device;
            }
            else
            {
                NN_RESULT_THROW(ResultELicenseArchiveScopeParseError());
            }
        }
    }

    if (path.Match("$.challenge"))
    {
        m_Challenge.emplace();
        char* pEnd;
        m_Challenge->value = std::strtoull(pValue, &pEnd, 16);
        NN_RESULT_THROW_UNLESS(*pEnd == '\0', ResultELicenseArchiveChallengeParseError());
    }
    else if (path.Match("$.elicense_archive_id"))
    {
        m_ELicenseArchiveId.emplace();
        NN_RESULT_THROW_UNLESS(length == ELicenseArchiveId::StringSize && m_ELicenseArchiveId->FromString(pValue), ResultELicenseArchiveELicenseArchiveIdParseError());
    }
    else if (path.Match("$.elicense_owner_na_id"))
    {
        char* pEnd;
        ELicenseOwnerId id = { std::strtoull(pValue, &pEnd, 16) };
        m_ELicenseOwnerId = id;
        NN_RESULT_THROW_UNLESS(*pEnd == '\0', ResultELicenseArchiveELicenseOwnerNaIdParseError());
    }

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveAdaptor::UpdateImpl(const JsonPathType& path, bool value) NN_NOEXCEPT
{
    // eLicense が有効なときだけ解釈する
    if (m_ELicense)
    {
        if (path.Match(GetCurrentELicensePath("available_after_reboot")))
        {
            // 再起動後に有効な eLicense ではなく、かつ再起動時の読み込みである場合は eLicenseList への読み込みを行わない
            if (!value && m_ReadAfterReboot)
            {
                m_ELicense = util::nullopt;
            }
        }
        else if (path.Match(GetCurrentELicensePath("recommends_server_interaction")))
        {
            if (value)
            {
                m_ELicense->flags = m_ELicense->flags | ELicenseInfoForSystemFlags::RecommendsServerInteraction;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveAdaptor::UpdateImpl(const JsonPathType& path, int64_t value) NN_NOEXCEPT
{
    NN_RESULT_DO(UpdateImpl(path, static_cast<uint64_t>(value)));

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveAdaptor::UpdateImpl(const JsonPathType& path, uint64_t value) NN_NOEXCEPT
{
    // eLicense が有効なときだけ解釈する
    if (m_ELicense)
    {
        if (path.Match(GetCurrentELicensePath("ticket_id")))
        {
            m_ELicense->ticketId = value;
        }
        else if (path.Match(GetCurrentELicensePath("ticket_owner_va_id")))
        {
            m_ELicense->ticketOwnerVaId = static_cast<es::AccountId>(value);
        }
        else if (path.Match(GetCurrentELicensePath("expire_date")))
        {
            m_ELicense->expireDate.time.value = value;
        }
    }

    else if (path.Match("$.publish_date"))
    {
        m_PublishDate = value;
    }

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveAdaptor::NotifyObjectBeginImpl(const JsonPathType& path) NN_NOEXCEPT
{
    if (path.Match(GetNextELicensePath()))
    {
        m_ELicense.emplace();
        m_ELicense->ownerNaId = m_DeclaredELicenseOwnerId;
        //TODO: ExpireDateValueAsPermanent を使う
        m_ELicense->expireDate.time.value = std::numeric_limits<int64_t>::max();

        m_CurrentIndex++;
    }

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveAdaptor::NotifyObjectEndImpl(const JsonPathType& path) NN_NOEXCEPT
{
    if (m_ELicense)
    {
        if (path.Match(GetCurrentELicensePath()))
        {
            if (!m_pList->Insert(*m_ELicense))
            {
                NN_RESULT_THROW(ResultELicenseListFull());
            }
            m_ELicense = util::nullopt;
        }
    }

    NN_RESULT_SUCCESS;
}

const char* ELicenseArchiveAdaptor::GetNextELicensePath() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(nn::util::SNPrintf(m_Path, sizeof(m_Path), "$.elicenses[%d]", m_CurrentIndex + 1) < sizeof(m_Path));
    return m_Path;
}

const char* ELicenseArchiveAdaptor::GetCurrentELicensePath() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(nn::util::SNPrintf(m_Path, sizeof(m_Path), "$.elicenses[%d]", m_CurrentIndex) < sizeof(m_Path));
    return m_Path;
}

const char* ELicenseArchiveAdaptor::GetCurrentELicensePath(const char* key) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(nn::util::SNPrintf(m_Path, sizeof(m_Path), "$.elicenses[%d].%s", m_CurrentIndex, key) < sizeof(m_Path));
    return m_Path;
}

}} // namespace nn::es
