﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>
#include <limits>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkSystemEventApi.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/es/es_ActiveRightsContext.h>
#include <nn/es/es_IETicketService.sfdl.h>
#include <nn/es/es_Result.h>
#include "es_ETicketServiceObject.h"

namespace nn { namespace es {

namespace detail {

struct ActiveRightsContextHandleObject
{
    IActiveRightsContext* pAccessor;
};

namespace {

    os::SdkMutex g_Mutex;
    detail::ActiveRightsContextHandleObject g_Objects[ActiveRightsContextCountMax];

    ActiveRightsContextHandleObject* CreateObject(sf::SharedPointer<IActiveRightsContext> pAccessor) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        for (auto&&e : g_Objects)
        {
            if (!e.pAccessor)
            {
                e.pAccessor = pAccessor.Detach();
                return &e;
            }
        }
        NN_ABORT("too many applications");
    }

    void DeleteObject(ActiveRightsContextHandleObject* p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        auto&& e = *p;
        sf::ReleaseSharedObject(e.pAccessor);
        e.pAccessor = nullptr;
    }

}   // namespace
}   // namespace detail


ActiveRightsContextHandle CreateActiveRightsContext() NN_NOEXCEPT
{
    sf::SharedPointer<IActiveRightsContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->CreateActiveRightsContext(&p));

    ActiveRightsContextHandle handle = {};
    handle._p = detail::CreateObject(std::move(p));
    return handle;
}

void CloseActiveRightsContextHandle(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    auto p = handle._p;
    if (p)
    {
        detail::DeleteObject(p);
    }
}

Result SetUsersOfAccountRestrictedRights(ActiveRightsContextHandle handle, const account::NintendoAccountId naIdList[], int naIdCount, bool anyAccount) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    sf::InArray<es::ELicenseUserId> userIdList(reinterpret_cast<const es::ELicenseUserId*>(naIdList), naIdCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetUsersOfAccountRestrictedRights(userIdList, anyAccount));

    NN_RESULT_SUCCESS;
}

bool GetUsersOfAccountRestrictedRights(int* pOutLength, account::NintendoAccountId outNaIdList[], int listLength, ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(listLength > 0);
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    int count = 0;
    sf::OutArray<ELicenseUserId> list(reinterpret_cast<ELicenseUserId*>(outNaIdList), listLength);
    auto result = p->GetUsersOfAccountRestrictedRights(&count, list);
    if (result <= ResultNoAccountRestrictedRightsUsers())
    {
        // アカウント限定権利ではない場合
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // アカウント限定権利だった場合
    *pOutLength = count;
    return true;
}

Result RegisterRightsIdList(ActiveRightsContextHandle handle, const es::RightsId rightsIdList[], int listLength) NN_NOEXCEPT
{
    if (listLength == 0)
    {
        NN_RESULT_SUCCESS;
    }

    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    sf::InArray<es::RightsId> list(rightsIdList, listLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->RegisterRightsIdList(list));

    NN_RESULT_SUCCESS;
}

void CheckRightsIdListValidity(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    NN_ABORT_UNLESS_RESULT_SUCCESS(p->CheckRightsIdListValidity());
}

void RemoveUnavailableRightsIds(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    NN_ABORT_UNLESS_RESULT_SUCCESS(p->RemoveUnavailableRightsIds());
}

Result BeginUsingActiveRightsContext(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    NN_ABORT_UNLESS_RESULT_SUCCESS(p->BeginUsingActiveRightsContext());
    NN_RESULT_SUCCESS;
}

void EndUsingActiveRightsContext(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    NN_ABORT_UNLESS_RESULT_SUCCESS(p->EndUsingActiveRightsContext());
}

Result ForceActivateRightsContextForExit(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    NN_ABORT_UNLESS_RESULT_SUCCESS(p->ForceActivateRightsContextForExit());
    NN_RESULT_SUCCESS;
}

int ListUsingRightsIds(RightsId outList[], int listLength, ActiveRightsContextHandle handle, bool isTemporaryOnly) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(listLength > 0);
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    int count = 0;
    sf::OutArray<RightsId> list(outList, listLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->ListUsingRightsIds(&count, list, isTemporaryOnly));
    return count;
}

int ListUsingELicenseIds(ELicenseId outList[], int listLength, ActiveRightsContextHandle handle, bool isTemporaryOnly, account::NintendoAccountId ownerNaId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(listLength > 0);
    NN_SDK_REQUIRES(ownerNaId != account::InvalidNintendoAccountId);

    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    int count = 0;
    sf::OutArray<ELicenseId> list(outList, listLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->ListUsingELicenseIds(&count, list, isTemporaryOnly, GetELicenseOwnerId(ownerNaId)));
    return count;
}

int ListUsingELicenseOwnerIds(account::NintendoAccountId outNaIdList[], int listLength, ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(listLength > 0);
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    int count = 0;
    sf::OutArray<ELicenseOwnerId> list(reinterpret_cast<ELicenseOwnerId*>(outNaIdList), listLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->ListUsingELicenseOwnerIds(&count, list));
    return count;
}

ActiveRightsContextStatus GetActiveRightsContextStatus(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    int status = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetActiveRightsContextStatus(&status));
    return static_cast<ActiveRightsContextStatus>(status);
}

os::Tick GetActiveRightsContextExpiredTime(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    int64_t expiredTime = std::numeric_limits<int64_t>::max();
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetActiveRightsContextExpiredTime(&expiredTime));
    return os::Tick(expiredTime);
}

sf::NativeHandle GetActiveRightsContextExpiredTimeChangedEvent(ActiveRightsContextHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle._p);
    auto p = handle._p->pAccessor;

    sf::NativeHandle sfHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetActiveRightsContextExpiredTimeChangedEvent(&sfHandle));
    return sfHandle;
}


}} // namespace nn::es
