﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdint>
#include <limits>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/es/es_Result.h>
#include <nn/es/es_Configuration.h>
#include <nn/es/es_ServiceLog.h>
#include <nn/es/es_RightsTypes.h>
#include <nn/es/es_ActiveRightsContextTypes.h>
#include <nn/es/es_TypesForInner.h>

#include <nn/time/time_PosixTime.h>
#include <nn/util/util_ScopedTransaction.h>
#include "es_ETicketServiceImpl.h"
#include "es_Clock.h"
#include "es_RightsAvailabilityMaster.h"
#include "es_ActiveRightsContextManager.h"

namespace nn { namespace es {

namespace {

    // 各権利コンテキストが使用する権利リスト
    // 権利コンテキストの数よりも少な目に静的に確保
    os::SdkMutex g_MutexUsingRightsArray;
    UsingRightsArray g_UsingRightsArray[UsingRightsArrayCountMax];
    bool g_IsUsingRightsArrayUsed[UsingRightsArrayCountMax] = {};


    UsingRightsArray* AllocateUsingRightsArray() NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MutexUsingRightsArray)> lk(g_MutexUsingRightsArray);
        for (int i=0; i<UsingRightsArrayCountMax; ++i)
        {
            if (!g_IsUsingRightsArrayUsed[i])
            {
                g_IsUsingRightsArrayUsed[i] = true;
                g_UsingRightsArray[i].clear();
                return &g_UsingRightsArray[i];
            }
        }
        NN_ABORT();
    }

    void FreeUsingRightsArray(UsingRightsArray* p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MutexUsingRightsArray)> lk(g_MutexUsingRightsArray);

        auto index = p - g_UsingRightsArray;
        NN_SDK_ASSERT(0 <= index && index < UsingRightsArrayCountMax);

        NN_SDK_ASSERT(g_IsUsingRightsArrayUsed[index]);
        g_IsUsingRightsArrayUsed[index] = false;
    }

}   // namespace

//----------------------------------------------------------------------------

ActiveRightsContext::ActiveRightsContext(ETicketServiceImpl* pETicketServiceImpl) NN_NOEXCEPT
    : m_pETicketServiceImpl(pETicketServiceImpl)
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext: ctor=0x%p\n", this);

    // TimeSpan に変換可能な Tick の最大値を計算しておく（= 292 年）
    this->m_ExpiredTickMax = os::ConvertToTick( TimeSpan::FromNanoSeconds( std::numeric_limits<int64_t>::max() ) );

    // 初期化
    for (int i=0; i<ELicenseUserCountMax; ++i)
    {
        this->m_UserIdList[i] = InvalidELicenseUserId;
    }

    this->m_ELicenseUserCount = 0;
    this->m_IsAnyAccount = true;
    this->m_IsFirstActivationDone = false;
    this->m_IsActive = false;
    this->m_IsForceActivatedForExit = false;
    m_pETicketServiceImpl->UnregisterAllAccountRestrictedRightsUser();

    this->m_pUsingRightsArray = nullptr;
    this->m_ExpiredTick = GetExpiredTickMax();

    // TORIAEZU:
    //  互換のため、デフォルトを true にしておく。
    //  RegisterRightsIdList() を呼ぶと nullopt（未チェック状態）に変わる。
    this->m_pIsAllRightsIdsCheckedAndValid = true;
}

ActiveRightsContext::~ActiveRightsContext() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext: dtor=0x%p\n", this);
    if (m_pUsingRightsArray)
    {
        FreeUsingRightsArray(m_pUsingRightsArray);
    }
}



Result ActiveRightsContext::SetUsersOfAccountRestrictedRights(sf::InArray<es::ELicenseUserId> userIdList, bool anyAccount) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::SetRestrictedAccountList() anyAccount=%s\n", anyAccount ? "true" : "false");

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    // TORIAEZU:
    //  ETicketServiceImpl が持つ UserList にアカウントリストを登録し、
    //  ETicketServiceImpl::RegisterTitleKey() を利用できるようにしておく。
    NN_RESULT_DO(m_pETicketServiceImpl->UnregisterAllAccountRestrictedRightsUser());
    if (anyAccount)
    {
        // デバイス共有権利を利用時（便宜上、0 にしておく）
        this->m_ELicenseUserCount = 0;
    }
    else
    {
        // アカウント限定権利の利用者を登録
        auto userCount = userIdList.GetLength();
        NN_SDK_REQUIRES(userCount <= ELicenseUserCountMax);
        for (size_t i=0; i<userCount; ++i)
        {
            this->m_UserIdList[i] = userIdList[i];
            NN_RESULT_DO(m_pETicketServiceImpl->RegisterAccountRestrictedRightsUser(userIdList[i]));
        }
        this->m_ELicenseUserCount = static_cast<uint8_t>(userCount);
    }

    this->m_IsAnyAccount = anyAccount;
    this->m_IsActive = false;
    this->m_IsForceActivatedForExit = false;
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::GetUsersOfAccountRestrictedRights(sf::Out<int> pOutCount, sf::OutArray<ELicenseUserId> outList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::GetRestrictedAccountList()\n");
    auto leftCount = outList.GetLength();
    NN_RESULT_THROW_UNLESS(leftCount > 0, ResultInvalidParameter());

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_RESULT_THROW_UNLESS(!m_IsAnyAccount, ResultNoAccountRestrictedRightsUsers());

    auto* p = outList.GetData();
    for (auto userId : m_UserIdList)
    {
        *p++ = userId;
        --leftCount;
        if (leftCount == 0)
        {
            break;
        }
    }

    *pOutCount = m_ELicenseUserCount;
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::RegisterRightsIdList(sf::InArray<es::RightsId> rightsIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::RegisterRightsIdList() length=%zu rightsId[0]=0x%016llx\n", rightsIdList.GetLength(), rightsIdList[0]);

    auto* pRightIdList = rightsIdList.GetData();
    auto  length       = rightsIdList.GetLength();

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    // 一旦未チェック状態に変更
    this->m_pIsAllRightsIdsCheckedAndValid = util::nullopt;

    // 権利リスト用の領域を確保
    if (!m_pUsingRightsArray)
    {
        this->m_pUsingRightsArray = AllocateUsingRightsArray();
    }

    // 重複を確認しながら rightsIdList を既存の m_pUsingRightsArray に追加登録
    for (const auto& itrRightsId : util::MakeSpan(pRightIdList, length))
    {
        auto end = m_pUsingRightsArray->end();
        auto itr = std::find_if(
                            m_pUsingRightsArray->begin(),
                            end,
                            [&](const UsingRights& usingRights) NN_NOEXCEPT
                            {
                                return usingRights.rightsId == itrRightsId;
                            }
                        );
        if (itr < end)
        {
            // 重複を発見した場合は登録せずにスキップ
            continue;
        }

        NN_RESULT_THROW_UNLESS(m_pUsingRightsArray->size() < m_pUsingRightsArray->max_size(), ResultUsingRightsListFull());

        UsingRights e = {};
        e.rightsId = itrRightsId;
        e.state = UsingRightsState::NotChecked;
        m_pUsingRightsArray->push_back(e);
    }

    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::CheckRightsIdListValidity() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::CheckRightsIdListValidity()\n");

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    // 権利リストの個々の RightsId に対応する eLicense をチェックする
    auto* pRightIdList = m_pUsingRightsArray ? m_pUsingRightsArray->data() : nullptr;
    auto  length       = m_pUsingRightsArray ? m_pUsingRightsArray->size() : 0;

    bool isValid = true;
    util::optional<time::PosixTime> pExpireDate = util::nullopt;

    for (auto& itrRightsId : util::MakeSpan(pRightIdList, length))
    {
        RightsStatus status;

        // 対応する eLicense があるかをチェック（失敗しない前提）
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetRightsAvailabilityMaster().CheckRightsStatus(
                &status, 1,
                &itrRightsId.rightsId, 1,
                m_UserIdList, m_ELicenseUserCount,
                m_IsAnyAccount
            )
        );

        const int userIndex = 0;
        bool isPermanent = status.rights.IsPermanentRights();
        switch(status.status)
        {
            // 利用可能
            case RightsStatus::Status::Available:
            case RightsStatus::Status::AvailableSinceAccountRestricted:
            {
                itrRightsId.state = UsingRightsState::Valid;
                itrRightsId.eLicenseId[userIndex] = status.rights.eLicenseId;
                itrRightsId.SetPermanent(userIndex, isPermanent);
                break;
            }
            // 利用不可
            case RightsStatus::Status::HasAccountRestrictedRights:
            case RightsStatus::Status::NotAvailable:
            {
                itrRightsId.state = UsingRightsState::NoELicense;
                itrRightsId.eLicenseId[userIndex] = InvalidELicenseId;
                itrRightsId.SetPermanent(userIndex, false);
                isValid = false;
                continue;
            }
            // 想定外
            default: NN_UNEXPECTED_DEFAULT;
        }

        // 見つかった eLicense に有効期限がある場合には最も近い時刻を算出
        if (!isPermanent)
        {
            if (!pExpireDate || status.rights.expireDate.time < *pExpireDate)
            {
                pExpireDate = status.rights.expireDate.time;
            }
        }
    }
    // 権利コンテキスト全体での状態を設定
    this->m_pIsAllRightsIdsCheckedAndValid = isValid;

    // 権利コンテキスト全体での有効期限の算出
    if (!pExpireDate)
    {
        // 無期限
        this->m_ExpiredTick = GetExpiredTickMax();
    }
    else
    {
        // 有効期限あり
        time::PosixTime currentTime;
        if (Clock::GetCurrentTime( &currentTime ).IsSuccess())
        {
            auto diff = os::ConvertToTick(*pExpireDate - currentTime);
            this->m_ExpiredTick = os::GetSystemTick() + diff;
        }
        else
        {
            // TORIAEZU:
            //  現在時刻を取得できない場合は、実質的に Expired 扱いとしておく
            this->m_ExpiredTick = os::GetSystemTick();
        }
    }

    this->m_StateChangedEvent.Signal();
    NN_RESULT_SUCCESS;
}   // NOLINT(impl/function_size)

Result ActiveRightsContext::RemoveUnavailableRightsIds() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::RemoveUnavailableRightsIds()\n");

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_RESULT_THROW_UNLESS(!m_IsFirstActivationDone, ResultInvalidCall());

    if (!m_pUsingRightsArray)
    {
        this->m_pIsAllRightsIdsCheckedAndValid = true;
        NN_RESULT_SUCCESS;
    }

    // 権利リストの個々の RightsId に対応する eLicense をチェックする
    for (auto p = m_pUsingRightsArray->begin(); p != m_pUsingRightsArray->end(); )
    {
        switch(p->state)
        {
            case UsingRightsState::Valid:
            {
                // 何もせずに次へ
                break;
            }
            case UsingRightsState::NoELicense:
            {
                // 対応する eLicense がないものは削除して次へ
                p = m_pUsingRightsArray->erase(p);
                continue;
            }
            case UsingRightsState::NotChecked:
            {
                // 未チェックの RightsId が残っている場合は、
                // 事前に CheckRightsIdListValidity() が呼ばれていない。
                // shim 側でアボートするようにしておく。
                NN_RESULT_THROW( ResultInvalidCall() );
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
        ++p;
    }
    // 権利コンテキスト全体での状態を設定
    this->m_pIsAllRightsIdsCheckedAndValid = true;

    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::ListUsingRightsIds(sf::Out<int> pOutCount, sf::OutArray<RightsId> outList, bool isTemporaryOnly) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::ListUsingRightsIds() UsingRightsArray.size(): %zu\n", m_pUsingRightsArray->size());

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    if (!m_pUsingRightsArray)
    {
        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }

    // とりあえず RegisterRightsIdList() で登録したものを返すだけ
    auto* pOut = outList.GetData();
    size_t count = 0;
    for (const auto& itrRights : util::MakeSpan(m_pUsingRightsArray->begin(), m_pUsingRightsArray->end()))
    {
        if (isTemporaryOnly)
        {
            // 有効な eLicense があり、かつ、無期限のものはスキップする
            if (itrRights.state == UsingRightsState::Valid && itrRights.IsPermanentAll())
            {
                continue;
            }
        }
        pOut[count] = itrRights.rightsId;
        ++count;
        if (count >= outList.GetLength())
        {
            break;
        }
    }
    *pOutCount = static_cast<int>(count);
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::ListUsingELicenseIds(sf::Out<int> pOutCount, sf::OutArray<ELicenseId> outList, bool isTemporaryOnly, ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::ListUsingELicenseIds() isTemporaryOnly=%s  ownerId=0x%016llx\n", isTemporaryOnly ? "true" : "false", ownerId.id);

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_RESULT_THROW_UNLESS(ActiveRightsContextStatus_NotChecked != GetActiveRightsContextStatusImpl(), ResultInvalidCall());
    NN_RESULT_THROW_UNLESS(ownerId != InvalidELicenseOwnerId, ResultInvalidParameter());

    if (!m_pUsingRightsArray)
    {
        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }

    auto* pOut = outList.GetData();
    auto lengthMax = static_cast<int>(outList.GetLength());
    NN_RESULT_THROW_UNLESS(lengthMax > 0, ResultInvalidCall());

    int count = 0;
    for (auto&& itrRights : util::MakeSpan(m_pUsingRightsArray->begin(), m_pUsingRightsArray->end()))
    {
        if (itrRights.state != UsingRightsState::Valid)
        {
            continue;
        }

        //  最大 8 利用者分の eLicenseId を uniq して返す。
        fixed_size_vector<ELicenseId, ELicenseUserCountMax> tmpList;
        int userIndexMax = m_IsAnyAccount ? 1 : m_ELicenseUserCount;
        for (int userIndex=0; userIndex<userIndexMax; ++userIndex)
        {
            auto& eLicenseId = itrRights.eLicenseId[userIndex];
            if (eLicenseId == InvalidELicenseId)
            {
                continue;
            }
            if (isTemporaryOnly && itrRights.IsPermanent(userIndex))
            {
                continue;
            }

            //  eLicenseId の重複チェック
            if (std::find(tmpList.begin(), tmpList.end(), eLicenseId) != tmpList.end())
            {
                continue;
            }

            // TORIAEZU:
            //  eLicense の ownerId が、指定された ownerId と一致するかを
            //  チェックする。一旦速度を気にせずに実装しておく。
            {
                RightsStatus status = {};

                // 対応する eLicense があるかをチェック（失敗しない前提）
                if (!GetRightsAvailabilityMaster().CheckRightsStatus(
                        &status, 1,
                        &itrRights.rightsId, 1,
                        m_UserIdList, m_ELicenseUserCount,
                        m_IsAnyAccount
                ).IsSuccess())
                {
                    continue;
                }

                if (GetELicenseOwnerId(status) != ownerId)
                {
                    continue;
                }
            }

            // 重複チェック用のリストに登録
            tmpList.push_back(eLicenseId);

            // eLicenseId を登録
            pOut[count] = eLicenseId;
            ++count;

            if (count >= lengthMax)
            {
                *pOutCount = count;
                NN_RESULT_SUCCESS;
            }
        }
    }
    *pOutCount = count;
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::ListUsingELicenseOwnerIds(sf::Out<int> pOutCount, sf::OutArray<ELicenseOwnerId> outList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::ListUsingELicenseOwnerIds()\n");

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_RESULT_THROW_UNLESS(ActiveRightsContextStatus_NotChecked != GetActiveRightsContextStatusImpl(), ResultInvalidCall());

    if (!m_pUsingRightsArray)
    {
        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }

    auto* pOut = outList.GetData();
    auto lengthMax = static_cast<int>(outList.GetLength());
    NN_RESULT_THROW_UNLESS(lengthMax > 0, ResultInvalidCall());

    int count = 0;
    for (auto&& itrRights : util::MakeSpan(m_pUsingRightsArray->begin(), m_pUsingRightsArray->end()))
    {
        if (itrRights.state != UsingRightsState::Valid)
        {
            continue;
        }

        //  最大 8 オーナー分の ELicenseOwnerId を uniq して返す。
        int userIndexMax = m_IsAnyAccount ? 1 : m_ELicenseUserCount;
        for (int userIndex=0; userIndex<userIndexMax; ++userIndex)
        {
            auto& eLicenseId = itrRights.eLicenseId[userIndex];
            if (eLicenseId == InvalidELicenseId)
            {
                continue;
            }

            // TORIAEZU:
            //  eLicense の ownerId を取得する。
            //  一旦速度を気にせずに実装しておく。
            RightsStatus status = {};

            // 対応する eLicense があるかをチェック（失敗しない前提）
            if (!GetRightsAvailabilityMaster().CheckRightsStatus(
                    &status, 1,
                    &itrRights.rightsId, 1,
                    m_UserIdList, m_ELicenseUserCount,
                    m_IsAnyAccount
            ).IsSuccess())
            {
                continue;
            }

            //  ownerId の重複チェック
            auto itrBegin = outList.GetData();
            auto itrEnd   = itrBegin + count;
            auto ownerId  = GetELicenseOwnerId(status);
            if (std::find(itrBegin, itrEnd, ownerId) != itrEnd)
            {
                continue;
            }

            // eLicenseId を登録
            pOut[count] = ownerId;
            ++count;

            if (count >= lengthMax)
            {
                *pOutCount = count;
                NN_RESULT_SUCCESS;
            }
        }
    }
    *pOutCount = count;
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::BeginUsingActiveRightsContext() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::BeginUsingActiveRightsContext()\n");

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (m_IsActive || m_IsForceActivatedForExit)
    {
        NN_RESULT_SUCCESS;
    }

    if (!m_IsFirstActivationDone)
    {
        // TORIAEZU:
        //  とりあえず状態変更のみ
        this->m_IsFirstActivationDone = true;
    }

    this->m_IsActive = true;
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::EndUsingActiveRightsContext() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::EndUsingActiveRightsContext()\n");

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (m_IsForceActivatedForExit)
    {
        // 既にアプリ終了用の強制実行中であれば何もしない
        NN_RESULT_SUCCESS;
    }

    // TORIAEZU:
    //  とりあえず状態変更のみ
    this->m_IsActive = false;
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::ForceActivateRightsContextForExit() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::ForceActivateRightsContextForExit()\n");

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    if (!m_IsForceActivatedForExit)
    {
        GetRightsAvailabilityMaster().SetAlreadyRegisteredTitleKeys(true);
        this->m_IsActive = true;
        this->m_IsForceActivatedForExit = true;
    }
    NN_RESULT_SUCCESS;
}

int ActiveRightsContext::GetActiveRightsContextStatusImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    // 権利コンテキスト全体の状態の更新
    if (!m_pIsAllRightsIdsCheckedAndValid)
    {
        return ActiveRightsContextStatus_NotChecked;
    }
    else if (!*m_pIsAllRightsIdsCheckedAndValid)
    {
        return ActiveRightsContextStatus_Inactive;
    }
    else if (m_ExpiredTick < os::GetSystemTick())
    {
        return ActiveRightsContextStatus_Expired;
    }
    return ActiveRightsContextStatus_Active;
}

Result ActiveRightsContext::GetActiveRightsContextStatus(sf::Out<int> pOutStatus) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    auto contextStatus = GetActiveRightsContextStatusImpl();
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::GetActiveRightsContextStatus() status=%d\n", contextStatus);
    *pOutStatus = contextStatus;
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::GetActiveRightsContextExpiredTime(sf::Out<int64_t> pOutExpiredTime) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::GetActiveRightsContextExpiredTime() leftTime=%lld(sec)\n", (m_ExpiredTick - os::GetSystemTick()).ToTimeSpan().GetSeconds());

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    NN_RESULT_THROW_UNLESS(ActiveRightsContextStatus_NotChecked != GetActiveRightsContextStatusImpl(), ResultInvalidCall());
    *pOutExpiredTime = m_ExpiredTick.GetInt64Value();
    NN_RESULT_SUCCESS;
}

Result ActiveRightsContext::GetActiveRightsContextExpiredTimeChangedEvent(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ActiveRightsContext::GetActiveRightsContextExpiredTimeChangedEvent()\n");

    *pOut = sf::NativeHandle(m_StateChangedEvent.GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}

bool ActiveRightsContext::IsValidRightsIdListed(const RightsId& rightsId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);

    if (!m_pUsingRightsArray)
    {
        return false;
    }

    return std::find_if(
        m_pUsingRightsArray->begin(),
        m_pUsingRightsArray->end(),
        [&](const UsingRights& usingRights) NN_NOEXCEPT
        {
            return true
                && usingRights.state == UsingRightsState::Valid
                && usingRights.rightsId == rightsId;
        }
    ) != m_pUsingRightsArray->end();
}


}} // namespace nn::es
