﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstdio>

#include <nn/nn_SdkAssert.h>
#include <nn/err/err_Result.h>
#include <nn/err/err_ResultPrivate.h>
#include <nn/util/util_FormatString.h>

#include "nn/err/err_ServiceStatusAccessorApi.h"

#define RAPIDJSON_NO_INT64DEFINE
#define RAPIDJSON_NAMESPACE             nne::rapidjson
#define RAPIDJSON_NAMESPACE_BEGIN       namespace nne { namespace rapidjson {
#define RAPIDJSON_NAMESPACE_END         }}
#define RAPIDJSON_ASSERT(x)             NN_SDK_ASSERT(x)
#define RAPIDJSON_HAS_CXX11_RVALUE_REFS 1 // NOLINT(readability/define)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 // NOLINT(readability/define)
// Windows 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif
#include <rapidjson/rapidjson.h>
#include <rapidjson/document.h>
#include <rapidjson/stringbuffer.h>
#include <rapidjson/writer.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif
namespace  {
    // CalendarTimeをYYYYMMDDHHMMSS(10進数)の形式に変換
    #define CONVERT_CALENDARTIME_TO_BIT64(year, month,day, hour, minute, second) \
    ( \
        ( static_cast<nn::Bit64>(year)   * ( 10000000000 ) ) + \
        ( static_cast<nn::Bit64>(month)  * ( 100000000 ) ) + \
        ( static_cast<nn::Bit64>(day)    * ( 1000000 ) ) + \
        ( static_cast<nn::Bit64>(hour)   * ( 10000 ) ) + \
        ( static_cast<nn::Bit64>(minute) * ( 100 ) ) + \
        ( static_cast<nn::Bit64>(second) )\
    )

    bool GreaterThanCalendarTime(nn::time::CalendarTime calendarTime1, nn::time::CalendarTime calendarTime2) NN_NOEXCEPT
    {
        // CalendarTimeをYYYYMMDDHHMMSSの形式に変換して比較
        nn::Bit64 time1 = CONVERT_CALENDARTIME_TO_BIT64(calendarTime1.year,
                                                    calendarTime1.month,
                                                    calendarTime1.day,
                                                    calendarTime1.hour,
                                                    calendarTime1.minute,
                                                    calendarTime1.second);
        nn::Bit64 time2 = CONVERT_CALENDARTIME_TO_BIT64(calendarTime2.year,
                                                    calendarTime2.month,
                                                    calendarTime2.day,
                                                    calendarTime2.hour,
                                                    calendarTime2.minute,
                                                    calendarTime2.second);

        return (time1 > time2);
    }

    nn::time::CalendarTime ToCalendarTimeForString(const char* pText) NN_NOEXCEPT
    {
        nn::time::CalendarTime utcTime = {};
        int year = 0;
        int month = 0;
        int day = 0;
        int hour = 0;
        int minute = 0;

        std::sscanf(pText, "%04d/%02d/%02d %02d:%02d",
            &year, &month, &day,
            &hour, &minute);

        NN_SDK_REQUIRES_MINMAX(year, 0, 9999);
        NN_SDK_REQUIRES_MINMAX(month, 0, 12); //暦上は0は存在しないがjsonデータとして0はありうる
        NN_SDK_REQUIRES_MINMAX(day, 0, 31);   //同上
        NN_SDK_REQUIRES_MINMAX(hour, 0, 23);
        NN_SDK_REQUIRES_MINMAX(minute, 0, 59);

        utcTime.year = static_cast<int16_t>(year);
        utcTime.month = static_cast<int8_t>(month);
        utcTime.day = static_cast<int8_t>(day);
        utcTime.hour = static_cast<int8_t>(hour);
        utcTime.minute = static_cast<int8_t>(minute);

        return utcTime;
    }

    bool IsMatchString( const char* pExpected, const char* pString, const size_t length ) NN_NOEXCEPT
    {
        if( std::strlen( pExpected ) == length
            && std::strncmp( pExpected, pString, length) == 0 )
        {
            return true;
        }

        return false;
    }
}

namespace nn { namespace err {
    void ServiceStatusAccessor::Initialize(void* pData, size_t size) NN_NOEXCEPT
    {
        ServiceStatusAccessor::m_Data = pData;
        ServiceStatusAccessor::m_Size = size;
    }
    nn::Result ServiceStatusAccessor::GetServiceStatus(ServiceStatus* pServiceStatus,
                                                       const ApplicationId& applicationId,
                                                       const char* pServerCode) const NN_NOEXCEPT
    {

        // パース
        nne::rapidjson::Document document;
        document.ParseInsitu((static_cast<char*>(m_Data)));
        if(0 != document.HasParseError())
        {
            // jsonデータの解析に失敗
            return nn::err::ResultServiceStatusParseError();
        }
        NN_SDK_ASSERT(document.IsObject());

        // アプリケーションID
        char appId[16 + 1] = {};
        util::SNPrintf(appId, sizeof(appId), "%016llx", applicationId.value);

        // 要素の抽出
        auto& maintenances = document["maintenances"];
        uint32_t maintenancesCount = maintenances.Size();

        // 初期値作成
        const nn::time::CalendarTime ZeroTime = {};
        pServiceStatus->hasMaintenance = false;
        pServiceStatus->startMaintenanceDate = ZeroTime;
        pServiceStatus->endMaintenanceDate = ZeroTime;
        pServiceStatus->hasIncident = false;
        pServiceStatus->startIncidentDate = ZeroTime;
        pServiceStatus->endIncidentDate = ZeroTime;

        for (uint32_t maintenanceIndex = 0; maintenanceIndex < maintenancesCount; ++maintenanceIndex)
        {
            auto& maintenance = maintenances[maintenanceIndex];
            auto& service = maintenance["service"];
            const char* pJsonServerCode = service["server_code"].GetString();
            const size_t jsonServerCodeLength = service["server_code"].GetStringLength();

            auto& titleList = maintenance["titles"];
            uint32_t titleCount = maintenance["titles"].Size();

            for (uint32_t titleIndex = 0; titleIndex < titleCount; ++titleIndex)
            {
                auto& title = titleList[titleIndex];
                const char* pJsonAppId = title["application_id"].GetString();
                const size_t jsonAppIdLength = title["application_id"].GetStringLength();

                if ( ( IsMatchString( "Any" , pJsonAppId, jsonAppIdLength)
                    || IsMatchString( appId, pJsonAppId, jsonAppIdLength ))
                    && IsMatchString( pServerCode, pJsonServerCode, jsonServerCodeLength ))
                {
                    pServiceStatus->hasMaintenance = true;
                    // 終了時刻が一番未来のものを設定する
                    if(GreaterThanCalendarTime(ToCalendarTimeForString(maintenance["end_date"].GetString()),
                                               pServiceStatus->endMaintenanceDate))
                    {
                        pServiceStatus->startMaintenanceDate = ToCalendarTimeForString(maintenance["start_date"].GetString());
                        pServiceStatus->endMaintenanceDate = ToCalendarTimeForString(maintenance["end_date"].GetString());
                    }
                }
            }
        }

        nn::time::CalendarTime startIncidentDateMostNew = ZeroTime;
        NN_SDK_ASSERT(document.HasMember("incidents"));
        auto& incidents = document["incidents"];
        NN_SDK_ASSERT(incidents.IsArray());
        uint32_t incidentsCount = incidents.Size();
        for (uint32_t incidentIndex = 0; incidentIndex < incidentsCount; ++incidentIndex)
        {
            auto& incident = incidents[incidentIndex];
            auto& service = incident["service"];
            const char* pJsonServerCode = service["server_code"].GetString();
            const size_t jsonServerCodeLength = service["server_code"].GetStringLength();

            auto& titleList = incident["titles"];
            uint32_t titleCount = incident["titles"].Size();

            for (uint32_t titleIndex = 0; titleIndex < titleCount; ++titleIndex)
            {
                auto& title = titleList[titleIndex];
                const char* pJsonAppId = title["application_id"].GetString();
                const size_t jsonAppIdLength = title["application_id"].GetStringLength();

                if ( ( IsMatchString( "Any" , pJsonAppId, jsonAppIdLength)
                    || IsMatchString( appId, pJsonAppId, jsonAppIdLength ))
                    && IsMatchString( pServerCode, pJsonServerCode, jsonServerCodeLength ))
                {
                    pServiceStatus->hasIncident = true;
                    // 障害復旧時刻が一番未来のものを設定する
                    if(GreaterThanCalendarTime(ToCalendarTimeForString(incident["restoration_date"].GetString()),
                                               pServiceStatus->endIncidentDate))
                    {
                        pServiceStatus->startIncidentDate = ToCalendarTimeForString(incident["occurrence_date"].GetString());
                        pServiceStatus->endIncidentDate = ToCalendarTimeForString(incident["restoration_date"].GetString());
                    }

                    // 障害復旧時刻がない場合の為に、障害発生時刻が一番未来のものを保存しておく
                    if(GreaterThanCalendarTime(ToCalendarTimeForString(incident["occurrence_date"].GetString()),
                                               startIncidentDateMostNew))
                    {
                        startIncidentDateMostNew = ToCalendarTimeForString(incident["occurrence_date"].GetString());
                    }
                }
            }
        }

        // 障害情報があるが、障害復旧時刻が0000/00/00 00:00の場合は、障害発生時刻の一番新しいものを設定する
        if(pServiceStatus->hasIncident == true && pServiceStatus->endIncidentDate == ZeroTime)
        {
            pServiceStatus->startIncidentDate = startIncidentDateMostNew;
        }

        return nn::ResultSuccess();
    }
}}
