﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_SelectorForSystemServices.h>
#include <nn/account/account_Types.h>
#include <nn/applet/applet.h>
#include <nn/err/err_Api.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/err_LibraryApi.h>
#include <nn/err/err_Result.h>
#include <nn/err/err_ShowApplicationErrorApi.h>
#include <nn/err/err_ShowErrorApi.h>
#include <nn/err/err_ShowNetworkServiceErrorApi.h>
#include <nn/err/err_ShowUnacceptableAddOnContentVersionErrorApi.h>
#include <nn/err/err_ShowUnacceptableApplicationVersionErrorApi.h>
#include <nn/err/err_SystemErrorArg.h>
#include <nn/err/detail/err_Log.h>
#include <nn/err/detail/err_SystemData.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/la/la_Api.h>
#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/pctl/pctl_ResultSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "detail/err_StringUtil.h"

namespace nn { namespace err {

    namespace
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        applet::LibraryAppletHandle PrepareCommon(const void* param, size_t paramSize) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(param);
            NN_SDK_REQUIRES_GREATER(paramSize, 0u);

            applet::LibraryAppletHandle libraryAppletHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(applet::CreateLibraryApplet(&libraryAppletHandle, applet::AppletId_LibraryAppletError, applet::LibraryAppletMode_AllForeground));
            la::CommonArgumentsWriter writer(0);
            writer.PushToInChannel(libraryAppletHandle);

            applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(applet::CreateStorage(&storageHandle, paramSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(applet::WriteToStorage(storageHandle, 0, param, paramSize));
            applet::PushToInChannel(libraryAppletHandle, storageHandle);

            return libraryAppletHandle;
        }

        ErrorViewerJumpDestination Invoke(applet::LibraryAppletHandle libraryAppletHandle) NN_NOEXCEPT
        {
            la::LibraryAppletStartHookUserArg startHookUserArg;
            startHookUserArg.isExtremity = true;
            NN_ABORT_UNLESS_RESULT_SUCCESS(applet::StartLibraryApplet(libraryAppletHandle, &startHookUserArg));
            applet::JoinLibraryApplet(libraryAppletHandle);
            auto exitReason = applet::GetLibraryAppletExitReason(libraryAppletHandle);
            NN_ABORT_UNLESS_NOT_EQUAL(exitReason, applet::LibraryAppletExitReason_Abnormal);
            NN_ABORT_UNLESS_NOT_EQUAL(exitReason, applet::LibraryAppletExitReason_Unexpected);
            switch( exitReason )
            {
            case applet::LibraryAppletExitReason_Normal:
                {
                    applet::StorageHandle storageHandle;
                    if( applet::TryPopFromOutChannel(&storageHandle, libraryAppletHandle) )
                    {
                        ErrorViewerReturnValue returnValue{};
                        applet::ReadFromStorage(storageHandle, 0, &returnValue, sizeof(returnValue));
                        nn::applet::ReleaseStorage(storageHandle);
                        return returnValue.destination;
                    }
                    else
                    {
                        return ErrorViewerJumpDestination::Nowhere;
                    }
                }
            case applet::LibraryAppletExitReason_Canceled:
                {
                    NN_DETAIL_ERR_TRACE("ErrorViewerApplet is canceled.\n");
                    return ErrorViewerJumpDestination::Nowhere;
                }
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
#endif

        ErrorViewerJumpDestination InvokeErrorViewerApplet(
            const void* param, size_t paramSize) NN_NOEXCEPT
        {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            auto libraryAppletHandle = PrepareCommon(param, paramSize);
            NN_UTIL_SCOPE_EXIT{ applet::CloseLibraryApplet(libraryAppletHandle); };
            return Invoke(libraryAppletHandle);
#else
            NN_UNUSED(param);
            NN_UNUSED(paramSize);
            return ErrorViewerJumpDestination::Nowhere;
#endif
        }

        ErrorViewerJumpDestination InvokeErrorViewerApplet(
            const void* param, size_t paramSize,
            const void* secondParam, size_t secondParamSize) NN_NOEXCEPT
        {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            auto libraryAppletHandle = PrepareCommon(param, paramSize);
            NN_UTIL_SCOPE_EXIT{ applet::CloseLibraryApplet(libraryAppletHandle); };

            applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(applet::CreateStorage(&storageHandle, secondParamSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(applet::WriteToStorage(storageHandle, 0, secondParam, secondParamSize));
            applet::PushToInChannel(libraryAppletHandle, storageHandle);

            return Invoke(libraryAppletHandle);
#else
            NN_UNUSED(param);
            NN_UNUSED(paramSize);
            NN_UNUSED(secondParam);
            NN_UNUSED(secondParamSize);
            return ErrorViewerJumpDestination::Nowhere;
#endif
        }

        ErrorViewerJumpDestination InvokeErrorViewerAppletWithLargeStorage(
            const void* param, size_t paramSize,
            void* secondParam, size_t secondParamSize) NN_NOEXCEPT
        {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            auto libraryAppletHandle = PrepareCommon(param, paramSize);
            NN_UTIL_SCOPE_EXIT{ applet::CloseLibraryApplet(libraryAppletHandle); };

            applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(applet::CreateLargeStorage(&storageHandle, secondParam, secondParamSize, false));
            nn::applet::PushToInChannel(libraryAppletHandle, storageHandle);

            return Invoke(libraryAppletHandle);
#else
            NN_UNUSED(param);
            NN_UNUSED(paramSize);
            NN_UNUSED(secondParam);
            NN_UNUSED(secondParamSize);
            return ErrorViewerJumpDestination::Nowhere;
#endif
        }
    }

//--- LibraryApi ---//

bool ErrorCode::IsValid() const NN_NOEXCEPT
{
    return (category > 0);
}

ErrorCode ErrorCode::GetInvalidErrorCode() NN_NOEXCEPT
{
    return ErrorCode{ 0, 0 };
}

ErrorCode MakeErrorCode(ErrorCodeCategory category, ErrorCodeNumber number) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(category, 0u);

    ErrorCode errorCode = { category, number };
    return errorCode;
}

//--- ShowErrorApi ---//

void ErrorViewerStartupParamForSystemData::SetResult(const nn::Result& _result) NN_NOEXCEPT
{
    this->result = _result;
    this->isErrorCode = false;
}

void ErrorViewerStartupParamForSystemData::SetErrorCode(const ErrorCode& _errorCode) NN_NOEXCEPT
{
    this->errorCode = _errorCode;
    this->isErrorCode = true;
}

void ShowError(nn::Result result) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(result.IsFailure());

    if( pctl::ResultRestricted::Includes(result) )
    {
        ErrorViewerStartupParamForParentalControl param;
        param.pctlResultRestricted = result;
        InvokeErrorViewerApplet(&param, sizeof(param));
    }
    else
    {
        ErrorViewerStartupParamForSystemData param;
        param.SetResult(result);

        InvokeErrorViewerApplet(&param, sizeof(param));
    }
}

void ShowError(ErrorCode errorCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(errorCode.IsValid());

    ErrorViewerStartupParamForSystemData param;
    param.SetErrorCode(errorCode);

    InvokeErrorViewerApplet(&param, sizeof(param));
}

void ShowError(const ErrorResultVariant& errorResultVariant) NN_NOEXCEPT
{
    switch( errorResultVariant.GetState() )
    {
    case ErrorResultVariantState_HasResult:
        {
            ShowError(static_cast<nn::Result>(errorResultVariant));
        }
        break;

    case ErrorResultVariantState_HasErrorCode:
        {
            ShowError(static_cast<ErrorCode>(errorResultVariant));
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void ShowUnacceptableApplicationVersionError() NN_NOEXCEPT
{
    ErrorViewerStartupParamForSystemData param;
    param.SetResult(err::ResultUnacceptableApplicationVersion());

    InvokeErrorViewerApplet(&param, sizeof(param));
}

void ShowUnacceptableAddOnContentVersionError() NN_NOEXCEPT
{
    ErrorViewerStartupParamForSystemData param;
    param.SetResult(err::ResultUnacceptableAddOnContentVersion());

    InvokeErrorViewerApplet(&param, sizeof(param));
}

//--- ShowApplicationError ---//

void ShowApplicationError(const ApplicationErrorArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(strlen(arg.GetDialogMessage()) > 0 || strlen(arg.GetFullScreenMessage()) > 0);
    NN_SDK_REQUIRES(strlen(arg.GetLanguageCode().string) > 0);

    InvokeErrorViewerApplet(&arg, sizeof(ErrorViewerStartupParamForApplicationError));
}

//--- ShowErrorApiForSystem ---//

void ShowError(Result displayResult, const ResultBacktrace& resultBacktrace) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(displayResult.IsFailure());

    ErrorViewerStartupParamForSystemData param;
    param.hasResultBacktrace = true;
    param.SetResult(displayResult);

    InvokeErrorViewerApplet(&param, sizeof(param), &resultBacktrace, sizeof(ResultBacktrace));
}

void ShowError(Result result, const ErrorContext& errorContext) NN_NOEXCEPT
{
    ErrorViewerStartupParamForSystemData param;
    param.hasErrorContext = true;
    param.SetResult(result);

    InvokeErrorViewerApplet(&param, sizeof(param), &errorContext, sizeof(ErrorContext));
}

void ShowError(ErrorCode errorCode, const ErrorContext& errorContext) NN_NOEXCEPT
{
    ErrorViewerStartupParamForSystemData param;
    param.hasErrorContext = true;
    param.SetErrorCode(errorCode);

    InvokeErrorViewerApplet(&param, sizeof(param), &errorContext, sizeof(ErrorContext));
}

void ShowSystemError(const SystemErrorArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(strlen(arg.GetDialogMessage()) > 0 || strlen(arg.GetFullScreenMessage()) > 0);
    NN_SDK_REQUIRES(strlen(arg.GetLanguageCode().string) > 0);

    auto& param = arg.GetStartupParam();
    if( param.hasErrorContext )
    {
        NN_SDK_ASSERT_NOT_NULL(arg.GetErrorContext());
        InvokeErrorViewerApplet(&param, sizeof(ErrorViewerStartupParamForSystemError), arg.GetErrorContext(), sizeof(ErrorContext));
    }
    else
    {
        NN_SDK_ASSERT(arg.GetErrorContext() == nullptr);
        InvokeErrorViewerApplet(&param, sizeof(ErrorViewerStartupParamForSystemError));
    }
}

ErrorViewerJumpDestination ShowErrorWithoutJump(nn::Result result) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(result.IsFailure());

    if( pctl::ResultRestricted::Includes(result) )
    {
        ErrorViewerStartupParamForParentalControl param;
        param.common.isJumpEnabled = false;
        param.pctlResultRestricted = result;

        return InvokeErrorViewerApplet(&param, sizeof(param));
    }
    else
    {
        ErrorViewerStartupParamForSystemData param;
        param.common.isJumpEnabled = false;
        param.SetResult(result);

        return InvokeErrorViewerApplet(&param, sizeof(param));
    }
}

ErrorViewerJumpDestination ShowErrorWithoutJump(ErrorCode errorCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(errorCode.IsValid());

    ErrorViewerStartupParamForSystemData param;
    param.common.isJumpEnabled = false;
    param.SetErrorCode(errorCode);

    return InvokeErrorViewerApplet(&param, sizeof(param));
}

void ShowEula(nn::settings::system::RegionCode region) NN_NOEXCEPT
{
    ErrorViewerStartupParamForEula param;
    param.regionCode = region;

    InvokeErrorViewerApplet(&param, sizeof(param));
}

void ShowSystemUpdateEula(nn::settings::system::RegionCode region, EulaData& eulaData) NN_NOEXCEPT
{
    ErrorViewerStartupParamForSystemUpdateEula param;
    param.regionCode = region;

    InvokeErrorViewerAppletWithLargeStorage(&param, sizeof(param), &eulaData, sizeof(eulaData));
}

void ShowErrorRecord(nn::Result result, nn::time::PosixTime timeOfOccurrence) NN_NOEXCEPT
{
    ErrorViewerStartupParamForRecordedSystemData param;
    param.errorCode = detail::ConvertResultToErrorCode(result);
    param.timeOfOccurrence = timeOfOccurrence;

    InvokeErrorViewerApplet(&param, sizeof(param));
}

void ShowErrorRecord(ErrorCode errorCode, nn::time::PosixTime timeOfOccurrence) NN_NOEXCEPT
{
    ErrorViewerStartupParamForRecordedSystemData param;
    param.errorCode = errorCode;
    param.timeOfOccurrence = timeOfOccurrence;

    InvokeErrorViewerApplet(&param, sizeof(param));
}

bool CreateErrorViewerStartupParamForRecordedError(void* outBuffer, size_t* outActualSize, size_t outBufferSize, const char* errorCodeString, const char* errorMessage, nn::time::PosixTime timeOfOccurrence) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_EQUAL(0u, reinterpret_cast<uintptr_t>(outBuffer) % ErrorViewerStartupParamAddressAlignment);
    NN_SDK_REQUIRES_GREATER_EQUAL(outBufferSize, static_cast<size_t>(ErrorViewerStartupParamSizeMax));
    NN_UNUSED(outBufferSize);
    ErrorCode errorCode;
    if( detail::TryParseErrorCodeString(&errorCode, errorCodeString) )
    {
        if( errorMessage == nullptr )
        {
            // システムデータのエラー
            auto pParam = reinterpret_cast<ErrorViewerStartupParamForRecordedSystemData*>(outBuffer);
            pParam->common.errorType = err::ErrorType::RecordedSystemData;
            pParam->common.isJumpEnabled = false;
            pParam->version = 0;
            pParam->errorCode = errorCode;
            pParam->timeOfOccurrence = timeOfOccurrence;
            *outActualSize = sizeof(ErrorViewerStartupParamForRecordedSystemData);
            return true;
        }
        else
        {
            // メッセージ指定のエラー
            auto pParam = reinterpret_cast<ErrorViewerStartupParamForRecordedSystemError*>(outBuffer);
            pParam->common.errorType = err::ErrorType::RecordedSystemError;
            pParam->common.isJumpEnabled = false;
            pParam->version = 0;
            pParam->errorCode = errorCode;
            pParam->timeOfOccurrence = timeOfOccurrence;
            util::Strlcpy(pParam->message, errorMessage, sizeof(pParam->message));
            *outActualSize = sizeof(ErrorViewerStartupParamForRecordedSystemError);
            return true;
        }
    }
    ns::ApplicationErrorCodeCategory category;
    ApplicationErrorCodeNumber number;
    if( detail::TryParseApplicationErrorCodeString(&category, &number, errorCodeString) )
    {
        if( errorMessage == nullptr )
        {
            // アプリケーション固有のエラーは必ずメッセージを持つ。
            NN_DETAIL_ERR_WARN("Recorded ApplicationError must have a valid message.\n");
            return false;
        }
        else
        {
            auto pParam = reinterpret_cast<ErrorViewerStartupParamForRecordedApplicationError*>(outBuffer);
            pParam->common.errorType = err::ErrorType::RecordedApplicationError;
            pParam->common.isJumpEnabled = false;
            pParam->version = 0;
            util::Strlcpy(pParam->applicationErrorCodeCategory.value, category.value, sizeof(pParam->applicationErrorCodeCategory.value));
            pParam->applicationErrorCodeNumber = number;
            pParam->timeOfOccurrence = timeOfOccurrence;
            util::Strlcpy(pParam->message, errorMessage, sizeof(pParam->message));
            *outActualSize = sizeof(ErrorViewerStartupParamForRecordedApplicationError);
            return true;
        }
    }
    NN_DETAIL_ERR_WARN("errorCodeString(%s) is neither ErrorCode nor ApplicationErrorCode.\n", errorCodeString);
    return false;
}

void ShowErrorRecord(const void* recordedErrorParam, size_t recordedErrorParamSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(recordedErrorParam);
    auto errorType = reinterpret_cast<const ErrorViewerStartupParamCommon*>(recordedErrorParam)->errorType;
    NN_SDK_REQUIRES(errorType == ErrorType::RecordedSystemData || errorType == ErrorType::RecordedSystemError || errorType == ErrorType::RecordedApplicationError);
    NN_UNUSED(errorType);
    InvokeErrorViewerApplet(recordedErrorParam, recordedErrorParamSize);
}

//--- SystemApi ---//

ErrorCode ConvertResultToErrorCode(const Result& result) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(result.IsFailure());
    return err::detail::ConvertResultToErrorCode(result);
}

void GetErrorCodeString(char* outBuffer, size_t outBufferLength, ErrorCode errorCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(outBufferLength, static_cast<size_t>(ErrorCode::StringLengthMax));
    NN_SDK_REQUIRES(errorCode.IsValid());
    detail::MakeErrorCodeString(outBuffer, outBufferLength, errorCode);
}

nn::Result GetErrorMessageDatabaseVersion(ErrorMessageDatabaseVersion* outVersion) NN_NOEXCEPT
{
    err::detail::ReadVersion(outVersion);
    NN_RESULT_SUCCESS;
}

void ExecuteJump(ErrorViewerJumpDestination destination) NN_NOEXCEPT
{
    NN_UNUSED(destination);
    NN_DETAIL_ERR_INFO("ExecuteJump is not implemented yet.\n");
}

void ResultBacktrace::Make(ResultBacktrace* pOutValue, const Result results[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(results);
    NN_SDK_REQUIRES_MINMAX(count, 1, 32); // ResultBacktrace::CountMax = 32

    pOutValue->count = count;
    for( int i = 0; i < count; i++ )
    {
        pOutValue->results[i] = results[i];
    }
}

//--- ShowNetworkServiceErrorApi ---//

namespace {
bool IsErrorCodeForLicenseRequirementsForNetworkService(const ErrorCode& errorCode) NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    switch (errorCode.category)
    {
    case 2306: // NEX
        if (false
            || errorCode.number == 821 // QERROR(Authentication, OnlinePlayLicenseRequired), SIGLO-67140
            || (errorCode.number >= 823 && errorCode.number <= 824)) // SIGLO-69397
        {
            return true;
        }
        break;
    case 2307: // NEX Coral
        if (false
            || errorCode.number == 2118) // QERROR(OnlineLounge, OnlinePlayLicenseRequired), SIGLO-78088
        {
            return true;
        }
        break;
    case 2618: // Pia
        if (false
            || errorCode.number == 590 // SIGLO-60993
            || (errorCode.number >= 592 && errorCode.number <= 593)) // SIGLO-60993
        {
            return true;
        }
        break;
    default:
        break;
    }
#else
    NN_UNUSED(errorCode);
#endif
    return false;
}

void ShowLicenseRequirementsForNetworkService(ErrorCode errorCode, const account::UserHandle& userHandle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    account::ShowLicenseRequirementsForNetworkService(userHandle, errorCode);
#else
    NN_UNUSED(&errorCode);
    NN_UNUSED(&userHandle);
    NN_DETAIL_ERR_INFO("ShowNetworkServiceError behaves as same as ShowError on os=Win.\n");
#endif
}

void ShowLicenseRequirementsForNetworkService(ErrorCode errorCode) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    int openUserCount;
    account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListOpenUsers(&openUserCount, &uid, 1));
    NN_ABORT_UNLESS_EQUAL(openUserCount, 1);
    account::ShowLicenseRequirementsForNetworkService(uid, errorCode);
#else
    NN_UNUSED(&errorCode);
    NN_DETAIL_ERR_INFO("ShowNetworkServiceError behaves as same as ShowError on os=Win.\n");
#endif
}
} // ~namespace nn::err::<anonymous>

bool IsNetworkServiceError(ErrorCode errorCode) NN_NOEXCEPT
{
    return false
        || IsErrorCodeForLicenseRequirementsForNetworkService(errorCode);
}

bool IsNetworkServiceError(Result result) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(result.IsFailure());
    return IsNetworkServiceError(ConvertResultToErrorCode(result));
}

void ShowNetworkServiceError(ErrorCode errorCode, const account::UserHandle& userHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsNetworkServiceError(errorCode));
    if (IsErrorCodeForLicenseRequirementsForNetworkService(errorCode))
    {
        ShowLicenseRequirementsForNetworkService(errorCode, userHandle);
        return;
    }
    ShowError(errorCode);
}

void ShowNetworkServiceError(ErrorCode errorCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsNetworkServiceError(errorCode));
    if (IsErrorCodeForLicenseRequirementsForNetworkService(errorCode))
    {
        ShowLicenseRequirementsForNetworkService(errorCode);
        return;
    }
    ShowError(errorCode);
}

void ShowNetworkServiceError(Result result, const account::UserHandle& userHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsNetworkServiceError(result));
    auto errorCode = ConvertResultToErrorCode(result);
    if (IsErrorCodeForLicenseRequirementsForNetworkService(errorCode))
    {
        ShowLicenseRequirementsForNetworkService(errorCode, userHandle);
        return;
    }
    ShowError(result);
}

void ShowNetworkServiceError(Result result) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsNetworkServiceError(result));
    auto errorCode = ConvertResultToErrorCode(result);
    if (IsErrorCodeForLicenseRequirementsForNetworkService(errorCode))
    {
        ShowLicenseRequirementsForNetworkService(errorCode);
        return;
    }
    ShowError(result);
}

}}
