﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/err/detail/err_SystemData.h>
#include <nn/err/detail/err_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_TFormatString.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/settings/settings_Language.h>
#include <algorithm>

namespace nn { namespace err { namespace detail {

namespace {

    const char* GetMessageFileSuffix(MessageKind file) NN_NOEXCEPT
    {
        switch( file )
        {
        case MessageKind::Dialog:
            return "DlgMsg";
        case MessageKind::DialogButton:
            return "DlgBtn";
        case MessageKind::FullScreen:
            return "FlvMsg";
        case MessageKind::FullScreenButton:
            return "FlvBtn";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // 指定された言語を、エラーメッセージが存在することが既知である言語に変換する（"エラーメッセージの存在する言語の数 <= 本体で設定できる言語の数" であるため）。
    settings::LanguageCode FallbackToKnownExistentErrorMessageLanguageCode(const settings::LanguageCode& languageCode) NN_NOEXCEPT
    {
        NN_UNUSED(languageCode);
        // 一律で英語にフォールバック。
        return settings::LanguageCode::Make(settings::Language_AmericanEnglish);
    }

    bool FileExists(const char* path) NN_NOEXCEPT
    {
        fs::DirectoryEntryType entry;
        return fs::GetEntryType(&entry, path).IsSuccess() && entry == fs::DirectoryEntryType_File;
    }
}

void MakeErrorInfoModuleDirectoryPath(char* outBuffer, size_t bufferLength, ErrorCodeCategory module) NN_NOEXCEPT
{
    util::TSNPrintf(outBuffer, bufferLength, "%s:/%04d", NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, module);
}

void MakeErrorInfoDirectoryPath(char* outBuffer, size_t bufferLength, ErrorCode errorCode) NN_NOEXCEPT
{
    util::TSNPrintf(outBuffer, bufferLength, "%s:/%04d/%04d", NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, errorCode.category, errorCode.number);
}

void MakeErrorInfoCommonFilePath(char* outBuffer, size_t bufferLength, ErrorCode errorCode) NN_NOEXCEPT
{
    util::TSNPrintf(outBuffer, bufferLength, "%s:/%04d/%04d/common", NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, errorCode.category, errorCode.number);
}

void MakeErrorInfoMessageFilePath(char* outBuffer, size_t bufferLength, ErrorCode errorCode, nn::settings::LanguageCode langCode, MessageKind messageKind) NN_NOEXCEPT
{
    util::TSNPrintf(outBuffer, bufferLength, "%s:/%04d/%04d/%s_%s", NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, errorCode.category, errorCode.number, langCode.string, GetMessageFileSuffix(messageKind));
}

bool ErrorMessageDataExists(ErrorCode errorCode) NN_NOEXCEPT
{
    const int DirectoryPathLengthMax = 32;
    char filePath[DirectoryPathLengthMax];
    MakeErrorInfoDirectoryPath(filePath, DirectoryPathLengthMax, errorCode);

    nn::fs::DirectoryHandle handle;
    auto result = nn::fs::OpenDirectory(&handle, filePath, nn::fs::OpenDirectoryMode_Directory);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
    {
        return false;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY
    nn::fs::CloseDirectory(handle);
    return true;
}

bool CategoryExists(ErrorCodeCategory category) NN_NOEXCEPT
{
    const int DirectoryPathLengthMax = 32;
    char filePath[DirectoryPathLengthMax];
    MakeErrorInfoModuleDirectoryPath(filePath, DirectoryPathLengthMax, category);

    nn::fs::DirectoryHandle handle;
    auto result = nn::fs::OpenDirectory(&handle, filePath, nn::fs::OpenDirectoryMode_Directory);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
    {
        return false;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY
    nn::fs::CloseDirectory(handle);
    return true;
}

bool DefaultErrorMessageDataExists(ErrorCodeCategory module) NN_NOEXCEPT
{
    const int DirectoryPathLengthMax = 32;
    char filePath[DirectoryPathLengthMax];
    ErrorCode moduleDefault = { module, DefaultMessageErrorNumberId };
    MakeErrorInfoDirectoryPath(filePath, DirectoryPathLengthMax, moduleDefault);

    nn::fs::DirectoryHandle handle;
    auto result = nn::fs::OpenDirectory(&handle, filePath, nn::fs::OpenDirectoryMode_Directory);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
    {
        return false;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY
    nn::fs::CloseDirectory(handle);
    return true;
}

nn::Result ReadMessageFile(char16_t* outBuffer, int* outMessageLength, size_t bufferSize, ErrorCode errorCode, nn::settings::LanguageCode langCode, MessageKind messageKind) NN_NOEXCEPT
{
    const int FilePathLengthMax = 32;
    char filePath[FilePathLengthMax];
    MakeErrorInfoMessageFilePath(filePath, FilePathLengthMax, errorCode, langCode, messageKind);
    if( !FileExists(filePath) )
    {
        MakeErrorInfoMessageFilePath(filePath, FilePathLengthMax, errorCode, FallbackToKnownExistentErrorMessageLanguageCode(langCode), messageKind);
    }

    nn::fs::FileHandle fileHandle;
    auto result = nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Read);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
    {
        // 全画面表示のみ対応しているエラー情報のダイアログ表示用メッセージファイルなど
        // 正常系でもファイルが存在しない場合があるため「ファイルが見つからない」は成功扱いにする。
        *outMessageLength = 0;
        outBuffer[0] = 0;
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, fileHandle));
    // メッセージファイルは UTF-16 のテキストファイルなので端数は出ないはず。
    NN_ABORT_UNLESS_EQUAL(0, fileSize % sizeof(char16_t));
    *outMessageLength = static_cast<int>(fileSize / sizeof(char16_t));
    NN_ABORT_UNLESS_GREATER(bufferSize, static_cast<size_t>(*outMessageLength));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(fileHandle, 0, outBuffer, static_cast<size_t>(fileSize)));
    outBuffer[*outMessageLength] = 0;

    NN_RESULT_SUCCESS;
}

void ReadVersion(ErrorMessageDatabaseVersion* outVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(outVersion != nullptr, "'outVersion' is null");
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const nn::ncm::SystemDataId ErrorMessageSystemDataId = { NN_ERR_SYSTEM_DATA_ID_OF_ERROR_MESSAGE };
    auto mountResult = nn::fs::MountSystemData(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, ErrorMessageSystemDataId);
    if( mountResult.IsFailure() )
    {
        NN_DETAIL_ERR_ERROR("Failed to mount ErrorMessage(0x%08x) due to error 0x%08x.\n", NN_ERR_SYSTEM_DATA_ID_OF_ERROR_MESSAGE, mountResult.GetInnerValueForDebug());
        return;
    }
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    // TORIAEZU: デバッグ用に PC 上のディレクトリをマウント
    const char* testHostDirectory = "C:/Temp/ErrorMessage";
    auto mountResult = nn::fs::MountHost(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, testHostDirectory);
    if( mountResult.IsFailure() )
    {
        NN_DETAIL_ERR_ERROR("Failed to mount ErrorMessage test directory.\n");
        return;
    }
#else
#error "unsupported os"
#endif
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME); };

    const size_t FilePathLengthMax = 32;
    char infoFilePath[FilePathLengthMax];
    util::TSNPrintf(infoFilePath, FilePathLengthMax, "%s:/DatabaseInfo", NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME);

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, infoFilePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };
    // DatabaseInfo にはバージョン情報しか含まれていないので直接 Read
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, outVersion, sizeof(ErrorMessageDatabaseVersion)));
}

void ReadMessageFile(char16_t* outBuffer, size_t bufferSize, const char* key, const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outBuffer);

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const nn::ncm::SystemDataId ErrorMessageSystemDataId = { NN_ERR_SYSTEM_DATA_ID_OF_ERROR_MESSAGE };
    auto mountResult = nn::fs::MountSystemData(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, ErrorMessageSystemDataId);
    if( mountResult.IsFailure() )
    {
        NN_DETAIL_ERR_ERROR("Failed to mount ErrorMessage(0x%08x) due to error 0x%08x.\n", NN_ERR_SYSTEM_DATA_ID_OF_ERROR_MESSAGE, mountResult.GetInnerValueForDebug());
        outBuffer[0] = 0;
        return;
    }
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    // TORIAEZU: デバッグ用に PC 上のディレクトリをマウント
    const char* testHostDirectory = "C:/Temp/ErrorMessage";
    auto mountResult = nn::fs::MountHost(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, testHostDirectory);
    if( mountResult.IsFailure() )
    {
        NN_DETAIL_ERR_ERROR("Failed to mount ErrorMessage test directory.\n");
        return;
    }
#else
#error "unsupported os"
#endif
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME); };

    const int FilePathLengthMax = 32;
    char filePath[FilePathLengthMax];
    util::TSNPrintf(filePath, FilePathLengthMax, "%s:/Messages/%s_%s", NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, languageCode.string, key);
    if( !FileExists(filePath) )
    {
        util::TSNPrintf(filePath, FilePathLengthMax, "%s:/Messages/%s_%s", NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, FallbackToKnownExistentErrorMessageLanguageCode(languageCode).string, key);
    }
    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };
    size_t readSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, 0, outBuffer, bufferSize));
    outBuffer[std::min(readSize / sizeof(char16_t), bufferSize - 1)] = 0;
}

}}}
