﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/edid.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_SystemApi.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/settings/system/settings_Backlight.h>
#include <nn/settings/system/settings_BatteryLot.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/settings/system/settings_Capture.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/settings/system/settings_DataDeletion.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Hid.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Ldn.h>
#include <nn/settings/system/settings_Network.h>
#include <nn/settings/system/settings_NetworkService.h>
#include <nn/settings/system/settings_News.h>
#include <nn/settings/system/settings_Nfc.h>
#include <nn/settings/system/settings_ProductModel.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Telemetry.h>
#include <nn/settings/system/settings_TimeZone.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/system/settings_Usb.h>
#include <nn/settings/system/settings_WirelessLan.h>
#include <nn/time.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include "eclct_Settings.h"
#include "eclct_Util.h"

namespace nn    {
namespace eclct {
namespace       {

const char* Combine(char* pBuffer, uint32_t bufferLength, const char* pString1, const char* pString2)
NN_NOEXCEPT
{
    nn::util::SNPrintf(pBuffer, bufferLength, "%s%s", pString1, pString2);
    return pBuffer;
}

void UpdateLanguageCode()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::CurrentLanguageInfo);

    nn::settings::LanguageCode value = {};
    nn::settings::GetLanguageCode(&value);

    context.Add(
        nn::erpt::CurrentLanguage,
        value.string,
        static_cast<uint32_t>(nn::util::Strnlen(value.string, sizeof(value.string)))
    );

    context.SubmitContext();
}

const char* RegionString(nn::settings::system::RegionCode value)
NN_NOEXCEPT
{
    switch (value)
    {
    case nn::settings::system::RegionCode_Japan:     return "Japan";
    case nn::settings::system::RegionCode_Usa:       return "Usa";
    case nn::settings::system::RegionCode_Europe:    return "Europe";
    case nn::settings::system::RegionCode_Australia: return "Australia";
    case nn::settings::system::RegionCode_China:     return "China";
    case nn::settings::system::RegionCode_Korea:     return "Korea";
    case nn::settings::system::RegionCode_Taiwan:    return "Taiwan";
    default:
        break;
    }
    return "RegionUnknown";
}

void UpdateRegionCode()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::RegionSettingInfo);

    nn::settings::system::RegionCode value;
    nn::settings::system::GetRegionCode(&value);

    char tmp[256];

    context.Add(
        nn::erpt::RegionSetting,
        Combine(tmp, sizeof(tmp), RegionString(value), ""),
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.SubmitContext();
}

void UpdateBacklightSettings()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::ScreenBrightnessInfo);

    nn::settings::system::BacklightSettings value = {};
    nn::settings::system::GetBacklightSettings(&value);

    char tmp[256];

    context.Add(
        nn::erpt::ScreenBrightnessAutoAdjustFlag,
        value.flags.Test<nn::settings::system::BacklightFlag::Automatic>()
    );

    nn::util::SNPrintf(tmp, sizeof(tmp), "%f", value.brightness);

    context.Add(
        nn::erpt::ScreenBrightnessLevel,
        tmp,
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.SubmitContext();
}

void UpdateNotificationSettings()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::NotificationInfo);

    nn::settings::system::NotificationSettings value = {};
    nn::settings::system::GetNotificationSettings(&value);

    context.Add(
        nn::erpt::NotifyInGameDownloadCompletionFlag,
        value.flags.Test<nn::settings::system::NotificationFlag::DownloadCompletion>()
    );

    context.Add(
        nn::erpt::NotificationSoundFlag,
        value.flags.Test<nn::settings::system::NotificationFlag::Ringtone>()
    );

    context.SubmitContext();
}

void UpdateVibrationMasterVolume()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::ControllerVibrationInfo);

    char tmp[256];

    nn::util::SNPrintf(tmp, sizeof(tmp), "%f", nn::settings::system::GetVibrationMasterVolume());

    context.Add(
        nn::erpt::ControllerVibrationVolume,
        tmp,
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.SubmitContext();
}

const char* ResolutionString(int32_t resolution)
NN_NOEXCEPT
{
    switch (resolution)
    {
    case nn::settings::system::TvResolution_Auto:  return "Auto";
    case nn::settings::system::TvResolution_1080p: return "1080p";
    case nn::settings::system::TvResolution_720p:  return "720p";
    case nn::settings::system::TvResolution_480p:  return "480p";
    default:
        break;
    }
    return "UnknownResolution";
}

const char* RGBRangeString(int32_t rgbrange)
NN_NOEXCEPT
{
    switch (rgbrange)
    {
    case nn::settings::system::RgbRange_Auto:    return "Auto";
    case nn::settings::system::RgbRange_Full:    return "Full";
    case nn::settings::system::RgbRange_Limited: return "Limited";
    default:
        break;
    }
    return "UnknownRGBRange";
}

void UpdateTvSettings()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::TVInfo);

    nn::settings::system::TvSettings value = {};
    nn::settings::system::GetTvSettings(&value);

    char tmp[256];

    context.Add(
        nn::erpt::ReduceScreenBurnFlag,
        value.flags.Test<nn::settings::system::TvFlag::PreventsScreenBurnIn>()
    );

    context.Add(
        nn::erpt::TVAllowsCecFlag,
        value.flags.Test<nn::settings::system::TvFlag::AllowsCec>()
    );

    context.Add(
        nn::erpt::TVResolutionSetting,
        Combine(tmp, sizeof(tmp), ResolutionString(value.tvResolution), ""),
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.Add(
        nn::erpt::RGBRangeSetting,
        Combine(tmp, sizeof(tmp), RGBRangeString(value.rgbRange), ""),
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.SubmitContext();
}

void UpdateForceMute()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::MuteOnHeadsetUnpluggedInfo);

    context.Add(
        nn::erpt::MuteOnHeadsetUnpluggedFlag,
        nn::settings::system::IsForceMuteOnHeadphoneRemoved()
    );

    context.SubmitContext();
}

void UpdateDeviceTimeZoneLocationName()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::TimeZoneInfo);

    nn::time::LocationName value = {};

    nn::settings::system::GetDeviceTimeZoneLocationName(&value);

    context.Add(
        nn::erpt::TimeZone,
        value._value,
        static_cast<uint32_t>(nn::util::Strnlen(value._value, sizeof(value._value)))
    );

    context.SubmitContext();
}

void UpdateUsb30EnableFlag()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::USB3AvailableInfo);

    context.Add(
        nn::erpt::USB3AvailableFlag,
        nn::settings::system::IsUsb30Enabled()
    );

    context.SubmitContext();
}

void UpdateNfcEnableFlag()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::EnableNFCInfo);

    context.Add(
        nn::erpt::EnableNFCFlag,
        nn::settings::system::IsNfcEnabled()
    );

    context.SubmitContext();
}

const char* HandheldSleepTimeString(int32_t value)
NN_NOEXCEPT
{
    switch (value)
    {
    case nn::settings::system::HandheldSleepPlan_1Min:  return "1Min";
    case nn::settings::system::HandheldSleepPlan_3Min:  return "3Min";
    case nn::settings::system::HandheldSleepPlan_5Min:  return "5Min";
    case nn::settings::system::HandheldSleepPlan_10Min: return "10Min";
    case nn::settings::system::HandheldSleepPlan_30Min: return "30Min";
    case nn::settings::system::HandheldSleepPlan_Never: return "Never";
    default:
        break;
    }
    return "Unknown";
}

const char* ConsoleSleepTimeString(int32_t value)
NN_NOEXCEPT
{
    switch (value)
    {
    case nn::settings::system::ConsoleSleepPlan_1Hour:  return "1Hour";
    case nn::settings::system::ConsoleSleepPlan_2Hour:  return "2Hour";
    case nn::settings::system::ConsoleSleepPlan_3Hour:  return "3Hour";
    case nn::settings::system::ConsoleSleepPlan_6Hour:  return "6Hour";
    case nn::settings::system::ConsoleSleepPlan_12Hour: return "12Hour";
    case nn::settings::system::ConsoleSleepPlan_Never:  return "Never";
    default:
        break;
    }
    return "Unknown";
}

void UpdateSleepSettings()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::SleepInfo);

    nn::settings::system::SleepSettings value = {};
    nn::settings::system::GetSleepSettings(&value);

    char tmp[256];

    context.Add(
        nn::erpt::HandheldModeTimeToScreenSleep,
        Combine(tmp, sizeof(tmp), HandheldSleepTimeString(value.handheldSleepPlan), ""),
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.Add(
        nn::erpt::ConsoleModeTimeToScreenSleep,
        Combine(tmp, sizeof(tmp), ConsoleSleepTimeString(value.consoleSleepPlan), ""),
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.Add(
        nn::erpt::StopAutoSleepDuringContentPlayFlag,
        value.flags.Test<nn::settings::system::SleepFlag::SleepsWhilePlayingMedia>()
    );

    context.SubmitContext();
}

void UpdateWirelessLanEnableFlag()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::EnableWifiInfo);

    context.Add(
        nn::erpt::EnableWifiFlag,
        nn::settings::system::IsWirelessLanEnabled()
    );

    context.SubmitContext();
}

void UpdateUserSystemClockAutomaticCorrection()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::UseNetworkTimeProtocolInfo);

    context.Add(
        nn::erpt::UseNetworkTimeProtocolFlag,
        nn::settings::system::IsUserSystemClockAutomaticCorrectionEnabled()
    );

    context.SubmitContext();
}

const char* AudioFormatString(nn::settings::system::AudioOutputMode value)
NN_NOEXCEPT
{
    switch (value)
    {
    case nn::settings::system::AudioOutputMode_1ch:   return "1ch";
    case nn::settings::system::AudioOutputMode_2ch:   return "2ch";
    case nn::settings::system::AudioOutputMode_5_1ch: return "5.1ch";
    case nn::settings::system::AudioOutputMode_7_1ch: return "7.1ch";
    case nn::settings::system::AudioOutputMode_5_1ch_Forced: return "5.1ch Forced";
    default:
        break;
    }
    return "Unknown";
}

void UpdateAudioOutputMode()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::AudioFormatInfo);
    char tmp[256];

    context.Add(
        nn::erpt::HdmiAudioOutputMode,
        Combine(
            tmp,
            sizeof(tmp),
            AudioFormatString(nn::settings::system::GetAudioOutputMode(nn::settings::system::AudioOutputModeTarget_Hdmi)),
            ""),
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    context.Add(
        nn::erpt::SpeakerAudioOutputMode,
        Combine(
            tmp,
            sizeof(tmp),
            AudioFormatString(nn::settings::system::GetAudioOutputMode(nn::settings::system::AudioOutputModeTarget_Speaker)),
            ""),
        static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
    );

    // NX platform does not support AudioTarget_Headphone. The value from AudioTarget_Speaker is used for the headphone's output setting.
    // So nn::erpt::HeadphoneAudioOutputMode is not added here.

    context.SubmitContext();
}

void UpdateLockScreenFlag()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::LockScreenInfo);

    context.Add(
        nn::erpt::LockScreenFlag,
        nn::settings::system::GetLockScreenFlag()
    );

    context.SubmitContext();
}

void UpdateDataDeletionSettings()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::DataDeletionInfo);

    nn::settings::system::DataDeletionSettings value = {};
    nn::settings::system::GetDataDeletionSettings(&value);

    context.Add(
        nn::erpt::StorageAutoOrganizeFlag,
        value.flags.Test<nn::settings::system::DataDeletionFlag::Automatic>()
    );

    context.SubmitContext();
}

void UpdateBluetoothEnableFlag()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::EnableBluetoothInfo);

    context.Add(
        nn::erpt::EnableBluetoothFlag,
        nn::settings::system::IsBluetoothEnabled()
    );

    context.SubmitContext();
}

void UpdateNetworkSettings()
NN_NOEXCEPT
{

}

void UpdateExternalSteadyClockSourceId()
NN_NOEXCEPT
{

}

void UpdateUserSystemClockContext()
NN_NOEXCEPT
{
    time::SystemClockContext userClockContext;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetSystemClockContext(&userClockContext));

    nn::erpt::Context context(erpt::CategoryId::UserClockContextInfo);
    context.Add(erpt::FieldId::UserClockContextOffset, userClockContext.offset);
    context.Add(erpt::FieldId::UserClockContextTimeStampValue, userClockContext.timeStamp.value);
    context.SubmitContext();
}

void UpdateAccountSettings()
NN_NOEXCEPT
{

}

void UpdateAudioVolume()
NN_NOEXCEPT
{

}

void UpdateEulaVersions()
NN_NOEXCEPT
{

}

void UpdateColorSetId()
NN_NOEXCEPT
{

}

void UpdateConsoleInformationUploadFlag()
NN_NOEXCEPT
{

}

void UpdateAutomaticApplicationDownloadFlag()
NN_NOEXCEPT
{

}

void UpdateQuestFlag()
NN_NOEXCEPT
{
    auto context = erpt::Context(erpt::CategoryId::RetailInteractiveDisplayInfo);
    context.Add(erpt::FieldId::RetailInteractiveDisplayFlag, settings::system::GetQuestFlag());
    context.SubmitContext();
}

void UpdateNetworkSystemClockContext()
NN_NOEXCEPT
{
    time::SystemClockContext networkClockContext;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardNetworkSystemClock::GetSystemClockContext(&networkClockContext));

    auto context = erpt::Context(erpt::CategoryId::NetworkClockContextInfo);
    context.Add(erpt::FieldId::NetworkClockContextOffset, networkClockContext.offset);
    context.Add(erpt::FieldId::NetworkClockContextTimeStampValue, networkClockContext.timeStamp.value);
    context.SubmitContext();
}

void UpdatePrimaryAlbumStorage()
NN_NOEXCEPT
{

}

void UpdateInitialLaunchSettings()
NN_NOEXCEPT
{
    settings::system::InitialLaunchSettings initialSettings;
    settings::system::GetInitialLaunchSettings(&initialSettings);

    if( initialSettings.flags.Test<settings::system::InitialLaunchFlag::HasTimeStamp>() )
    {
        erpt::SetInitialLaunchSettingsCompletionTime(initialSettings.timeStamp);
    }
    else
    {
        erpt::ClearInitialLaunchSettingsCompletionTime();
    }
}

void UpdateDeviceNickName()
NN_NOEXCEPT
{

}

void UpdateLdnChannel()
NN_NOEXCEPT
{

}

static bool GetDisplayModes(const nn::edid::DisplayModeInfo* pMode, const nn::edid::DisplayTimingInfo* pTimingInfo, const nn::edid::ImageSizeInfo* pSizeInfo, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pTimingInfo);
    NN_UNUSED(pSizeInfo);

    bool* list = static_cast<bool*>(pUserData);

    if( pMode->width == 3840 &&
        pMode->height == 2160 &&
        pMode->refreshRate == 60.00f &&
        pMode->imageAspectRatio == 16.f / 9.f &&
        pMode->stereoMode == nn::edid::StereoMode_None &&
        pMode->isInterlaced == false )
      {
          list[0] = true;
      }

    if( pMode->width == 3840 &&
        pMode->height == 2160 &&
        pMode->refreshRate == 30.00f &&
        pMode->imageAspectRatio == 16.f / 9.f &&
        pMode->stereoMode == nn::edid::StereoMode_None &&
        pMode->isInterlaced == false )
      {
          list[1] = true;
      }
    if( pMode->width == 1920 &&
        pMode->height == 1080 &&
        pMode->refreshRate == 60.00f &&
        pMode->imageAspectRatio == 16.f / 9.f &&
        pMode->stereoMode == nn::edid::StereoMode_None &&
        pMode->isInterlaced == false )
      {
          list[2] = true;
      }

    if( pMode->width == 1280 &&
        pMode->height == 720 &&
        pMode->refreshRate == 60.00f &&
        pMode->imageAspectRatio == 16.f / 9.f &&
        pMode->stereoMode == nn::edid::StereoMode_None &&
        pMode->isInterlaced == false )
      {
          list[3] = true;
      }

    return true;
}
static bool GetAudioModes(const nn::edid::AudioFormatInfo* pMode, void* pUserData) NN_NOEXCEPT
{
    int* count = static_cast<int*>(pUserData);
    if(pMode->format == nn::edid::AudioFormat_Pcm)
        *count = pMode->channelCountMax;
    return true;
}

void UpdateMonitorCapability(nn::settings::system::Edid* edid) NN_NOEXCEPT
{

    nn::edid::Edid edidHandle;
    if(nn::edid::Error_None == nn::edid::OpenEdid(&edidHandle, edid, sizeof(nn::settings::system::Edid)))
    {

        auto context = nn::erpt::Context(nn::erpt::CategoryId::MonitorCapability);

        nn::edid::DisplayInfo display;
        nn::edid::GetDisplayInfo(&display, &edidHandle);

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::MonitorManufactureCode, display.manufacturerCode, sizeof(display.manufacturerCode)));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::MonitorProductCode, display.productCode));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::MonitorSerialNumber, display.serialNumber));

        int year = 0;
        nn::edid::GetDisplayYearOfManufacture(&year, &display);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::MonitorManufactureYear, year));

        std::uint16_t address = 0;
        nn::edid::GetSourcePhysicalAddress(&address, &edidHandle);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PhysicalAddress, address));

        bool mode[4] ={ false };

        nn::edid::VisitDisplayModes(&edidHandle, GetDisplayModes, mode);

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::Is4k60Hz, mode[0]));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::Is4k30Hz, mode[1]));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::Is1080P60Hz, mode[2]));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::Is720P60Hz, mode[3]));

        int channel = 0;
        nn::edid::VisitAudioFormats(&edidHandle, GetAudioModes, &channel);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PcmChannelMax, channel));

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }
}

void UpdateEdid()
NN_NOEXCEPT
{
    Bit8 buffer[sizeof(nn::settings::system::Edid)];
    nn::erpt::Context context(nn::erpt::CategoryId::EdidInfo, buffer, sizeof(buffer));

    nn::settings::system::Edid edid;
    nn::settings::system::GetEdid(&edid);

    context.Add(nn::erpt::FieldId::EdidBlock, edid.data0, sizeof(edid.data0));
    context.Add(nn::erpt::FieldId::EdidExtensionBlock, edid.data1, sizeof(edid.data1));

    context.SubmitContext();

    UpdateMonitorCapability(&edid);
}

void UpdateErrorReportSharePermission()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::CategoryId::ErrorReportSharePermissionInfo);

    auto permission = nn::settings::system::GetErrorReportSharePermission();
    context.Add(nn::erpt::FieldId::ErrorReportSharePermission, static_cast<uint8_t>(permission));

    context.SubmitContext();
}

} // ~nn::eclct::<anonymous>

void UpdateStaticFields()
NN_NOEXCEPT
{
    nn::settings::system::BatteryLot value = {};
    nn::settings::system::GetBatteryLot(&value);

    nn::erpt::Context context(nn::erpt::InternalBatteryLotNumberInfo);

    context.Add(
        nn::erpt::InternalBatteryLotNumber,
        value.string,
        static_cast<uint32_t>(nn::util::Strnlen(value.string, sizeof(value.string)))
    );

    context.SubmitContext();
}

void UpdateSettingsInfo()
NN_NOEXCEPT
{
    nn::settings::system::TelemetryDirtyFlagSet flags = {};
    nn::settings::system::GetTelemetryDirtyFlags(&flags);

    if (flags.CountPopulation())
    {
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::LanguageCode>())                       UpdateLanguageCode();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::NetworkSettings>())                    UpdateNetworkSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::RegionCode>())                         UpdateRegionCode();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::LockScreenFlag>())                     UpdateLockScreenFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::BacklightSettings>())                  UpdateBacklightSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::BluetoothEnableFlag>())                UpdateBluetoothEnableFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::ExternalSteadyClockSourceId>())        UpdateExternalSteadyClockSourceId();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::UserSystemClockContext>())             UpdateUserSystemClockContext();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::AccountSettings>())                    UpdateAccountSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::AudioVolume>())                        UpdateAudioVolume();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::EulaVersions>())                       UpdateEulaVersions();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::ColorSetId>())                         UpdateColorSetId();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::ConsoleInformationUploadFlag>())       UpdateConsoleInformationUploadFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::AutomaticApplicationDownloadFlag>())   UpdateAutomaticApplicationDownloadFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::NotificationSettings>())               UpdateNotificationSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::VibrationMasterVolume>())              UpdateVibrationMasterVolume();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::TvSettings>())                         UpdateTvSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::AudioOutputMode>())                    UpdateAudioOutputMode();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::ForceMute>())                          UpdateForceMute();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::QuestFlag>())                          UpdateQuestFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::DataDeletionSettings>())               UpdateDataDeletionSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::DeviceTimeZoneLocationName>())         UpdateDeviceTimeZoneLocationName();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::NetworkSystemClockContext>())          UpdateNetworkSystemClockContext();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::UserSystemClockAutomaticCorrection>()) UpdateUserSystemClockAutomaticCorrection();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::PrimaryAlbumStorage>())                UpdatePrimaryAlbumStorage();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::Usb30EnableFlag>())                    UpdateUsb30EnableFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::NfcEnableFlag>())                      UpdateNfcEnableFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::SleepSettings>())                      UpdateSleepSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::WirelessLanEnableFlag>())              UpdateWirelessLanEnableFlag();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::InitialLaunchSettings>())              UpdateInitialLaunchSettings();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::DeviceNickName>())                     UpdateDeviceNickName();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::LdnChannel>())                         UpdateLdnChannel();
        if (flags.Test<nn::settings::system::TelemetryDirtyFlag::Edid>())                               UpdateEdid();
        if( flags.Test<nn::settings::system::TelemetryDirtyFlag::ErrorReportSharePermission>() )        UpdateErrorReportSharePermission();
    }
}

}}
