﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ec/ec_ShopApiPurchaseAuthorization.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/ec/detail/ec_ShopApiInternal.h>
#include <nn/ec/detail/ec_ShopTypesInternal.h>
#include <nn/ec/detail/ec_TypesInternal.h>
#include <nn/ec/detail/ec_Utils.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ec {

namespace
{
    bool IsValidPriceRawValue(const char* priceRawValue) NN_NOEXCEPT
    {
        // 正数の形式かどうかを判定
        // OK: 「1203」「12.345」「.23」「3.」「0」「0.4」「0.0」「0.」
        // NG: 「012」「.」「12.34.」「-1」「」「1AB」
        // (正規表現: (0|[1-9][0-9]*){0,1}(\.[0-9]*){0,1} (ただし空文字列や「.」のみは許容しない))
        if (priceRawValue == nullptr)
        {
            return false;
        }
        if (*priceRawValue == '\0')
        {
            return false;
        }
        // '.' のみは許容しない
        if (*priceRawValue == '.' && *(priceRawValue + 1) == '\0')
        {
            return false;
        }
        if (*priceRawValue == '0')
        {
            ++priceRawValue;
            if (!(*priceRawValue == '\0' || *priceRawValue == '.'))
            {
                return false;
            }
        }
        else
        {
            // (先頭の「.」は許可する)
            while (*priceRawValue != '\0' && *priceRawValue != '.')
            {
                if (!(*priceRawValue >= '0' && *priceRawValue <= '9'))
                {
                    return false;
                }
                ++priceRawValue;
            }
        }
        if (*priceRawValue == '\0')
        {
            return true;
        }
        if (*priceRawValue != '.')
        {
            return false;
        }
        ++priceRawValue;
        // ('.' で終わるのは許容する)
        while (*priceRawValue != '\0')
        {
            if (!(*priceRawValue >= '0' && *priceRawValue <= '9'))
            {
                return false;
            }
            ++priceRawValue;
        }
        return true;
    }
}

nn::Result RequestPurchaseAuthorization(const nn::account::UserHandle& selectedUser,
    const char* totalPriceRawValue) NN_NOEXCEPT
{
    nn::account::Uid uid = nn::account::InvalidUid;
    NN_RESULT_DO(nn::account::GetUserId(&uid, selectedUser));

    if (!IsValidPriceRawValue(totalPriceRawValue))
    {
        NN_SDK_LOG("warning: 'totalPriceRawValue' is not valid price raw-value\n");
        NN_RESULT_THROW(nn::ec::ResultShopInvalidParameter());
    }

    detail::ShowShopPageArgWrapper arg(detail::g_ShopPageArgStorage, detail::g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        detail::GeneratePath(arg, detail::PathShopBase, SourceId_Default, nullptr,
            "scene", "service_prepurchase",
            "price", totalPriceRawValue,
            nullptr, nullptr);
    }

    NN_RESULT_THROW_UNLESS(detail::ShowShopPageWrapper(arg, uid), nn::ec::ResultShopCanceled());

    auto& returnValue = detail::GetShopPageReturnValue();
    // success を表す戻り値返したときのみ NN_RESULT_SUCCESS として
    // それ以外を nn::ec::ResultShopCanceled とする
    NN_RESULT_THROW_UNLESS(returnValue.GetShopExitReason() == detail::ShopExitReason_CallbackUrlReached &&
        returnValue.GetLastUrlSize() == detail::ShopSuccessLastUrlSize &&
        std::strncmp(returnValue.GetLastUrl(), detail::ShopSuccessLastUrl, detail::ShopSuccessLastUrlSize) == 0, // NULL文字込み
        nn::ec::ResultShopCanceled());

    NN_RESULT_SUCCESS;
}

}}
