﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dmnt/dmnt_Api.h>
#include <nn/dmnt/detail/dmnt_Interface.h>

#include <nn/svc/svc_Handle.h>

#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include "..\..\TargetTools\SnapShotDumper\coredump\coredump_Data.h"

//==============================================================================

namespace nn { namespace dmnt {

    namespace
    {
        struct AllocatorTag
        {
        };

        typedef nn::sf::ExpHeapStaticAllocator<1024, AllocatorTag> MyAllocator;

        nn::sf::SharedPointer<nn::dmnt::detail::IInterface> g_RefLoadingInterface;
        nn::sf::SharedPointer<nn::dmnt::detail::IInterface> g_RefDebugEventInterface;
        nn::sf::SharedPointer<nn::dmnt::detail::IInterface> g_RefInterfacePool[nn::dmnt::detail::PoolSessionCount];

        int g_NextInterface = 0;
        nn::sf::SharedPointer<nn::dmnt::detail::IInterface> GetNext()
        {
            if( g_NextInterface == nn::dmnt::detail::PoolSessionCount )
            {
                g_NextInterface = 0;
            }
            nn::sf::SharedPointer<nn::dmnt::detail::IInterface> Next = g_RefInterfacePool[g_NextInterface];
            g_NextInterface++;

            return Next;

        }

        bool g_IsInitialized = false;
    }

//==============================================================================

    Result Initialize() NN_NOEXCEPT
    {
        if( g_IsInitialized )
        {
            return ResultSuccess();
        }

        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);

        Result result;
        result = nn::sf::CreateHipcProxyByName<nn::dmnt::detail::IInterface, MyAllocator::Policy>(
            &g_RefLoadingInterface, nn::dmnt::detail::PortName);
        NN_SDK_ASSERT( result.IsSuccess() );

        result = nn::sf::CreateHipcProxyByName<nn::dmnt::detail::IInterface, MyAllocator::Policy>(
            &g_RefDebugEventInterface, nn::dmnt::detail::PortName);
        NN_SDK_ASSERT( result.IsSuccess() );

        int PoolSize = nn::dmnt::detail::PoolSessionCount;
        while( PoolSize-- )
        {
            result = nn::sf::CreateHipcProxyByName<nn::dmnt::detail::IInterface, MyAllocator::Policy>(
                &g_RefInterfacePool[PoolSize], nn::dmnt::detail::PortName);

            NN_SDK_ASSERT( result.IsSuccess() );
        }

        if( result.IsSuccess() )
        {
            g_IsInitialized = true;
        }

        return result;
    }

//==============================================================================

    Result Finalize() NN_NOEXCEPT
    {
        NN_ABORT("not implemented");
        return ResultSuccess();
    }

//==============================================================================

    Result BreakDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
    {
        Initialize();
        return GetNext()->BreakDebugProcess( debug );
    }

//==============================================================================

    Result TerminateDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
    {
        Initialize();
        return GetNext()->TerminateDebugProcess( debug );
    }

//==============================================================================

    Result CloseHandle(nn::svc::Handle debug) NN_NOEXCEPT
    {
        Initialize();
        return GetNext()->CloseHandle( debug );
    }

//==============================================================================

    Result LoadImage( nn::svc::Handle* pOut, const char* pFileName, const char* pArgs ) NN_NOEXCEPT
    {
        Initialize();

        nn::sf::InBuffer sfFileName( pFileName, strlen( pFileName ) + 1 );
        nn::sf::InBuffer sfArgs    ( pArgs    , strlen( pArgs     ) + 1 );

        uint32_t handleValue = 0;
        nn::Result result = g_RefLoadingInterface->LoadImage( nn::sf::Out<uint32_t>(&handleValue), sfFileName, sfArgs );
        *pOut = nn::svc::Handle(handleValue);
        return result;
    }

//==============================================================================

    Result AttachByProgramId( nn::svc::Handle* pOut, nn::Bit64* pPID, nn::Bit64 ProgramId ) NN_NOEXCEPT
    {
        Initialize();

        uint32_t handleValue = 0;
        nn::Result result = GetNext()->AttachByProgramId( nn::sf::Out<uint32_t>(&handleValue), nn::sf::Out<nn::Bit64>(pPID), ProgramId );
        *pOut = nn::svc::Handle(handleValue);

        return result;
    }

//==============================================================================

    Result AttachOnLaunch( nn::svc::Handle* pOut, nn::Bit64* pPID, nn::Bit64 ProgramId ) NN_NOEXCEPT
    {
        Initialize();

        uint32_t handleValue = 0;
        nn::Result result = g_RefLoadingInterface->AttachOnLaunch( nn::sf::Out<uint32_t>(&handleValue), nn::sf::Out<nn::Bit64>(pPID), ProgramId );
        *pOut = nn::svc::Handle(handleValue);

        return result;
    }

//==============================================================================

    Result GetDebugMonitorProcessId(nn::Bit64* pOut) NN_NOEXCEPT
    {
        Initialize();
        nn::Result result = GetNext()->GetDebugMonitorProcessId( nn::sf::Out<nn::Bit64>(pOut) );
        return result;
    }

//==============================================================================

    Result GetProcessId(nn::Bit64* pOut, nn::svc::Handle process) NN_NOEXCEPT
    {
        Initialize();
        nn::Result result = GetNext()->GetProcessId( nn::sf::Out<nn::Bit64>(pOut), process );
        return result;
    }

//==============================================================================

    Result GetProcessHandle(nn::svc::Handle* pOut, nn::Bit64 processId) NN_NOEXCEPT
    {
        Initialize();
        uint32_t handleValue = 0;
        nn::Result result = GetNext()->GetProcessHandle( nn::sf::Out<uint32_t>(&handleValue), processId );
        *pOut = nn::svc::Handle(handleValue);

        return result;
    }

//==============================================================================

    Result WaitSynchronization(nn::svc::Handle handle, int64_t ns) NN_NOEXCEPT
    {
        Initialize();
        nn::Result result = g_RefDebugEventInterface->WaitSynchronization( handle, (nn::Bit64)ns );
        return result;
    }

//==============================================================================

    Result GetDebugEvent(nn::svc::DebugEventInfo* pEvent, nn::svc::Handle process ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer sfEvent( (char*)pEvent, sizeof(nn::svc::DebugEventInfo) );
        nn::Result result = g_RefDebugEventInterface->GetDebugEvent( sfEvent, process );
        return result;
    }

//==============================================================================

    Result GetProcessModuleInfo(int* pOutCount, nn::dbg::ModuleInfo* pOutModules, int num, nn::os::ProcessId pid) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer sfModules( (char*)pOutModules, sizeof(nn::dbg::ModuleInfo) * num );
        nn::Result result = GetNext()->GetProcessModuleInfo( nn::sf::Out<int>(pOutCount), sfModules, num, pid );
        return result;
    }

//==============================================================================

    Result GetProcessList(int32_t* pNumProcesses, nn::Bit64 pProcessIds[], int32_t arraySize) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer sfProcessIds( (char*)pProcessIds, sizeof(nn::Bit64) * arraySize );
        nn::Result result = GetNext()->GetProcessList( nn::sf::Out<int32_t>(pNumProcesses), sfProcessIds );
        return result;
    }

//==============================================================================

    Result GetThreadList(int32_t* pNumThreads, nn::Bit64 pThreadIds[], int32_t arraySize, nn::svc::Handle domain) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer sfThreads( (char*)pThreadIds, sizeof(nn::Bit64) * arraySize );
        nn::Result result = GetNext()->GetThreadList( nn::sf::Out<int32_t>(pNumThreads), sfThreads, domain );
        return result;
    }

//==============================================================================

    Result GetDebugThreadContext(nn::svc::ThreadContext* pContext, nn::svc::Handle debug, nn::Bit64 threadId, nn::Bit32 controlFlags) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer sfContext( (char*)pContext, sizeof(nn::svc::ThreadContext) );
        nn::Result result = GetNext()->GetDebugThreadContext( sfContext, debug, threadId, controlFlags );
        return result;
    }

//==============================================================================

    Result GetAllDebugThreadInfo(nn::svc::Handle debug, void* pThreadData, nn::Bit32 ElementSize, nn::osdbg::ThreadInfo ThreadInfo[], nn::Bit32 ArraySize) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfThreadInfo( (char*)ThreadInfo, sizeof(nn::osdbg::ThreadInfo) * ArraySize );
        nn::sf::OutBuffer sfThreadData( (char*)pThreadData, ArraySize*ElementSize );
        nn::Result result = GetNext()->GetAllDebugThreadInfo( debug, sfThreadData, sfThreadInfo, ArraySize );
        return result;
    }

//==============================================================================

    Result ContinueDebugEvent(nn::svc::Handle debug, nn::Bit32 flags, nn::Bit64 threadIds[], nn::Bit32 size) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfThreadIds( (char*)threadIds, sizeof(nn::Bit64) * size );
        nn::Result result = GetNext()->ContinueDebugEvent( debug, flags, sfThreadIds, size );
        return result;
    }

//==============================================================================

    Result ReadDebugProcessMemory(uintptr_t buf, nn::svc::Handle debug, uintptr_t addr, size_t size) NN_NOEXCEPT
    {
        nn::sf::OutBuffer sfBuffer( (char*)buf, size );
        nn::Result result = GetNext()->ReadDebugProcessMemory( sfBuffer, debug, int64_t(addr), int64_t(size) );
        return result;
    }

//==============================================================================

    Result WriteDebugProcessMemory(nn::svc::Handle debug, uintptr_t buf, uintptr_t addr, size_t size) NN_NOEXCEPT
    {
        nn::sf::InBuffer sfBuffer( (char*)buf, size );
        nn::Result result = GetNext()->WriteDebugProcessMemory( debug, sfBuffer, int64_t(addr), int64_t(size) );
        return result;
    }

//==============================================================================

    Result SetDebugThreadContext(nn::svc::Handle debug, nn::Bit64 threadId, const nn::svc::ThreadContext& context, nn::Bit32 controlFlags) NN_NOEXCEPT
    {
        nn::sf::InBuffer sfContext( (char*)&context, sizeof(nn::svc::ThreadContext) );
        nn::Result result = GetNext()->SetDebugThreadContext( debug, threadId, sfContext, controlFlags );
        return result;
    }

//==============================================================================

    Result GetDebugThreadParam(nn::Bit64* pOut1, nn::Bit32* pOut2, nn::svc::Handle debug, nn::Bit64 threadId, nn::svc::DebugThreadParam select) NN_NOEXCEPT
    {
        nn::Result result = GetNext()->GetDebugThreadParam( nn::sf::Out<nn::Bit64>(pOut1), nn::sf::Out<nn::Bit32>(pOut2), debug, threadId, (uint32_t)select );
        return result;
    }

//==============================================================================

    Result InitializeThreadInfo( nn::osdbg::ThreadInfo* pThreadInfo, nn::svc::Handle debugHandle, const nn::svc::DebugInfoCreateProcess* pDebugInfoCreateProcess, const nn::svc::DebugInfoCreateThread* pDebugInfoCreateThread ) NN_NOEXCEPT
    {
        nn::sf::OutBuffer sfThreadInfo   ( (      char*)pThreadInfo            , sizeof(nn::osdbg::ThreadInfo          ) );
        nn::sf::InBuffer  sfCreateProcess( (const char*)pDebugInfoCreateProcess, sizeof(nn::svc::DebugInfoCreateProcess) );
        nn::sf::InBuffer  sfCreateThread ( (const char*)pDebugInfoCreateThread , sizeof(nn::svc::DebugInfoCreateThread ) );
        nn::Result result = GetNext()->InitializeThreadInfo( sfThreadInfo, debugHandle, sfCreateProcess, sfCreateThread );
        return result;
    }

//==============================================================================

    Result SetHardwareBreakPoint( nn::svc::HardwareBreakPointRegisterName regNo,  nn::Bit64 control, nn::Bit64 value ) NN_NOEXCEPT
    {
        Initialize();
        return GetNext()->SetHardwareBreakPoint( regNo, control, value );
    }

//==============================================================================

    Result QueryDebugProcessMemory(nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle debugHandle, uintptr_t addr) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer BlockInfo( (char*)pBlockInfo, sizeof(nn::svc::MemoryInfo) );
        nn::sf::OutBuffer PageInfo( (char*)pPageInfo, sizeof(nn::svc::PageInfo) );

        return GetNext()->QueryDebugProcessMemory( BlockInfo, PageInfo, debugHandle, int64_t(addr) );
    }

//==============================================================================

    Result GetProcessMemoryDetails(int32_t* pNumOfMemoryBlocks, nn::svc::MemoryInfo pBlockInfo[], int32_t arraySize, nn::svc::Handle debugHandle) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer sfData( (char*)pBlockInfo, sizeof(nn::Bit64) * arraySize );
        nn::Result result = GetNext()->GetProcessMemoryDetails( nn::sf::Out<int32_t>(pNumOfMemoryBlocks), sfData, debugHandle );
        return result;
    }

//==============================================================================

    Result GetJitDebugProcessList(int32_t* pNumProcesses, os::ProcessId pProcessIds[], int32_t arraySize) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::OutBuffer sfData( (char*)pProcessIds, sizeof(nn::Bit64) * arraySize );
        nn::Result result = GetNext()->GetJitDebugProcessList( nn::sf::Out<int32_t>(pNumProcesses), sfData );
        return result;
    }

//==============================================================================

    Result CreateCoreDump( char *pFileName, coredump::coredump_compiled_data* pDetails, nn::svc::Handle debugHandle ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfFileName( pFileName, strlen( pFileName ) + 1 );
        nn::sf::InBuffer sfDetails( (const char*)pDetails, sizeof(coredump::coredump_compiled_data ) );

        //===================================================================================
        // Because the module list and thread list are pointers, we'll use the IPC mechanism
        // so the DMNT side will have the correct pointers.
        //===================================================================================
        nn::sf::InBuffer sfModules( (const char*)pDetails->m_pModules, sizeof(coredump::coredump_compiled_data_module ) * pDetails->m_NumberOfModules );
        nn::sf::InBuffer sfThreads( (const char*)pDetails->m_pThreads, sizeof(coredump::coredump_compiled_data_thread) * pDetails->m_NumberOfThreads );

        nn::Result result = GetNext()->CreateCoreDump( sfFileName, sfDetails, sfModules, sfThreads, debugHandle );
        return result;
    }

//==============================================================================

    Result InitiateCoreDump( char *pFileName, coredump::coredump_compiled_data* pDetails, nn::svc::Handle debugHandle, uint64_t* pDumpHandle ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfFileName( pFileName, strlen( pFileName ) + 1 );
        nn::sf::InBuffer sfDetails( (const char*)pDetails, sizeof( coredump::coredump_compiled_data ) );

        //===================================================================================
        // Because the module list and thread list are pointers, we'll use the IPC mechanism
        // so the DMNT side will have the correct pointers.
        //===================================================================================
        nn::sf::InBuffer sfModules( (const char*)pDetails->m_pModules, sizeof(coredump::coredump_compiled_data_module ) * pDetails->m_NumberOfModules );
        nn::sf::InBuffer sfThreads( (const char*)pDetails->m_pThreads, sizeof(coredump::coredump_compiled_data_thread) * pDetails->m_NumberOfThreads );

        return GetNext()->InitiateCoreDump( sfFileName, sfDetails, sfModules, sfThreads, debugHandle, nn::sf::Out<uint64_t>(pDumpHandle) );
    }

//==============================================================================

    Result ContinueCoreDump( uint64_t DumpHandle, int32_t* pProgress ) NN_NOEXCEPT
    {
        Initialize();
        return GetNext()->ContinueCoreDump( DumpHandle, nn::sf::Out<int32_t>(pProgress) );
    }

//==============================================================================

    Result AddTTYToCoreDump( char* pData, uint32_t SizeOfData, uint64_t DumpHandle ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfData( pData, SizeOfData );
        return GetNext()->AddTTYToCoreDump( sfData, DumpHandle );
    }

//==============================================================================

    Result AddImageToCoreDump( char* pData, uint32_t SizeOfData, uint64_t DumpHandle ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfData( pData, SizeOfData );
        return GetNext()->AddImageToCoreDump( sfData, DumpHandle );
    }

//==============================================================================

    Result CloseCoreDump( uint64_t DumpHandle ) NN_NOEXCEPT
    {
        Initialize();
        return GetNext()->CloseCoreDump( DumpHandle );
    }

//==============================================================================
// TargetIO interface

//==============================================================================

    Result TargetIO_FileOpen( const char* pPath, int OpenMode, int32_t CreationDisposition, nn::fs::FileHandle* pHandle ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );
        nn::sf::OutBuffer sfHandle( (char*)pHandle, sizeof( nn::fs::FileHandle ) );

        return GetNext()->TargetIO_FileOpen( sfPath, (uint32_t)OpenMode, CreationDisposition, sfHandle );
    }

//==============================================================================

    Result TargetIO_FileClose( nn::fs::FileHandle* pHandle ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfHandle( (char*)pHandle, sizeof( nn::fs::FileHandle ) );
        return GetNext()->TargetIO_FileClose( sfHandle );
    }

//==============================================================================

    Result TargetIO_FileRead( nn::fs::FileHandle* pHandle, void* pBuffer, int32_t NumberOfBytesToRead, int32_t* pNumberOfBytesRead, int64_t Offset ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfHandle( (char*)pHandle, sizeof( nn::fs::FileHandle ) );
        nn::sf::OutBuffer sfBuffer( (char*)pBuffer, NumberOfBytesToRead );

        return GetNext()->TargetIO_FileRead( sfHandle, sfBuffer, nn::sf::Out<int32_t>(pNumberOfBytesRead), Offset );
    }

//==============================================================================

    Result TargetIO_FileWrite( nn::fs::FileHandle* pHandle, void* pBuffer, int32_t NumberOfBytesToWrite, int32_t* pNumberOfBytesWritten, int64_t Offset ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfHandle( (char*)pHandle, sizeof( nn::fs::FileHandle ) );
        nn::sf::InBuffer sfBuffer( (char*)pBuffer, NumberOfBytesToWrite );

        return GetNext()->TargetIO_FileWrite( sfHandle, sfBuffer, nn::sf::Out<int32_t>(pNumberOfBytesWritten), Offset );
    }

//==============================================================================

    Result TargetIO_FileSetAttributes( const char* pPath, void* pAttribs, int32_t SizeOfAttributes ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );
        nn::sf::InBuffer sfBuffer( (char*)pAttribs, SizeOfAttributes );
        return GetNext()->TargetIO_FileSetAttributes( sfPath, sfBuffer );
    }

//==============================================================================

    Result TargetIO_FileGetInformation( const char* pPath, int32_t* pIsDirectory, uint64_t* pSize, uint64_t* pCreateTime, uint64_t* pAccessTime, uint64_t* pModifyTime ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );
        uint64_t pInfo[4];
        nn::sf::OutBuffer sfInfo( (char*)pInfo, sizeof(pInfo) );

        nn::Result result = GetNext()->TargetIO_FileGetInformation( sfPath, nn::sf::Out<int32_t>(pIsDirectory), sfInfo );
        if( result.IsSuccess() )
        {
            *pSize = pInfo[0];
            *pCreateTime = pInfo[1];
            *pAccessTime = pInfo[2];
            *pModifyTime = pInfo[3];
        }
        return result;
    }

//==============================================================================

    Result TargetIO_FileSetTime( const char* pPath, uint64_t CreateTime, uint64_t AccessTime, uint64_t ModifyTime ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );
        return GetNext()->TargetIO_FileSetTime( sfPath, CreateTime, AccessTime, ModifyTime );
    }

//==============================================================================

    // Basic file operations
    Result TargetIO_FileSetSize( const char* pPath, int64_t size ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );

        nn::Result result = GetNext()->TargetIO_FileSetSize( sfPath, size );
        return result;
    }

//==============================================================================

    Result TargetIO_FileDelete( const char* pPath ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );

        nn::Result result = GetNext()->TargetIO_FileDelete( sfPath );
        return result;
    }

//==============================================================================

    Result TargetIO_FileMove( const char* pFromName, const char* pToName ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfFromPath( pFromName, strlen( pFromName ) + 1 );
        nn::sf::InBuffer sfToPath( pToName, strlen( pToName ) + 1 );

        return GetNext()->TargetIO_FileMove( sfFromPath, sfToPath );
    }

//==============================================================================

    Result TargetIO_DirectoryCreate( const char* pPath, nn::fs::OpenMode access ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );

        nn::Result result = GetNext()->TargetIO_DirectoryCreate( sfPath );
        return result;
    }

//==============================================================================

    Result TargetIO_DirectoryDelete( const char* pPath ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );

        return GetNext()->TargetIO_DirectoryDelete( sfPath );
    }

//==============================================================================

    Result TargetIO_DirectoryRename( const char* pFromName, const char* pToName ) NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfFromPath( pFromName, strlen( pFromName ) + 1 );
        nn::sf::InBuffer sfToPath( pToName, strlen( pToName ) + 1 );

        return GetNext()->TargetIO_DirectoryRename( sfFromPath, sfToPath );
    }

//==============================================================================

    Result  TargetIO_DirectoryGetCount( const char* pPath, int32_t* pNumberOfEntries )  NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );
        return GetNext()->TargetIO_DirectoryGetCount( sfPath, nn::sf::Out<int32_t>(pNumberOfEntries) );
    }

//==============================================================================

    Result  TargetIO_DirectoryOpen( const char* pPath, nn::fs::DirectoryHandle* pDirectory )  NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfPath( pPath, strlen( pPath ) + 1 );
        nn::sf::OutBuffer sfHandle( (char*)pDirectory, sizeof( nn::fs::DirectoryHandle ) );
        return GetNext()->TargetIO_DirectoryOpen( sfPath, sfHandle );
    }

//==============================================================================

    Result  TargetIO_DirectoryGetNext( nn::fs::DirectoryHandle* pDirectory, nn::fs::DirectoryEntry* pEntry )  NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfHandle( (char*)pDirectory, sizeof( nn::fs::DirectoryHandle ) );
        nn::sf::OutBuffer sfEntry( (char*)pEntry, sizeof( nn::fs::DirectoryEntry ) );
        return GetNext()->TargetIO_DirectoryGetNext( sfHandle, sfEntry );
    }

//==============================================================================

    Result  TargetIO_DirectoryClose( nn::fs::DirectoryHandle* pDirectory )  NN_NOEXCEPT
    {
        Initialize();
        nn::sf::InBuffer sfHandle( (char*)pDirectory, sizeof( nn::fs::DirectoryHandle ) );
        return GetNext()->TargetIO_DirectoryClose( sfHandle );
    }

//==============================================================================

    Result TargetIO_GetFreeSpace( int64_t* pFreeBytesAvailable, int64_t* pTotalNumberOfBytes, int64_t* pTotalNumberOfFreeBytes )
    {
        Initialize();
        int64_t pFreeSpace[3];
        nn::sf::OutBuffer sfFreeSpace( (char*)pFreeSpace, sizeof(pFreeSpace) );
        nn::Result result = GetNext()->TargetIO_GetFreeSpace( sfFreeSpace );
        if( result.IsSuccess() )
        {
            *pFreeBytesAvailable = pFreeSpace[0];
            *pTotalNumberOfBytes = pFreeSpace[1];
            *pTotalNumberOfFreeBytes = pFreeSpace[2];
        }
        return result;
    }

//==============================================================================

    Result TargetIO_GetVolumeInformation( char* pVolumeNameBuffer, int32_t VolumeNameSize, char* pFileSystemNameBuffer, int32_t FileSystemNameSize,
        int32_t* pVolumeSerialNumber, int32_t* pMaximumComponentLength, int32_t* pFileSystemFlags )
    {
        Initialize();
        nn::sf::OutBuffer sfVolumeName( (char*)pVolumeNameBuffer, VolumeNameSize );
        nn::sf::OutBuffer sfFileSystemName( (char*)pFileSystemNameBuffer, FileSystemNameSize );

        return GetNext()->TargetIO_GetVolumeInformation( sfVolumeName, sfFileSystemName, nn::sf::Out<int32_t>(pVolumeSerialNumber),
            nn::sf::Out<int32_t>(pMaximumComponentLength), nn::sf::Out<int32_t>(pFileSystemFlags) );
    }

//==============================================================================

}}
