﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/socket/socket_Api.h>
#include <nn/util/util_FormatString.h>
#include <cstring>
#include <cctype>
#include <algorithm>
#include "dhcps_Log.h"

extern "C"
{                   // adding nolint because C++ versions do not exist
                    // in rynda and this causes a compilation error
#include <stdio.h>  // NOLINT(others/c_header)
#include <stdlib.h> // NOLINT(others/c_header)
}

namespace nn { namespace dhcps { namespace detail {

const char* g_LogLevelTable[] = {
    "Major",
    "Minor",
    "Info",
    "Debug",
    "Hex"
};

const unsigned int HexCharactersPerByte = 3;
const unsigned int NewlinePerLine = 1;
const unsigned int BytesPerLine = 16;
const unsigned int NewlineAndNull = 2;

static
const unsigned int GetHexStringBufferSize(unsigned bufferSize) NN_NOEXCEPT
{
#define LINES(size)  (                                          \
        (size / BytesPerLine) == 0 ?                            \
        1 :                                                     \
        ((size / BytesPerLine) + (size % BytesPerLine)))

#define GET_HEXDUMP_BUFFER_SIZE(size) (                 \
        (HexCharactersPerByte * size)           +       \
        (NewlinePerLine * (LINES(size)))        +       \
        NewlineAndNull                                  \
        )
    size_t sz = GET_HEXDUMP_BUFFER_SIZE(bufferSize);
    return sz;
#undef GET_HEXDUMP_BUFFER_SIZE
#undef LINES
}

static
void HexFill(char* pOutString,
             unsigned int strSize,
             const uint8_t* pBytes,
             unsigned int byteSize)  NN_NOEXCEPT
{
    char* pHex = pOutString;

    for (unsigned int idx=0; idx<byteSize; ++idx)
    {
        pHex += nn::util::SNPrintf(pHex,
                                   strSize - (pHex - pOutString),
                                   "%02x ", *pBytes);
        pBytes += 1;

        if (idx == (byteSize - 1))
        {
            *pHex = '\0';
            ++pHex;
        };
    };
};

const char* GetFilename(const char* line) NN_NOEXCEPT
{
    unsigned int length = 0;
    if (nullptr == line)
    {
        goto bail;
    };

    length = strlen(line);
    for (unsigned int idx=length; idx!=0; --idx)
    {
        if (line[idx] == '/' || line[idx] == '\\')
        {
            return &line[idx + 1];
        };
    };

bail:
    return line;
};

const char* GetLevelString(unsigned int level) NN_NOEXCEPT
{
    if (level > NN_DETAIL_DHCPS_LOG_LEVEL_MAX)
    {
        return "undefined";
    }
    return g_LogLevelTable[level];
};

void DhcpServerLog(const void* hexbuffer,
                   unsigned int hexsize,
                   const char* module,
                   unsigned int line,
                   const char* function,
                   unsigned int level,
                   const char* format, ...) NN_NOEXCEPT
{
    va_list args;
    va_start (args, format);
    int hexSize = nullptr == hexbuffer ? 0 : GetHexStringBufferSize(hexsize);

    int stringSize = vsnprintf(nullptr, 0, format, args);
    char* pString = static_cast<char*>(alloca(stringSize + hexSize));
    memset(pString, 0, stringSize + hexSize);
    vsnprintf(pString, stringSize, format, args);
    HexFill(pString + strlen(pString), hexSize, reinterpret_cast<const uint8_t*>(hexbuffer), hexsize);
    NN_SDK_LOG("%s %u %s %llu %s\n", module, line, function, os::GetThreadId(os::GetCurrentThread()), pString);
    va_end(args);
};

}}}; // nn::dhcps::detail
