﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_Common.h"
#include "dhcps_DhcpManager.h"
#include "dhcps_Coordinator.h"

//#define NN_DETAIL_DHCPS_LOG_LEVEL NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "DhcpManager"
#include "dhcps_Log.h"

namespace nn { namespace dhcps { namespace detail {

extern Config g_Config;


/**
 * ---------------- Anonymous namespace utils ------------------
 */

const char* DhcpManagerStateToString(DhcpManager::State in) NN_NOEXCEPT
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::Uninitialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::Initialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::Ready);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::OnDiscover);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::OnDecline);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::OnRequest);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::OnInform);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::OnRelease);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::OnUnhandled);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(DhcpManager::State::Error);
    default:
        ;
    };
    return "Unknown DhcpManager::State";
};

/**
 * @brief return the ip address in host byte format
 */
void GetClientIdentityFromBootp(DhcpClientIdentity* pOutIdentity, const BootpDhcpMessage* pRequest, size_t requestSize) NN_NOEXCEPT
{
    if (nullptr == pOutIdentity)
    {
        NN_SDK_ASSERT(false);
        NN_DETAIL_DHCPS_LOG_MAJOR("pOutIdentity is null\n");
        return;
    };

    pOutIdentity->hash = ClientIdHashFromMessage(pRequest, requestSize);

    pOutIdentity->ipAddress = pRequest->clientIpAddress;

    // copies the option value if it is available
    CopyOptionValue(&pOutIdentity->ipAddress,
                    sizeof(pOutIdentity->ipAddress),
                    pRequest->options,
                    requestSize - sizeof(*pRequest),
                    DhcpOptionCode::RequestedAddress);

    std::memcpy(pOutIdentity->macAddress, pRequest->clientHardwareAddress, sizeof(pOutIdentity->macAddress));
};

int GetRecordByHashOrMac(LeaseRecord *& pOutRecord, const DhcpClientIdentity& origin, LeaseTableManager* pLeaseTableManager) NN_NOEXCEPT
{
#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char ipString1[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char macAddressString[static_cast<uint32_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    NN_DETAIL_DHCPS_LOG_DEBUG("Performing lookup by %s for Hash: %llx, IP: %s, Mac: %s.\n",
                              0 == origin.hash ? "EthernetMacAddress" : "ClientIdentifierHash",
                              origin.hash,
                              nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, ipString1, sizeof(ipString1)),
                              MacAddressToString(macAddressString, sizeof(macAddressString), origin.macAddress));

    if (0 == origin.hash)
    {
        return pLeaseTableManager->GetRecordByEthernetMacAddress(pOutRecord, origin.macAddress);
    };
    return pLeaseTableManager->GetRecordByClientIdentifierHash(pOutRecord, origin.hash);
}

/**
 * -------------------- DHCP Manager Section ------------------------
 */

DhcpManager::DhcpManager()  NN_NOEXCEPT :
m_State(State::Uninitialized)
{
};

DhcpManager::~DhcpManager() NN_NOEXCEPT
{
};

int DhcpManager::Initialize(Coordinator* pCoordinator, LeaseTableManager* pLeaseTableManager) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pCoordinator: %p, pLeaseTableManager: %p\n",
                              pCoordinator, pLeaseTableManager);

    int rc = -1;

    if ( nullptr == (m_pCoordinator = pCoordinator))
    {
        ChangeState(State::Error);
        goto bail;
    };

    if (nullptr == (m_pLeaseTableManager = pLeaseTableManager))
    {
        ChangeState(State::Error);
        goto bail;
    };

    ChangeState(State::Initialized);
    rc = 0;

bail:
    return rc;
};

void DhcpManager::ChangeState(State next) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("Changing state from: %s (%u) to %s (%u)\n",
                              DhcpManagerStateToString(m_State), m_State,
                              DhcpManagerStateToString(next), next);

    SetGlobalState(GlobalState::DhcpManager, static_cast<uint8_t>(next));

    m_State = next;
};

int DhcpManager::Finalize() NN_NOEXCEPT
{
    ChangeState(State::Uninitialized);

    m_pCoordinator = nullptr;

    ChangeState(State::Uninitialized);
    return 0;
};

void DhcpManager::OnEvent(const InternalEvent& e) NN_NOEXCEPT
{
    switch (e.GetType())
    {
    case EventType::OnPacketRead:
        OnNewMessage(reinterpret_cast<const NetworkLayersContainer*>(e.GetValue()), e.GetSize());
        break;
    case EventType::OnTimerExpired:
        OnTimerExpired();
        break;
    default:
        ;
    };

    return;
};

void DhcpManager::OnTimerExpired() NN_NOEXCEPT
{
};

void DhcpManager::OnNewMessage(const NetworkLayersContainer pPacketIn[], size_t stackInCount) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pPacketIn: %p, stackInCount: %zu\n", pPacketIn, stackInCount);

#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char ipString1[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char macAddressString[static_cast<uint32_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    size_t rc = -1;
    NetworkLayersContainer containerIn;
    BootpDhcpMessage* pRequest = nullptr;

    const size_t stackOutCount = stackInCount;
    NetworkLayersContainer pPacketOut[static_cast<size_t>(NetworkLayer::Max)];
    uint8_t pOutBuffer[static_cast<uint32_t>(LibraryConstants::DhcpMaxMtu)] = { 0 };
    BootpDhcpMessage* pResponse = nullptr;
    size_t responseSize = 0;
    DhcpClientIdentity origin;
    DhcpMessageType messageType;

    if (-1 == NetworkLayersGetLayer(&containerIn,
                                    NetworkLayer::BootProtocol,
                                    pPacketIn, stackInCount))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (nullptr == (pRequest = reinterpret_cast<BootpDhcpMessage*>(containerIn.pBuffer)))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (-1 == (rc = CopyOptionValue(&messageType,
                                         sizeof(messageType),
                                         pRequest->options,
                                         containerIn.size - sizeof(BootpDhcpMessage),
                                         DhcpOptionCode::DhcpMessageType)))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (-1 == (rc = NetworkLayersCopy(pPacketOut,
                                           sizeof(pPacketOut) / sizeof(NetworkLayersContainer),
                                           pOutBuffer,
                                           sizeof(pOutBuffer),
                                           pPacketIn,
                                           stackInCount)))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (nullptr == (pResponse = reinterpret_cast<BootpDhcpMessage*>(
                             pPacketOut[static_cast<uint32_t>(NetworkLayer::BootProtocol)].pBuffer)))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    };

    GetClientIdentityFromBootp(&origin, pRequest, containerIn.size);
    responseSize = sizeof(pOutBuffer);

    for (unsigned idx = 0; idx != static_cast<uint32_t>(NetworkLayer::BootProtocol); ++idx)
    {
        responseSize -= pPacketOut[idx].size;
    };

    NN_DETAIL_DHCPS_LOG_DEBUG("transactionId: %04x, message: %s, Hash: %llx, IP: %s, Mac: %s.\n",
                              nn::socket::InetNtohl(pRequest->transactionId),
                              DhcpMessageTypeToString(messageType),
                              origin.hash,
                              nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, ipString1, sizeof(ipString1)),
                              MacAddressToString(macAddressString, sizeof(macAddressString), origin.macAddress));

    switch (messageType)
    {
    case DhcpMessageType::Discover:
        if (-1 != (rc = OnDhcpDiscover(pResponse, responseSize, origin, pRequest, containerIn.size)))
        {
            pPacketOut[static_cast<size_t>(NetworkLayer::BootProtocol)].size = rc;
            SendDhcpResponse(pPacketOut, stackOutCount);
        };
        break;

    case DhcpMessageType::Decline:
        rc = OnDhcpDecline(pResponse, responseSize, origin, pRequest, containerIn.size);
        break;

    case DhcpMessageType::Request:
        if (-1 != (rc = OnDhcpRequest(pResponse, responseSize, origin, pRequest, containerIn.size)))
        {
            pPacketOut[static_cast<size_t>(NetworkLayer::BootProtocol)].size = rc;
            SendDhcpResponse(pPacketOut, stackOutCount);
        };
        break;

    case DhcpMessageType::Inform:
        if (-1 != (rc = OnDhcpInform(pResponse, responseSize, origin, pRequest, containerIn.size)))
        {
            pPacketOut[static_cast<size_t>(NetworkLayer::BootProtocol)].size = rc;
            SendDhcpResponse(pPacketOut, stackOutCount);
        };
        break;

    case DhcpMessageType::Release:
        rc = OnDhcpRelease(pResponse, responseSize, origin, pRequest, containerIn.size);
        break;

    default:
    {
        ChangeState(State::OnUnhandled);
        InternalEvent e(static_cast<EventType>(Event::OnDhcpUnhandledMessage),
                        reinterpret_cast<const void*>(&origin),
                        sizeof(origin));
        m_pCoordinator->OnEvent(e);
        break;
    };
    };

bail:
    ChangeState(State::Ready);
    return;
}; //NOLINT(impl/function_size)

void DhcpManager::GetTimeout(nn::socket::TimeVal* pTv)
{
    if (pTv == nullptr)
    {
        NN_SDK_ASSERT(false);
        return;
    };

    static const unsigned int Timeout = 1;
    pTv->tv_sec = Timeout;
    pTv->tv_usec = 0;
};

ssize_t DhcpManager::OnDhcpDiscover(BootpDhcpMessage* pOutResponse,
                                    size_t outResponseSize,
                                    const DhcpClientIdentity& origin,
                                    const BootpDhcpMessage* pRequest,
                                    size_t requestSize) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, outResponseSize: %zu, "
                              "pRequest: %p, requestSize: %zu\n",
                              pOutResponse, outResponseSize, pRequest, requestSize);

    ChangeState(State::OnDiscover);

    ssize_t rc = -1;
    LeaseRecord* pRecord  = nullptr;
    Event action;
    uint32_t now;

    const void* eventPointer = reinterpret_cast<const void*>(&origin);
    size_t eventPointerSize = sizeof(origin);
    DhcpLease l = {};

#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char ipString1[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char macAddressString[static_cast<uint32_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    if (-1 != GetRecordByHashOrMac(pRecord, origin, m_pLeaseTableManager))
    {
        // When a lease is released to the table or if it expires then
        // the DhcpClient is zeroed out and lease state
        // set to State::Unbound and a zeroed out client is not a
        // valid originating client address so the only two cases
        // where we will get a valid offer will be 1) if the state is
        // already set to Offered (i.e. when a client tries to broker
        // a deal for a better lease) or if the state is in some other
        // advanced state; this second case is unusual and something
        // worth keeping an eye out for because it might imply that a
        // certain type of client has weird timer behavior.
        if (pRecord->state == ClientState::Offered)
        {
            action = Event::OnLeaseOfferResent;
        }
        else
        {
            action = Event::OnLeaseOfferAdvancedState;
        };
        // event pointer and size set below
    }
    // the client requested an ip address, so if all these conditions are true then we
    // give it to them
    else if (0 != origin.ipAddress.S_addr &&
             -1 != m_pLeaseTableManager->GetRecordByIpAddress(pRecord, origin.ipAddress) &&
             nullptr != pRecord &&
             pRecord->state == ClientState::Unbound)
    {
        action = Event::OnLeaseOfferReused;
    }
    // the DHCP client is not in the table so go get a new record
    // this is the common case.
    else if (-1 != m_pLeaseTableManager->AllocateRecord(pRecord))
    {
        action = Event::OnLeaseOffered;
        // event pointer and size set below
    }
    // In this case server resources are exhausted and there is
    // literally nothing the server can do so the best thing is to
    // remain silent and not respond.
    else
    {
        // event pointer & size already set
        NN_DETAIL_DHCPS_LOG_MAJOR("Unable to reuse or allocate a record.\n");
        action = Event::OnDiscoverWhileTableFull;
        // event pointer and size already set
        goto bail;
    };


    now = Time();
    pRecord->client.hash = origin.hash;
    std::memcpy(pRecord->client.macAddress, origin.macAddress, sizeof(pRecord->client.macAddress));
    pRecord->lease = now + g_Config.GetOfferTimerSeconds();
    pRecord->state = ClientState::Offered;
    pRecord->priority += 1;

    if (-1 != (rc = CreateCannedResponse(pOutResponse,
                                         outResponseSize,
                                         pRequest,
                                         DhcpMessageType::Offer,
                                         pRecord)))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("%s for Hash: %llx, IP: %s, Mac: %s\n",
                                  EventToString(action),
                                  origin.hash,
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &pRecord->client.ipAddress, ipString1, sizeof(ipString1)),
                                  MacAddressToString(macAddressString, sizeof(macAddressString), origin.macAddress));
    };

    LeaseRecordToDhcpLease(&l, *pRecord);
    eventPointer = reinterpret_cast<void*>(&l);
    eventPointerSize = sizeof(l);

bail:

    InternalEvent e(static_cast<EventType>(action), eventPointer, eventPointerSize);
    m_pCoordinator->OnEvent(e);

    return rc;
};

ssize_t DhcpManager::OnDhcpDecline(BootpDhcpMessage* pOutResponse,
                                   size_t outResponseSize,
                                   const DhcpClientIdentity& origin,
                                   const BootpDhcpMessage* pRequest,
                                   size_t requestSize) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, outResponseSize: %zu, "
                              "pRequest: %p, requestSize: %zu\n",
                              pOutResponse, outResponseSize, pRequest, requestSize);

    ChangeState(State::OnDecline);

    ssize_t rc = -1;
    LeaseRecord *pRecord = nullptr;

    Event action;
    const void* eventPointer = reinterpret_cast<const void*>(&origin);
    size_t eventPointerSize = sizeof(origin);
    DhcpLease l;

#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char ipString1[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char macAddressString[static_cast<uint32_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    // According to RFC 2131 a Decline message is a message from a
    // client to a server that indicates the address is already used.
    // This means that, from the client's perspective, the address
    // is already in use.
    if (-1 == GetRecordByHashOrMac(pRecord, origin, m_pLeaseTableManager))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("Hash: %llx, IP: %s, Mac: %s sent decline but no associated lease.\n",
                                  origin.hash,
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, ipString1, sizeof(ipString1)),
                                  MacAddressToString(macAddressString, sizeof(macAddressString), origin.macAddress));

        action = Event::OnLeaseDeclineInvalid;
    }
    // The client that sent the DHCP Decline has a valid record in the
    // lease table. This implies that the client has rejected the
    // address and that it should be released back to the lease table
    // for future use.
    else
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("Hash: %llx, IP: %s, Mac: %s sent decline, setting lease to Unbound.\n",
                                  origin.hash,
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, ipString1, sizeof(ipString1)),
                                  MacAddressToString(macAddressString, sizeof(macAddressString), origin.macAddress));

        action = Event::OnLeaseDecline;
        pRecord->priority += 1;

        LeaseRecordToDhcpLease(&l, *pRecord);
        eventPointer = reinterpret_cast<void*>(&l);
        eventPointerSize = sizeof(l);

        ZeroRecord(pRecord);
    };

    InternalEvent e(static_cast<EventType>(action), eventPointer, eventPointerSize);
    m_pCoordinator->OnEvent(e);

    return rc;
};

ssize_t DhcpManager::OnDhcpRequest(BootpDhcpMessage* pOutResponse,
                                   size_t outResponseSize,
                                   const DhcpClientIdentity& origin,
                                   const BootpDhcpMessage* pRequest,
                                   size_t requestSize) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, outResponseSize: %zu, "
                              "pRequest: %p, requestSize: %zu\n",
                              pOutResponse, outResponseSize, pRequest, requestSize);

    ChangeState(State::OnRequest);

    ssize_t rc = -1;
    LeaseRecord *pRecord = nullptr;
    uint32_t now;

    Event action = Event::OnLeaseRequestInvalid;
    const void* eventPointer = reinterpret_cast<const void*>(&origin);
    size_t eventPointerSize = sizeof(origin);
    DhcpLease l;

#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char ipString1[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char macAddressString1[static_cast<size_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
    char macAddressString2[static_cast<size_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    // DHCP Request is a bit different than the other messages:
    //
    // The normal flow is for a client to connect to a network,
    // send the DHCP Discover, receive an Offer, then Request the
    // lease, and get a Acknowledgment containing the lease values.
    //
    // However this is not the only flow for dhcp requests for when
    // a client connects to a network it first detects whether or not
    // it is a network that it was previously connected to and if so
    // then it attempts to skip the Discover / Offer phase and performs
    // a "fast request" in an attempt to secure a prior address.
    //
    // So, in our case we first try to see whether or not the client
    // exists in the lease table but if not then we check the lease
    // table by IP. If the lease is available (i.e. it is in the
    // Unbound state) then we allow that IP to be reused.
    //

    // handle "fast request" case where we don't have the client in
    // the lease table but we do have the IP address.
    if (-1 != m_pLeaseTableManager->GetRecordByIpAddress(pRecord, origin.ipAddress))
    {
        // If the client is a valid client originally created from
        // some message then we need to compare that the client in the
        // request is equivalent with the client in the LeaseRecord
        // for we don't want one client stealing another client's lease
        if (origin.hash != pRecord->client.hash)
        {
            NN_DETAIL_DHCPS_LOG_DEBUG(
                "Hash: %llx/Mac: %s requested ip %s but Hash: %llx/Mac: %s already owns it.\n",
                origin.hash,
                MacAddressToString(macAddressString1, sizeof(macAddressString1), origin.macAddress),
                nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, ipString1, sizeof(ipString1)),
                pRecord->client.hash,
                MacAddressToString(macAddressString2, sizeof(macAddressString1), pRecord->client.macAddress));

            rc = CreateNakResponse(pOutResponse, outResponseSize, pRequest);
            // action, event pointer, and size already set
            goto bail;
        };
        // continue to request code
    }
    else if (-1 != GetRecordByHashOrMac(pRecord, origin, m_pLeaseTableManager))
    {
        // don't dole out more than 1 address per DHCP client id hash
        if (origin.ipAddress.S_addr != pRecord->client.ipAddress.S_addr)
        {
            NN_DETAIL_DHCPS_LOG_DEBUG(
                "Hash: %llx/Mac: %s requested ip %s but Hash:  %llx/Mac: %s already owns it.\n",
                origin.hash,
                MacAddressToString(macAddressString1, sizeof(macAddressString1), origin.macAddress),
                nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, ipString1, sizeof(ipString1)),
                pRecord->client.hash,
                MacAddressToString(macAddressString2, sizeof(macAddressString2), pRecord->client.macAddress));

            rc = CreateNakResponse(pOutResponse, outResponseSize, pRequest);
            // action, event pointer, and size already set
            goto bail;
        };
        // continue to request code
    }
    else
    {
        NN_DETAIL_DHCPS_LOG_DEBUG(
            "Hash: %llx/Mac: %s requested unavailable IP %s.\n",
            origin.hash,
            MacAddressToString(macAddressString2, sizeof(macAddressString2), pRecord->client.macAddress),
            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &pRecord->client.ipAddress, ipString1, sizeof(ipString1)));

        rc = CreateNakResponse(pOutResponse, outResponseSize, pRequest);
        goto bail;
    };

    action = Event::OnLeaseRequest;
    now = Time();

    pRecord->client.hash = origin.hash;
    std::memcpy(pRecord->client.macAddress, origin.macAddress, sizeof(pRecord->client.macAddress));
    pRecord->lease = now + g_Config.GetLeaseTimerSeconds();
    pRecord->t1 = now + (g_Config.GetLeaseTimerSeconds() * g_Config.GetT1Ratio());
    pRecord->t2 = now + (g_Config.GetLeaseTimerSeconds() * g_Config.GetT2Ratio());
    pRecord->state = ClientState::Bound;
    pRecord->priority += 1;

    LeaseRecordToDhcpLease(&l, *pRecord);
    eventPointer = reinterpret_cast<void*>(&l);
    eventPointerSize = sizeof(l);

    rc = CreateCannedResponse(pOutResponse,
                              outResponseSize,
                              pRequest,
                              DhcpMessageType::Ack,
                              pRecord);
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("Renewing lease for Hash: %llx, IP: %s, Mac: %s to "
                                  "lease: %d, t1: %d, t2: %d\n",
                                  origin.hash,
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, ipString1, sizeof(ipString1)),
                                  MacAddressToString(macAddressString1, sizeof(macAddressString1), pRecord->client.macAddress),
                                  pRecord->lease,
                                  pRecord->t1,
                                  pRecord->t2);
    };

bail:
    InternalEvent e(static_cast<EventType>(action), eventPointer, eventPointerSize);
    m_pCoordinator->OnEvent(e);

    return rc;
};

ssize_t DhcpManager::OnDhcpInform(BootpDhcpMessage* pOutResponse,
                                  size_t outResponseSize,
                                  const DhcpClientIdentity& origin,
                                  const BootpDhcpMessage* pRequest,
                                  size_t requestSize) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, outResponseSize: %zu, "
                              "pRequest: %p, requestSize: %zu\n",
                              pOutResponse, outResponseSize, pRequest, requestSize);

    ChangeState(State::OnInform);

    ssize_t rc = -1;
    LeaseRecord* pRecord = nullptr;
    Event action;

    const void* eventPointer = reinterpret_cast<const void*>(&origin);
    size_t eventPointerSize = sizeof(origin);
    DhcpLease l;

#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char tmp1[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char tmp2[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char macAddressString[static_cast<size_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    // The DHCP Inform message is similar to the Request message but
    // it does nothing to the lease. It is one way that a client
    // might determine clock drift between client and server.
    // Unlike DHCP request there is no "fast inform": an inform message
    // is only sent when the client thinks there is a valid lease.

    if (-1 == GetRecordByHashOrMac(pRecord, origin, m_pLeaseTableManager))
    {
        // Inform for client that isn't in the lease table.
        NN_DETAIL_DHCPS_LOG_DEBUG("Hash: %llx, IP: %s, Mac: %s unable to get lease record.\n",
                                  origin.hash,
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, tmp1, sizeof(tmp1)),
                                  MacAddressToString(macAddressString, sizeof(macAddressString), origin.macAddress));

        action = Event::OnLeaseInformInvalid;
        // event pointer & size already set
    }
    else if (origin.ipAddress.S_addr != pRecord->client.ipAddress.S_addr)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("Dhcp Inform from Hash: %llx, IP: %s, Mac: %s. "
                                  "but expected IP: %s instead.\n",
                                  origin.hash,
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &origin.ipAddress, tmp1, sizeof(tmp1)),
                                  MacAddressToString(macAddressString, sizeof(macAddressString), origin.macAddress),
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &pRecord->client.ipAddress, tmp2, sizeof(tmp2)));

        // You would think that now is a good time to send a NAK
        // response but in fact when you do that Windows goes sort of
        // crazy and starts sending lots of DHCP Info messages. So the
        // only thing we can do is signal the application of the
        // condition and move on.

        action = Event::OnLeaseInformInvalid;
        // event pointer & size already set
    }
    else
    {
        if ( -1 != (rc = CreateCannedResponse(pOutResponse,
                                              outResponseSize,
                                              pRequest,
                                              DhcpMessageType::Ack,
                                              pRecord)))
        {
            NN_DETAIL_DHCPS_LOG_DEBUG(
                "Dhcp Inform from Hash: %llx, IP: %s, Mac: %s. Sending Ack.\n",
                origin.hash,
                nn::socket::InetNtop(nn::socket::Family::Af_Inet, &pRecord->client.ipAddress, tmp1, sizeof(tmp1)),
                MacAddressToString(macAddressString, sizeof(macAddressString), pRecord->client.macAddress));
        };

        action = Event::OnLeaseInform;

        LeaseRecordToDhcpLease(&l, *pRecord);
        eventPointer = reinterpret_cast<void*>(&l);
        eventPointerSize = sizeof(l);
    };

    InternalEvent e(static_cast<EventType>(action), eventPointer, eventPointerSize);
    m_pCoordinator->OnEvent(e);

    return rc;
};

ssize_t DhcpManager::OnDhcpRelease(BootpDhcpMessage* pOutResponse,
                                   size_t outResponseSize,
                                   const DhcpClientIdentity& origin,
                                   const BootpDhcpMessage* pRequest,
                                   size_t requestSize) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, outResponseSize: %zu, "
                              "pRequest: %p, requestSize: %zu\n",
                              pOutResponse, outResponseSize,
                              pRequest, requestSize);

    ChangeState(State::OnRelease);

    ssize_t rc = -1;
    LeaseRecord *pRecord = nullptr;

    Event action;
    const void* eventPointer = reinterpret_cast<const void*>(&origin);
    size_t eventPointerSize = sizeof(origin);
    DhcpLease l;

#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char tmp1[static_cast<uint32_t>(LibraryConstants::IpAddressStringBufferSize)] = { 0 };
    char macAddressString[static_cast<uint32_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    if (-1 == GetRecordByHashOrMac(pRecord, origin, m_pLeaseTableManager))
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("Cannot find lease record for Hash: %llx in table, unable to release!\n",
                                  origin.hash);

        action = Event::OnLeaseReleaseInvalid;
        // event pointer and size already set
    }
    else
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("Resetting lease with Hash: %llx, IP: %s, Mac: %s to State::Unbound\n",
                                  pRecord->client.hash,
                                  nn::socket::InetNtop(nn::socket::Family::Af_Inet, &pRecord->client.ipAddress, tmp1, sizeof(tmp1)),
                                  MacAddressToString(macAddressString, sizeof(macAddressString), pRecord->client.macAddress));

        pRecord->priority += 1;
        action = Event::OnLeaseRelease;

        LeaseRecordToDhcpLease(&l, *pRecord);
        eventPointer = reinterpret_cast<void*>(&l);
        eventPointerSize = sizeof(l);

        ZeroRecord(pRecord);
    };

    InternalEvent e(static_cast<EventType>(action), eventPointer, eventPointerSize);
    m_pCoordinator->OnEvent(e);

    return rc;
};

void DhcpManager::SendDhcpResponse(NetworkLayersContainer pPacket[],
                                   size_t packetCount) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pCoordinator: %p, pPacket: %p, stackSize: %zu\n",
                              m_pCoordinator, pPacket, packetCount);

    NetworkLayersContainer ipContainer;
    nn::socket::Ip* pInternet = nullptr;

    NetworkLayersContainer udpContainer;
    nn::socket::UdpHdr* pUserDatagram = nullptr;

    NetworkLayersContainer dhcpContainer;
    BootpDhcpMessage* pDhcpMessage = nullptr;

    bool isBroadcast = false;

    if (-1 == NetworkLayersGetLayer(&ipContainer,
                                    NetworkLayer::Internet,
                                    pPacket,
                                    packetCount))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (nullptr == (pInternet = reinterpret_cast<nn::socket::Ip*>(ipContainer.pBuffer)))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (-1 == NetworkLayersGetLayer(&udpContainer,
                                         NetworkLayer::UserDatagram,
                                         pPacket,
                                         packetCount))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (nullptr == (pUserDatagram = reinterpret_cast<nn::socket::UdpHdr*>(udpContainer.pBuffer)))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (-1 == NetworkLayersGetLayer(&dhcpContainer,
                                         NetworkLayer::BootProtocol,
                                         pPacket,
                                         packetCount))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else if (nullptr == (pDhcpMessage = reinterpret_cast<BootpDhcpMessage*>(dhcpContainer.pBuffer)))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    }
    else
    {
        pUserDatagram->uh_ulen = sizeof(nn::socket::UdpHdr) + dhcpContainer.size;
        pInternet->ip_len = sizeof(nn::socket::Ip) + pUserDatagram->uh_ulen;

        isBroadcast = (pDhcpMessage->flags >> static_cast<uint32_t>(LibraryConstants::DhcpBroadcastShift));
        NetworkLayersFlip(pPacket, packetCount);
        pInternet->ip_src.S_addr = nn::socket::InetNtohl(g_Config.GetIpAddress().S_addr);

        if (pInternet->ip_dst.S_addr == 0 || true == isBroadcast)
        {
            pInternet->ip_dst.S_addr =
                static_cast<unsigned int>(LibraryConstants::InternetBroadcast);
        };

        NetworkLayersHton(pPacket, packetCount);
        NetworkLayersChecksum(pPacket, packetCount);

        InternalEvent e(EventType::OnPacketWrite, pPacket, packetCount);
        m_pCoordinator->OnEvent(e);
    };

bail:
    return;
};

}}}; // nn::dhcps::detail
