﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/http/json/http_JsonPath.h>

#include "dauth_InternalTypes.h"

namespace nn { namespace dauth { namespace detail {

/**
 * @brief   JSONパス文字列の検索用ユーティリティ
 * @details あるパス文字列で表されるJonPathがこれまで渡されたかどうかを保持する
 */
struct LookupEntry
{
    const char* path;
    bool found;

    explicit LookupEntry(const char* p) NN_NOEXCEPT
        : path(p)
        , found(false)
    {
        NN_SDK_ASSERT(p);
    }

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return found;
    }

    template <typename JsonPathType>
    bool CanAccept(const JsonPathType& jsonPath) const NN_NOEXCEPT
    {
        return !(*this) && jsonPath.Match(path);
    }

    void MarkAccepted() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!found);
        found = true;
    }
};

class BaseErrorAdaptor
{
    NN_DISALLOW_COPY(BaseErrorAdaptor);

private:
    LookupEntry     m_ErrorCodeEntry {"$.errors[0].code"};
    LookupEntry     m_ErrorMsgEntry {"$.errors[0].message"};
    uint16_t        m_ErrorCode {0};

    void AcceptErrorCode(const char* value, int valueLength) NN_NOEXCEPT;
    void AcceptErrorMsg(const char* value, int valueLength) NN_NOEXCEPT;

public:
    BaseErrorAdaptor() NN_NOEXCEPT = default;
    Result GetError() const NN_NOEXCEPT;

    template <typename JsonPathType>
    bool TryUpdate(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
    {
        if (m_ErrorCodeEntry.CanAccept(jsonPath))
        {
            AcceptErrorCode(value, valueLength);
            return true;
        }
        else if (m_ErrorMsgEntry.CanAccept(jsonPath))
        {
            AcceptErrorMsg(value, valueLength);
            return true;
        }
        return false;
    }
};

/**
 * @brief   チャレンジレスポンス要求アダプタ。
 */
class ChallengeAdaptor
    : BaseErrorAdaptor
{
    NN_DISALLOW_COPY(ChallengeAdaptor);

public:
    using JsonPathType = nn::http::json::JsonPath<8, 32>;

private:
    struct ResponseStore
    {
        KeySource   keySource;
        Challenge   challenge;
    } m_Response;

    LookupEntry     m_EntryChallenge;
    LookupEntry     m_EntryData;

public:
    ChallengeAdaptor() NN_NOEXCEPT;

    Result Adapt() NN_NOEXCEPT;
    const KeySource& GetKeySourceRef() const NN_NOEXCEPT
    {
        return m_Response.keySource;
    }
    const Challenge& GetChallengeRef() const NN_NOEXCEPT
    {
        return m_Response.challenge;
    }

    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;

    // 以下考慮しない入力
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, bool) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, double) NN_NOEXCEPT
    { /*NOP*/ }
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT
    { /*NOP*/ }
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT
    { /*NOP*/ }
};

/**
 * @brief   デバイス認証トークン要求アダプタ。
 */
class DeviceAuthenticationAdaptor
    : BaseErrorAdaptor
{
    NN_DISALLOW_COPY(DeviceAuthenticationAdaptor);

public:
    using JsonPathType = nn::http::json::JsonPath<8, 32>;

private:
    struct Token
    {
        char* buffer;
        size_t bufferSize;
        int length;
    } m_Token;
    TimeSpan m_Expiration;

    LookupEntry m_ExpirationEntry;
    LookupEntry m_TokenEntry;

public:
    DeviceAuthenticationAdaptor(char* buffer, size_t bufferSize) NN_NOEXCEPT;

    Result Adapt() NN_NOEXCEPT;
    TimeSpan GetExpiration() const NN_NOEXCEPT
    {
        return m_Expiration;
    }
    int GetTokenLength() const NN_NOEXCEPT
    {
        return m_Token.length;
    }

    void Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;

    // 以下考慮しない入力
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, bool) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, double) NN_NOEXCEPT
    { /*NOP*/ }
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT
    { /*NOP*/ }
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT
    { /*NOP*/ }
};

class EdgeTokenAuthenticationAdaptor
    : BaseErrorAdaptor
{
    NN_DISALLOW_COPY(EdgeTokenAuthenticationAdaptor);

public:
    using JsonPathType = nn::http::json::JsonPath<8, 32>;

private:
    struct Token
    {
        char* buffer;
        size_t bufferSize;
        int length;
    } m_Token;
    TimeSpan m_Expiration;

    LookupEntry m_ExpirationEntry;
    LookupEntry m_TokenEntry;

public:
    EdgeTokenAuthenticationAdaptor(char* buffer, size_t bufferSize) NN_NOEXCEPT;

    Result Adapt() NN_NOEXCEPT;
    TimeSpan GetExpiration() const NN_NOEXCEPT
    {
        return m_Expiration;
    }
    int GetTokenLength() const NN_NOEXCEPT
    {
        return m_Token.length;
    }

    void Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;

    // 以下考慮しない入力
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, bool) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, double) NN_NOEXCEPT
    { /*NOP*/ }
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT
    { /*NOP*/ }
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT
    { /*NOP*/ }
};

}}} // ~namespace nn::dauth::detail
