﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dauth_AuthenticationAdaptor.h"

#include <cctype>

#include <nn/nn_SdkLog.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/dauth/dauth_Types.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace dauth { namespace detail {
namespace {
inline bool ExtractErrorCodeFromString(uint16_t* pOut, const char* value, int valueLength) NN_NOEXCEPT
{
    const int NdasErrorCodeLength = 4;

    if (!(valueLength == NdasErrorCodeLength))
    {
        return false;
    }
    NN_SDK_ASSERT(value[NdasErrorCodeLength] == '\0');

    int16_t code = 0;
    for (auto i = 0; i < NdasErrorCodeLength; ++i)
    {
        auto c = value[i];
        if (!std::isdigit(c))
        {
            return false;
        }
        code = (code * 10) + c - '0';
    }
    *pOut = code;
    return true;
}

inline util::optional<Result> ConvertDeviceAuthErrorCodeToResult(uint16_t errorCode) NN_NOEXCEPT
{
    const Result NdasDeviceAuthResultTable[] = {
        ResultNotImplemented(),
        ResultNotImplemented(),
        ResultNotImplemented(),
        ResultNotImplemented(),
        ResultNdasStatusNo0004(),
        ResultNotImplemented(),
        ResultNotImplemented(),
        ResultNdasStatusNo0007(),
        ResultNdasStatusNo0008(),
        ResultNdasStatusNo0009(),
        ResultNdasStatusNo0010(),
        ResultNdasStatusNo0011(),
        ResultNdasStatusNo0012(),
        ResultNdasStatusNo0013(),
        ResultNdasStatusNo0014(),
        ResultNdasStatusNo0015(),   // v3 から追加
    };
    static_assert(
        std::extent<decltype(NdasDeviceAuthResultTable)>::value == 16,
        "NDAS DeviceAuth Result definition"
    );

    const auto Length = static_cast<int>(std::extent<decltype(NdasDeviceAuthResultTable)>::value);
    if (errorCode >= 0 && errorCode < Length)
    {
        const auto result = NdasDeviceAuthResultTable[errorCode];
        return ResultNotImplemented::Includes(result) ? util::nullopt : util::optional<Result>(result);
    }
    return util::nullopt;
}
} // ~namespace nn::dauth::detail::<anonymous>

#define NN_DAUTH_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

//!---------------------------------------------------------------------------------
//! BaseErrorAdaptor
//!---------------------------------------------------------------------------------
Result BaseErrorAdaptor::GetError() const NN_NOEXCEPT
{
    if (m_ErrorCodeEntry)
    {
        NN_SDK_LOG("[nn::dauth] -----------------------------------------------\n");
        NN_SDK_LOG("  Error: NDAS BaseErrorAdaptor detects error: %04d\n", m_ErrorCode);
        NN_DAUTH_PRINT_ENTRY_STATE(m_ErrorCodeEntry);
        NN_DAUTH_PRINT_ENTRY_STATE(m_ErrorMsgEntry);

        const auto r = ConvertDeviceAuthErrorCodeToResult(m_ErrorCode);
        NN_RESULT_THROW(r ? *r : ResultInvalidNdasData());
    }
    NN_RESULT_SUCCESS;
}
void BaseErrorAdaptor::AcceptErrorCode(const char* value, int valueLength) NN_NOEXCEPT
{
    uint16_t code;
    if (!ExtractErrorCodeFromString(&code, value, valueLength))
    {
        return;
    }
    m_ErrorCode = code;
    m_ErrorCodeEntry.MarkAccepted();
}
void BaseErrorAdaptor::AcceptErrorMsg(const char* value, int valueLength) NN_NOEXCEPT
{
    NN_UNUSED(valueLength);
    NN_UNUSED(value);
    NN_SDK_LOG(
        "[nn::dauth] NDAS BaseErrorAdaptor ---------------------\n"
        "  Error: \"%s\"\n", value);
    m_ErrorMsgEntry.MarkAccepted();
}

//!---------------------------------------------------------------------------------
//! DeviceAuthenticationAdaptor
//!---------------------------------------------------------------------------------
DeviceAuthenticationAdaptor::DeviceAuthenticationAdaptor(char* buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Token {buffer, bufferSize}
    , m_ExpirationEntry("$.expires_in")
    , m_TokenEntry("$.device_auth_token")
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize >= DeviceAuthenticationTokenLengthMax + 1);
}

Result DeviceAuthenticationAdaptor::Adapt() NN_NOEXCEPT
{
    NN_RESULT_DO(BaseErrorAdaptor::GetError());
    if (!(m_ExpirationEntry && m_TokenEntry))
    {
        NN_SDK_LOG("[nn::dauth] -----------------------------------------------\n");
        NN_SDK_LOG("  Error: NDAS DeviceAuthentication Failed\n");
        NN_DAUTH_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_DAUTH_PRINT_ENTRY_STATE(m_TokenEntry);
        NN_RESULT_THROW(ResultInvalidNdasData());
    }
    NN_RESULT_SUCCESS;
}

void DeviceAuthenticationAdaptor::Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        if (!(value > 0))
        {
            return;
        }
        m_Expiration = os::GetSystemTick().ToTimeSpan() + TimeSpan::FromSeconds(value);
        m_ExpirationEntry.MarkAccepted();
    }
}

void DeviceAuthenticationAdaptor::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (BaseErrorAdaptor::TryUpdate(jsonPath, value, valueLength))
    {
        return;
    }

    if (m_TokenEntry.CanAccept(jsonPath))
    {
        if (!(valueLength <= DeviceAuthenticationTokenLengthMax))
        {
            return;
        }
        m_Token.length = valueLength;
        std::strncpy(m_Token.buffer, value, m_Token.bufferSize);
        NN_SDK_ASSERT(m_Token.buffer[valueLength] == '\0');
        m_TokenEntry.MarkAccepted();
    }
}

//!---------------------------------------------------------------------------------
//! ChallengeAdaptor
//!---------------------------------------------------------------------------------
ChallengeAdaptor::ChallengeAdaptor() NN_NOEXCEPT
    : m_EntryChallenge("$.challenge")
    , m_EntryData("$.data")
{
}

Result ChallengeAdaptor::Adapt() NN_NOEXCEPT
{
    NN_RESULT_DO(BaseErrorAdaptor::GetError());
    if (!(m_EntryData && m_EntryChallenge))
    {
        NN_SDK_LOG("[nn::dauth] -----------------------------------------------\n");
        NN_SDK_LOG("  Error: NDAS Challenge reponse Failed\n");
        NN_DAUTH_PRINT_ENTRY_STATE(m_EntryChallenge);
        NN_DAUTH_PRINT_ENTRY_STATE(m_EntryData);
        NN_RESULT_THROW(ResultInvalidNdasData());
    }
    NN_RESULT_SUCCESS;
}

void ChallengeAdaptor::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (BaseErrorAdaptor::TryUpdate(jsonPath, value, valueLength))
    {
        return;
    }

    if (m_EntryChallenge.CanAccept(jsonPath))
    {
        // Challenge文字列は終端文字含めて 48 byte以内。
        if (!(valueLength < sizeof(Challenge::data)))
        {
            return;
        }
        Challenge challenge {};
        std::strncpy(challenge.data, value, sizeof(challenge.data) - 1);
        m_Response.challenge = challenge;
        m_EntryChallenge.MarkAccepted();
    }
    else if (m_EntryData.CanAccept(jsonPath))
    {
        // data パラメータは KeySource を Base64 エンコードしたもの
        size_t decodedLength;
        detail::KeySource keySource;
        auto b64Error = util::Base64::FromBase64String(&decodedLength, keySource.data, sizeof(keySource.data), value, util::Base64::Mode_UrlSafe);
        if (!(true
            && b64Error == util::Base64::Status_Success
            && decodedLength == sizeof(keySource.data)))
        {
            return;
        }
        m_Response.keySource = keySource;
        m_EntryData.MarkAccepted();
    }
}

//!---------------------------------------------------------------------------------
//! EdgeTokenAuthenticationAdaptor
//!---------------------------------------------------------------------------------
EdgeTokenAuthenticationAdaptor::EdgeTokenAuthenticationAdaptor(char* buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Token {buffer, bufferSize}
    , m_ExpirationEntry("$.expires_in")
    , m_TokenEntry("$.dtoken")
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize >= EdgeTokenLengthMax + 1);
}

Result EdgeTokenAuthenticationAdaptor::Adapt() NN_NOEXCEPT
{
    NN_RESULT_DO(BaseErrorAdaptor::GetError());
    if (!(m_ExpirationEntry && m_TokenEntry))
    {
        NN_SDK_LOG("[nn::dauth] -----------------------------------------------\n");
        NN_SDK_LOG("  Error: NDAS EdgeToken Authentication Failed\n");
        NN_DAUTH_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_DAUTH_PRINT_ENTRY_STATE(m_TokenEntry);
        NN_RESULT_THROW(ResultInvalidNdasData());
    }
    NN_RESULT_SUCCESS;
}

void EdgeTokenAuthenticationAdaptor::Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        if (!(value > 0))
        {
            return;
        }
        m_Expiration = os::GetSystemTick().ToTimeSpan() + TimeSpan::FromSeconds(value);
        m_ExpirationEntry.MarkAccepted();
    }
}

void EdgeTokenAuthenticationAdaptor::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (BaseErrorAdaptor::TryUpdate(jsonPath, value, valueLength))
    {
        return;
    }

    if (m_TokenEntry.CanAccept(jsonPath))
    {
        if (!(valueLength <= EdgeTokenLengthMax))
        {
            return;
        }
        m_Token.length = valueLength;
        std::strncpy(m_Token.buffer, value, m_Token.bufferSize);
        NN_SDK_ASSERT(m_Token.buffer[valueLength] == '\0');
        m_TokenEntry.MarkAccepted();
    }
}

#undef NN_DAUTH_PRINT_ENTRY_STATE

}}} // ~namespace nn::dauth::detail
