﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/codec_OpusPacket.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

#include "opus.h"

namespace nn {
namespace codec {

std::size_t GetOpusPacketSize(const uint8_t pPacket[], size_t packetSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(packetSize >= detail::OpusPacketInternal::HeaderSize);
    NN_UNUSED(packetSize);

    detail::OpusPacketInternal::Header header;
    detail::OpusPacketInternal::GetHeaderFromPacket(&header, pPacket);
    const size_t size = header.packetSize + detail::OpusPacketInternal::HeaderSize;
    return (size > OpusPacketSizeMaximum) ? 0 : size;
}

/**
 * @brief       Opus パケットの、フレーム当たりのサンプル数を取得します。
 * @details
 * opus_packet_get_samples_per_frame() を呼び出します。
 * 必要バイト数は 1 です。
 */
int GetOpusPacketSampleCountPerFrame(const uint8_t packet[], size_t packetSize, int sampleRate) NN_NOEXCEPT
{
    if (packetSize < detail::OpusPacketInternal::HeaderSize + 1)
    {
        return OpusPacketErrorCode;
    }
    int sampleCount = 0;
    auto opusResult = GetOpusPacketSampleCountPerFrame(&sampleCount, packet, packetSize, sampleRate);
    return OpusResult_Success == opusResult ? sampleCount : OpusPacketErrorCode;
}

OpusResult GetOpusPacketSampleCountPerFrame(int* pOutSampleCount, const uint8_t packet[], size_t packetSize, int sampleRate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES_NOT_NULL(pOutSampleCount);
    NN_SDK_REQUIRES_NOT_NULL(packet);
    NN_SDK_ASSERT_GREATER_EQUAL(packetSize, detail::OpusPacketInternal::HeaderSize + 1);
    NN_UNUSED(packetSize);
    *pOutSampleCount = opus_packet_get_samples_per_frame(&packet[detail::OpusPacketInternal::HeaderSize], sampleRate);
    return OpusResult_Success;
}

/**
 * @brief       Opus パケットの、フレーム内のサンプル数を取得します。
 * @details
 * opus_packet_get_nb_samples() を呼び出します。
 * 必要バイト数は 1-2 です。
 */
int GetOpusPacketSampleCountInPacket(const uint8_t packet[], size_t packetSize, int sampleRate) NN_NOEXCEPT
{
    if (packetSize < detail::OpusPacketInternal::HeaderSize + 2)
    {
        return OpusPacketErrorCode;
    }
    int sampleCount = 0;
    auto opusResult = GetOpusPacketSampleCountInPacket(&sampleCount, packet, packetSize, sampleRate);
    return OpusResult_Success == opusResult ? sampleCount : OpusPacketErrorCode;
}

OpusResult GetOpusPacketSampleCountInPacket(int* pOutSampleCount, const uint8_t packet[], size_t packetSize, int sampleRate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES_NOT_NULL(pOutSampleCount);
    NN_SDK_REQUIRES_NOT_NULL(packet);
    NN_SDK_ASSERT_GREATER_EQUAL(packetSize, detail::OpusPacketInternal::HeaderSize + 1);
    const int value = opus_packet_get_nb_samples(
        packet + detail::OpusPacketInternal::HeaderSize,
        static_cast<int>(packetSize - detail::OpusPacketInternal::HeaderSize),
        sampleRate);
    if (OPUS_BAD_ARG == value)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    if (OPUS_INVALID_PACKET == value)
    {
        return OpusResult_BrokenData;
    }
    *pOutSampleCount = value;
    return OpusResult_Success;
}

/**
 * @brief       Opus パケットの、フレーム数を取得します。
 * @details
 * opus_packet_get_nb_frames() を呼び出します。
 * 必要バイト数は 1-2 です。
 */
int GetOpusPacketFrameCount(const uint8_t packet[], size_t packetSize) NN_NOEXCEPT
{
    if (packetSize < detail::OpusPacketInternal::HeaderSize + 2)
    {
        return OpusPacketErrorCode;
    }
    int frameCount = 0;
    auto opusResult = GetOpusPacketFrameCount(&frameCount, packet, packetSize);
    return OpusResult_Success == opusResult ? frameCount : OpusPacketErrorCode;
}

OpusResult GetOpusPacketFrameCount(int* pOutFrameCount, const uint8_t packet[], size_t packetSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFrameCount);
    NN_SDK_REQUIRES_NOT_NULL(packet);
    NN_SDK_ASSERT_GREATER_EQUAL(packetSize, detail::OpusPacketInternal::HeaderSize + 1);
    const int value = opus_packet_get_nb_frames(
        packet + detail::OpusPacketInternal::HeaderSize,
        static_cast<int>(packetSize - detail::OpusPacketInternal::HeaderSize));
    if (OPUS_BAD_ARG == value)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    if (OPUS_INVALID_PACKET == value)
    {
        return OpusResult_BrokenData;
    }
    *pOutFrameCount = value;
    return OpusResult_Success;
}

/**
 * @brief       Opus パケットの、フレーム数を取得します。
 * @details
 * opus_packet_get_nb_channels() を呼び出します。
 * 必要バイト数は 1 です。
 */
int GetOpusPacketChannelCount(const uint8_t packet[], size_t packetSize) NN_NOEXCEPT
{
    if (packetSize < detail::OpusPacketInternal::HeaderSize + 1)
    {
        return OpusPacketErrorCode;
    }
    int channelCount = 0;
    auto opusResult = GetOpusPacketChannelCount(&channelCount, packet, packetSize);
    return OpusResult_Success == opusResult ? channelCount : OpusPacketErrorCode;
}

OpusResult GetOpusPacketChannelCount(int* pOutChannelCount, const uint8_t packet[], size_t packetSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutChannelCount);
    NN_SDK_REQUIRES_NOT_NULL(packet);
    NN_SDK_ASSERT_GREATER_EQUAL(packetSize, detail::OpusPacketInternal::HeaderSize + 1);
    NN_UNUSED(packetSize);
    *pOutChannelCount = opus_packet_get_nb_channels(&packet[detail::OpusPacketInternal::HeaderSize]);
    return OpusResult_Success;
}

int CalculateOpusFrameSampleCount(int frame, int sampleRate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(frame == 2500 || frame == 5000 || frame == 10000 || frame == 20000);
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    return static_cast<int>((frame * sampleRate) / 1000 / 1000);
}

}}  // nn::codec
