﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>  // std::memset

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/codec/codec_OpusPacket.h>
#include <nn/codec/codec_OpusDecoder.h>
#include <nn/codec/detail/codec_OpusCommonInternal.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

#include "opus.h"

namespace nn {
namespace codec {

namespace {

OpusResult ConvertToOpusResult(int result) NN_NOEXCEPT
{
    switch(result)
    {
    case OPUS_UNIMPLEMENTED:
        return OpusResult_UnsupportedFormat;
    case OPUS_INTERNAL_ERROR:
        return OpusResult_InternalError;
    default:
        return OpusResult_BrokenData;
    }
}

} // anonymous namespace

OpusDecoder::OpusDecoder() NN_NOEXCEPT
    : m_Decoder()
{
}

OpusDecoder::~OpusDecoder() NN_NOEXCEPT
{
}

OpusResult OpusDecoder::Initialize(int sampleRate, int channelCount, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    return InitializeOpusDecoder(&m_Decoder, sampleRate, channelCount, buffer, size);
}

void OpusDecoder::Finalize() NN_NOEXCEPT
{
    FinalizeOpusDecoder(&m_Decoder);
}

size_t GetOpusDecoderWorkBufferSize(int sampleRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= 2);

    NN_UNUSED(sampleRate);

    return opus_decoder_get_size(channelCount);
}

OpusResult InitializeOpusDecoder(OpusDecoderType* pOutDecoder, int sampleRate, int channelCount, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDecoder);
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= 2);
    NN_SDK_REQUIRES(buffer != nullptr);
    NN_SDK_REQUIRES(size >= GetOpusDecoderWorkBufferSize(sampleRate, channelCount));

    auto pOpusDecoder = static_cast<::OpusDecoder*>(buffer);
    if (opus_decoder_init(pOpusDecoder, sampleRate, channelCount) != OPUS_OK)
    {
        return OpusResult_OutOfResource;  // TODO: 適切なエラーを返すように
    }

    pOutDecoder->_handle = pOpusDecoder;
    pOutDecoder->_buffer = buffer;
    pOutDecoder->_size = size;
    pOutDecoder->_sampleRate = sampleRate;
    pOutDecoder->_channelCount = channelCount;

    return OpusResult_Success;
}

void FinalizeOpusDecoder(OpusDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    pDecoder->_handle = nullptr;
    pDecoder->_buffer = nullptr;
    pDecoder->_size = 0;
    pDecoder->_sampleRate = 0;
    pDecoder->_channelCount = 0;
}

int GetOpusDecoderSampleRate(const OpusDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_sampleRate;
}

int GetOpusDecoderChannelCount(const OpusDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_channelCount;
}

OpusResult DecodeOpusInterleaved(OpusDecoderType* pDecoder, size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutConsumed);
    NN_SDK_REQUIRES_NOT_NULL(pOutSampleCount);
    NN_SDK_REQUIRES_NOT_NULL(outputBuffer);
    NN_SDK_REQUIRES_NOT_NULL(inputBuffer);

    nn::codec::detail::OpusPacketInternal::Header header;
    const size_t frameHeaderSize = sizeof(header);  // TODO: ちゃんとした値は後日決めるが、今は opus_demo に合わせておく
    if (inputSize < frameHeaderSize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    auto packet = static_cast<const uint8_t*>(inputBuffer);
    nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, packet);
    const auto frameBodySize = header.packetSize;
    if (inputSize < frameHeaderSize + frameBodySize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    int sampleCount = 0;
    const auto result = GetOpusPacketSampleCountInPacket(&sampleCount, packet, inputSize, GetOpusDecoderSampleRate(pDecoder));
    if (OpusResult_Success != result)
    {
        return result;
    }
    if (outputSize < sampleCount * GetOpusDecoderChannelCount(pDecoder) * sizeof(int16_t))
    {
        return OpusResult_InsufficientPcmBuffer;
    }
    auto opusFrame = packet + frameHeaderSize;
    // opus_decoder_ctl と opus_decode が排他的に呼び出されるように制御する。
    std::lock_guard<nn::os::Mutex> lock(detail::GetOpusGlobalMutex());
    if (pDecoder->_isResetRequested)
    {
        pDecoder->_isResetRequested = false;

        if (const auto ret = opus_decoder_ctl(static_cast<::OpusDecoder*>(pDecoder->_handle), OPUS_RESET_STATE) < 0)
        {
            return ConvertToOpusResult(ret);
        }
    }

    auto ret = opus_decode(static_cast<::OpusDecoder*>(pDecoder->_handle), opusFrame, static_cast<int>(frameBodySize), outputBuffer,  static_cast<opus_int32>(outputSize / sizeof(int16_t) / GetOpusDecoderChannelCount(pDecoder)), 0);
    if (ret < 0)
    {
        return ConvertToOpusResult(ret);
    }

    *pOutConsumed = frameHeaderSize + frameBodySize;
    *pOutSampleCount = ret;

    return OpusResult_Success;
}

void ResetOpusDecoder(OpusDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

    pDecoder->_isResetRequested = true;
}

}  // namespace codec
}  // namespace nn
