﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_LowLevelInterface.h"
#include "cec_ReceiveBuffer.h"
#include "cec_TransmitterFiniteStateMachine.h"
#include "cec_Receiver.h"
#include "cec_Manager.h"
#include "cec_Opcodes.h"

namespace nn { namespace cec { namespace detail {

const uint8_t   CecReceiveParamMaxOsdStringLength           = 14;
/* per CEC 1.3a spec, sections 5.2 and 9.1 */
const uint32_t  CecReceiveParamBitPeriodNanoSeconds         = 2400 * 1000;
const uint32_t  CecReceiveParamFrameToFrameDelayNanoSeconds = 7 * CecReceiveParamBitPeriodNanoSeconds;
const uint32_t  CecReceiveParamErrorToNextDelayNanoSeconds  = 3 * CecReceiveParamBitPeriodNanoSeconds;
const uint32_t  CecReceiveParamCradleFirmwareBugWorkAroundNanoSeconds = 10 * CecReceiveParamBitPeriodNanoSeconds;

typedef struct {
    uint32_t            flags;
    uint32_t            count;
    nn::os::ThreadType  thread;
    nn::os::EventType   startEvent;
    nn::os::EventType   doneEvent;
    nn::os::MutexType   mutex;
    OperatingStateType* pOpState;
    EdidVariationType*  pEdidVariation;
    uint8_t             receiveBuffer[CecReceiveParamMaxMessageLength];
    uint8_t             osdString[CecReceiveParamMaxOsdStringLength];
} ReceiveStateType;

namespace {

const uint32_t CecReceiveStateFlagInit         = 1 << 0;
const uint32_t CecReceiveStateFlagRunning      = 1 << 1;
const uint32_t CecReceiveStateFlagReceiving    = 1 << 2;
const uint32_t CecReceiveStateFlagVid          = 1 << 3;
const uint32_t CecReceiveStateFlagActiveSource = 1 << 4;

const size_t   CecReceiverThreadStackSizeBytes = 8 * 1024;
const uint32_t CecReceiverRingbufferSizeBytes  = 256;

const uint8_t CecReceiveParamLogicalAddressMask = 0x0F;

uint8_t s_CecRxRingbuffer[CecReceiverRingbufferSizeBytes];

ReceiveStateType s_CecReceiveState;
NN_ALIGNAS(os::MemoryPageSize) char s_CecReceiverStack[CecReceiverThreadStackSizeBytes];

const uint32_t CecBcMin_HoldoffUs  = 500 * 1000;
const uint32_t CecBcMin_BusQuietUs = 80 * 1000;
const uint32_t CecReceiverParamLongQuietPeriodInMilliseconds = 750;
const uint32_t CecReceiverParamShortQuietPeriodInMilliseconds = 250;

const uint8_t AddrType_Direct = 0;
const uint8_t AddrType_Broadcast = 1;
const uint8_t AddrType_Other = 2;

void    ReceiverSendFeatureAbort(ReceiveStateType* pState, uint8_t sourceLogicalAddress, uint8_t destinationLogicalAddress, uint8_t opcode, uint8_t reason) NN_NOEXCEPT
{
    nn::os::LockMutex(&pState->pEdidVariation->mutex);
    if(pState->pEdidVariation->featureAbortHandler != nullptr)
    {
        pState->pEdidVariation->featureAbortHandler(sourceLogicalAddress, destinationLogicalAddress, opcode, reason);
    }
    nn::os::UnlockMutex(&pState->pEdidVariation->mutex);
}

bool    ReceiverDispatchVendorCommands(ReceiveStateType* pState,
                                       uint8_t sourceLogicalAddress,
                                       uint8_t addrType) NN_NOEXCEPT
{
    uint8_t abortReason = CecAbortReasonReserved;
    bool    retVal;
    bool    handled = true;

    switch(pState->receiveBuffer[CecBufferOpcodeIndex])
    {
        case CecOpcodeVendorCommandWithId:
            if(pState->count > CecBufferElementCountIsFive)
            {
                retVal = CecManagerHandleVendorCommandWithId(sourceLogicalAddress,
                           &pState->receiveBuffer[CecBufferPayloadIndex],
                           pState->count - CecBufferElementCountIsFive,
                           &pState->receiveBuffer[CecBufferElementCountIsFive]);
                if((retVal == false) && (addrType == AddrType_Direct))
                {
                    abortReason = CecAbortReasonNotCorrectMode;
                }
            }
            else if(addrType == AddrType_Direct)
            {
                abortReason = CecAbortReasonInvalidOperand;
            }
            break;
        case CecOpcodeVendorRemoteButtonDown:
            if(pState->count > CecBufferElementCountIsTwo)
            {
                retVal = CecManagerHandleVendorRemoteControlButtonDown(sourceLogicalAddress,
                            pState->count - CecBufferElementCountIsTwo,
                            &pState->receiveBuffer[CecBufferPayloadIndex]);
                if((retVal == false) && (addrType == AddrType_Direct))
                {
                    abortReason = CecAbortReasonNotCorrectMode;
                }
            }
            else if(addrType == AddrType_Direct)
            {
                abortReason = CecAbortReasonInvalidOperand;
            }
            break;
        case CecOpcodeVendorRemoteButtonUp:
            retVal = CecManagerHandleVendorRemoteControlButtonUp(sourceLogicalAddress);
            if((retVal == false) && (addrType == AddrType_Direct))
            {
                abortReason = CecAbortReasonNotCorrectMode;
            }
            break;
        default:
            handled = false;
            break;
    }
    if(abortReason != CecAbortReasonReserved)
    {
        ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                                 pState->receiveBuffer[CecBufferOpcodeIndex], abortReason);
    }
    return handled;
}

void ReceiverDispatchTwoByteOpcodes(ReceiveStateType* pState,
                                    uint8_t sourceLogicalAddress,
                                    uint8_t directLogicalAddressValue,
                                    uint8_t broadcastLogicalAddressValue) NN_NOEXCEPT
{
    uint8_t     txBuffer[CecReceiveParamMaxMessageLength];

    switch(pState->receiveBuffer[CecBufferOpcodeIndex])
    {
        case CecOpcodeAbort:
            if(sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress)
            {
                ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                                         pState->receiveBuffer[CecBufferOpcodeIndex], CecAbortReasonRefused);
            }
            break;
        case CecOpcodeGetCecVersion:
            if(sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress)
            {
                txBuffer[CecBufferLogicalAddressIndex] = directLogicalAddressValue;
                txBuffer[CecBufferOpcodeIndex] = CecOpcodeCecVersion;
                txBuffer[CecBufferPayloadIndex] = CecVersionNumber13a;
                CecTransmitBestEffort(txBuffer, CecBufferElementCountIsThree, true);
            }
            break;
        case CecOpcodeGivePhysicalAddress:
            txBuffer[CecBufferLogicalAddressIndex] = broadcastLogicalAddressValue;
            txBuffer[CecBufferOpcodeIndex] = CecOpcodeReportPhysicalAddress;
            txBuffer[CecBufferPayloadIndex + 0] = pState->pOpState->physAddr[0];
            txBuffer[CecBufferPayloadIndex + 1] = pState->pOpState->physAddr[1];
            txBuffer[CecBufferPayloadIndex + 2] = CecDeviceTypePlaybackDevice;
            CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFive, false);
            break;
        case CecOpcodeGiveDeviceVendorId:
            txBuffer[CecBufferLogicalAddressIndex] = broadcastLogicalAddressValue;
            txBuffer[CecBufferOpcodeIndex] = CecOpcodeDeviceVendorId;
            txBuffer[CecBufferPayloadIndex + 0] = CecDeviceVendorIdLow;
            txBuffer[CecBufferPayloadIndex + 1] = CecDeviceVendorIdMid;
            txBuffer[CecBufferPayloadIndex + 2] = CecDeviceVendorIdHigh;
            CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFive, false);
            break;
        case CecOpcodeGiveDevicePowerStatus:
            if(sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress)
            {
                nn::os::LockMutex(&pState->pOpState->mutex);
                txBuffer[CecBufferLogicalAddressIndex] = directLogicalAddressValue;
                txBuffer[CecBufferOpcodeIndex] = CecOpcodeReportPowerStatus;
                txBuffer[CecBufferPayloadIndex] = pState->pOpState->powerState;
                nn::os::UnlockMutex(&pState->pOpState->mutex);
                CecTransmitBestEffort(txBuffer, CecBufferElementCountIsThree, true);
            }
            break;
        case CecOpcodeGiveOsdName:
            if(sourceLogicalAddress != CecReceiveParamBroadcastLogicalAddress)
            {
                uint8_t index;
                bool    doNotSend;

                os::LockMutex(&pState->pEdidVariation->mutex);
                doNotSend = !!(pState->pEdidVariation->flags & EdidVariationFlagsNoOsdReply);
                os::UnlockMutex(&pState->pEdidVariation->mutex);
                if(doNotSend == false) {
                    nn::os::LockMutex(&pState->pOpState->mutex);
                    txBuffer[CecBufferLogicalAddressIndex] = directLogicalAddressValue;
                    txBuffer[CecBufferOpcodeIndex] = CecOpcodeSetOsdName;
                    for(index = 0; index < CecReceiveParamMaxOsdStringLength; index++)
                    {
                        if(pState->osdString[index] == '\0')
                        {
                            break;
                        }
                        txBuffer[CecBufferPayloadIndex + index] = pState->osdString[index];
                    }
                    nn::os::UnlockMutex(&pState->pOpState->mutex);
                    CecTransmitBestEffort(txBuffer, CecBufferElementCountIsTwo + index, true);
                }
            }
            break;
        case CecOpcodeCecVersion:
        case CecOpcodeReportPowerStatus:
        case CecOpcodeVendorCommand:
        case CecOpcodeFeatureAbort:
        case CecOpcodeMenuRequest:
        case CecOpcodeMenuStatus:
            if(sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress)
            {
                ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                                         pState->receiveBuffer[CecBufferOpcodeIndex], CecAbortReasonInvalidOperand);
            }
            break;
        case CecOpcodeActiveSource:
        case CecOpcodeRoutingChange:
        case CecOpcodeRoutingInformation:
        case CecOpcodeRequestActiveSource:
        case CecOpcodeSetStreamPath:
        case CecOpcodeSetMenuLanguage:
        case CecOpcodeReportPhysicalAddress:
        case CecOpcodeDeviceVendorId:
        case CecOpcodeUserControlReleased:
            break;
        default:
            if(sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress)
            {
                ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                                     pState->receiveBuffer[CecBufferOpcodeIndex], CecAbortReasonUnrecognizedOpcode);
            }
            break;
    }
} //NOLINT(impl/function_size)

void ReceiverDispatchMoreThanTwoByteOpcodes(ReceiveStateType* pState,
                                            uint8_t sourceLogicalAddress,
                                            uint8_t directLogicalAddressValue,
                                            uint8_t broadcastLogicalAddressValue) NN_NOEXCEPT
{
    uint8_t abortReason = CecAbortReasonReserved;
    bool    retVal;
    uint8_t txBuffer[CecReceiveParamMaxMessageLength];

    switch(pState->receiveBuffer[CecBufferOpcodeIndex])
    {
        case CecOpcodeAbort:
        case CecOpcodeGetCecVersion:
        case CecOpcodeGivePhysicalAddress:
        case CecOpcodeGiveDeviceVendorId:
        case CecOpcodeGiveDevicePowerStatus:
        case CecOpcodeGiveOsdName:
        case CecOpcodeUserControlReleased:
            ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                    pState->receiveBuffer[CecBufferOpcodeIndex], CecAbortReasonInvalidOperand);
            break;
        case CecOpcodeUserControlPressed:
            if(pState->count != CecBufferElementCountIsThree)
            {
                ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                        pState->receiveBuffer[CecBufferOpcodeIndex], CecAbortReasonInvalidOperand);
            }
            break;
        case CecOpcodeCecVersion:
            if(pState->count == CecBufferElementCountIsThree)
            {
                CecManagerHandleCecVersion(sourceLogicalAddress, pState->receiveBuffer[CecBufferPayloadIndex]);
            }
            else
            {
                abortReason = CecAbortReasonInvalidOperand;
            }
            break;
        case CecOpcodeReportPowerStatus:
            if(pState->count == CecBufferElementCountIsThree)
            {
                CecManagerHandleReportPowerStatus(sourceLogicalAddress, pState->receiveBuffer[CecBufferPayloadIndex]);
            }
            else
            {
                abortReason = CecAbortReasonInvalidOperand;
            }
            break;
        case CecOpcodeMenuRequest:
            if(pState->count == CecBufferElementCountIsThree)
            {
                uint8_t currentState;

                os::LockMutex(&pState->pOpState->mutex);
                switch(pState->receiveBuffer[CecBufferPayloadIndex])
                {
                    case CecMenuRequestTypeDeactivate:
                        pState->pOpState->opStatus &= ~CecOpStatusFlagMenuActiveState;
                        currentState = CecMenuStateDeactivated;
                        break;
                    case CecMenuRequestTypeActivate:
                        pState->pOpState->opStatus |= CecOpStatusFlagMenuActiveState;
                        currentState = CecMenuStateActivated;
                        break;
                    case CecMenuRequestTypeQuery:
                        currentState = (pState->pOpState->opStatus & CecOpStatusFlagMenuActiveState) ?
                                       CecMenuStateActivated : CecMenuStateDeactivated;
                        break;
                    default:
                        currentState = CecMenuStateInvalidState;
                        abortReason = CecAbortReasonInvalidOperand;
                        break;
                }
                os::UnlockMutex(&pState->pOpState->mutex);
                if(currentState != CecMenuStateInvalidState)
                {
                    txBuffer[CecBufferLogicalAddressIndex] = directLogicalAddressValue;
                    txBuffer[CecBufferOpcodeIndex] = CecOpcodeMenuStatus;
                    txBuffer[CecBufferPayloadIndex + 0] = currentState;
                    CecTransmitBestEffort(txBuffer, CecBufferElementCountIsThree, true);
                }
            }
            else
            {
                abortReason = CecAbortReasonInvalidOperand;
            }
            break;
        case CecOpcodeVendorCommand:
            retVal = CecManagerHandleVendorCommand(sourceLogicalAddress,
                                                   pState->count - CecBufferElementCountIsTwo,
                                                   &pState->receiveBuffer[CecBufferPayloadIndex]);
            if(retVal == false)
            {
                abortReason = CecAbortReasonNotCorrectMode;
            }
            break;
        case CecOpcodeFeatureAbort:
            if(pState->count == CecBufferElementCountIsFour)
            {
                CecManagerHandleFeatureAbort(sourceLogicalAddress,
                                             pState->receiveBuffer[CecBufferPayloadIndex + 0],
                                             pState->receiveBuffer[CecBufferPayloadIndex + 1]);
            }
            break;
        case CecOpcodeMenuStatus:
        case CecOpcodeActiveSource:
        case CecOpcodeRoutingChange:
        case CecOpcodeRoutingInformation:
        case CecOpcodeRequestActiveSource:
        case CecOpcodeSetStreamPath:
        case CecOpcodeSetMenuLanguage:
        case CecOpcodeReportPhysicalAddress:
        case CecOpcodeDeviceVendorId:
            break;
        default:
            abortReason = CecAbortReasonUnrecognizedOpcode;
            break;
    }
    if(abortReason != CecAbortReasonReserved)
    {
        ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                                 pState->receiveBuffer[CecBufferOpcodeIndex], abortReason);
    }
} // NOLINT(impl/function_size)

void ReceiverDispatchBroadcastOpcodes(ReceiveStateType* pState,
                                      uint8_t sourceLogicalAddress,
                                      uint8_t broadcastLogicalAddressValue) NN_NOEXCEPT
{
    bool    isSame;
    bool    isActive;
    uint8_t activeSourceFlag;
    int32_t rval;
    uint8_t txBuffer[CecReceiveParamMaxMessageLength];

    switch(pState->receiveBuffer[CecBufferOpcodeIndex])
    {
        case CecOpcodeRoutingChange:
            if(pState->count == CecBufferElementCountIsSix)
            {
                rval = CecLowLevelInterfaceGetPhysicalAddress(pState->pOpState->physAddr);
                if(rval == CecNoError)
                {
                    isSame = ((pState->pOpState->physAddr[0] ==
                               pState->receiveBuffer[CecBufferPayloadIndex + 2]) &&
                              (pState->pOpState->physAddr[1] ==
                               pState->receiveBuffer[CecBufferPayloadIndex + 3]));
                    activeSourceFlag = (isSame == true) ? CecOpStatusFlagActiveSource : 0;
                    nn::os::LockMutex(&pState->pOpState->mutex);
                    pState->pOpState->opStatus &= ~CecOpStatusFlagActiveSource;
                    pState->pOpState->opStatus |= activeSourceFlag;
                    nn::os::UnlockMutex(&pState->pOpState->mutex);
                    CecManagerHandleActiveSource(sourceLogicalAddress, isSame, true);
                }
            }
            break;
        case CecOpcodeActiveSource:
        case CecOpcodeSetStreamPath:
            if((pState->count == CecBufferElementCountIsFour) &&
               (sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress))
            {
                isSame =
                  ((pState->pOpState->physAddr[0] == pState->receiveBuffer[CecBufferPayloadIndex + 0]) &&
                   (pState->pOpState->physAddr[1] == pState->receiveBuffer[CecBufferPayloadIndex + 1]));
                activeSourceFlag = (isSame == true) ? CecOpStatusFlagActiveSource : 0;
                nn::os::LockMutex(&pState->pOpState->mutex);
                pState->pOpState->opStatus &= ~CecOpStatusFlagActiveSource;
                pState->pOpState->opStatus |= activeSourceFlag;
                nn::os::UnlockMutex(&pState->pOpState->mutex);
                CecManagerHandleActiveSource(sourceLogicalAddress, isSame, false);
            }
            break;
        case CecOpcodeRequestActiveSource:
            if(pState->count == CecBufferElementCountIsTwo)
            {
                nn::os::LockMutex(&pState->pOpState->mutex);
                isActive = ((pState->pOpState->opStatus & CecOpStatusFlagActiveSource) &&
                            (pState->pOpState->opStatus & CecOpStatusFlagPhysicalAddress));
                nn::os::UnlockMutex(&pState->pOpState->mutex);
                if(isActive)
                {
                    txBuffer[CecBufferLogicalAddressIndex] = broadcastLogicalAddressValue;
                    txBuffer[CecBufferOpcodeIndex] = CecOpcodeActiveSource;
                    txBuffer[CecBufferPayloadIndex + 0] = pState->pOpState->physAddr[0];
                    txBuffer[CecBufferPayloadIndex + 1] = pState->pOpState->physAddr[1];
                    CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFour, false);
                }
            }
            break;
        case CecOpcodeDeviceVendorId:
            if((pState->count == CecBufferElementCountIsFive) &&
               (sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress))
            {
                CecManagerHandleDeviceVendorId(sourceLogicalAddress, &pState->receiveBuffer[CecBufferPayloadIndex]);
            }
            break;
        default:
            break;
    }
}

void    ReceiverDispatchPacket(ReceiveStateType* pState) NN_NOEXCEPT
{
    uint8_t     addrType;
    uint8_t     destinationLogicalAddress;
    uint8_t     sourceLogicalAddress;
    uint8_t     broadcastLogicalAddress;
    uint8_t     directLogicalAddress;
    bool        isDone = false;

    destinationLogicalAddress = pState->receiveBuffer[CecBufferLogicalAddressIndex] &
                                CecReceiveParamLogicalAddressMask;
    sourceLogicalAddress = (pState->receiveBuffer[CecBufferLogicalAddressIndex] >> 4) &
                           CecReceiveParamLogicalAddressMask;
    if(destinationLogicalAddress == CecReceiveParamBroadcastLogicalAddress)
    {
        addrType = AddrType_Broadcast;
    }
    else if(destinationLogicalAddress == pState->pOpState->logicalAddress)
    {
        addrType = AddrType_Direct;
    }
    else
    {
        addrType = AddrType_Other;
    }
    if(pState->count > CecReceiveParamMaxMessageLength)
    {
        NN_CEC_INFO("%s: too many bytes received %d\n", NN_CURRENT_FUNCTION_NAME, pState->count);
        if(addrType == AddrType_Direct)
        {
            ReceiverSendFeatureAbort(pState, pState->pOpState->logicalAddress, sourceLogicalAddress,
                                 pState->receiveBuffer[CecBufferOpcodeIndex], CecAbortReasonInvalidOperand);
        }
    }
    else if(pState->count > CecBufferElementCountIsOne)
    {
        broadcastLogicalAddress = ((pState->pOpState->logicalAddress & CecReceiveParamLogicalAddressMask) << 4) |
                                        CecReceiveParamBroadcastLogicalAddress;
        directLogicalAddress = ((pState->pOpState->logicalAddress & CecReceiveParamLogicalAddressMask) << 4) +
                                        sourceLogicalAddress;
        if((addrType == AddrType_Broadcast) || (addrType == AddrType_Direct))
        {
            isDone = CecManagerSequenceCompletionHandler(pState->receiveBuffer, pState->count);
            if(pState->receiveBuffer[CecBufferOpcodeIndex] == CecOpcodeStandby)
            {
                CecManagerHandleGoStandby();
                isDone = true;
            }
            if((isDone == false) && (sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress))
            {
                isDone = ReceiverDispatchVendorCommands(pState, sourceLogicalAddress, addrType);
            }
            if((isDone == false) && (addrType == AddrType_Direct))
            {
                if(pState->count == CecBufferElementCountIsTwo)
                {
                    ReceiverDispatchTwoByteOpcodes(pState,
                                                   sourceLogicalAddress,
                                                   directLogicalAddress,
                                                   broadcastLogicalAddress);
                }
                else if(sourceLogicalAddress != CecReceiveParamUnassignedLogicalAddress)
                {
                    ReceiverDispatchMoreThanTwoByteOpcodes(pState,
                                                           sourceLogicalAddress,
                                                           directLogicalAddress,
                                                           broadcastLogicalAddress);
                }
            }
            else if((isDone == false) && (addrType == AddrType_Broadcast))
            {
                ReceiverDispatchBroadcastOpcodes(pState, sourceLogicalAddress, broadcastLogicalAddress);
            }
        }
    }
}

void    ReceiverReceiveDataStream(ReceiveStateType* pState) NN_NOEXCEPT
{
    int32_t     retVal;

    retVal = CecRxBufferDequeuePacket(pState->receiveBuffer, &pState->count, -1);
    if((retVal == CecNoError) && (pState->count > 0))
    {
        ReceiverDispatchPacket(pState);
    }
}

bool ReceiverIsRunning(ReceiveStateType* pState) NN_NOEXCEPT
{
    bool rval;

    nn::os::LockMutex(&pState->mutex);
    rval = (pState->flags & CecReceiveStateFlagRunning);
    nn::os::UnlockMutex(&pState->mutex);
    return rval;
}

bool ReceiverIsReceiving(ReceiveStateType* pState) NN_NOEXCEPT
{
    bool rval;

    nn::os::LockMutex(&pState->mutex);
    rval = (pState->flags & CecReceiveStateFlagReceiving);
    nn::os::UnlockMutex(&pState->mutex);
    return rval;
}

void    ReceiverReceiveThread(void* pArg) NN_NOEXCEPT
{
    ReceiveStateType*    pState;

    pState = static_cast<ReceiveStateType*>(pArg);
    while(ReceiverIsRunning(pState) == true)
    {
        nn::os::WaitEvent(&pState->startEvent);
        nn::os::ClearEvent(&pState->doneEvent);
        while(ReceiverIsReceiving(pState) == true)
        {
            ReceiverReceiveDataStream(pState);
        }
        nn::os::SignalEvent(&pState->doneEvent);
    }
    nn::os::WaitEvent(&pState->startEvent);
    nn::os::SignalEvent(&pState->doneEvent);
}

void    ReceiverRequestActiveSourceCallback(uint8_t opcode, void* pContext) NN_NOEXCEPT
{
    ReceiveStateType*    pState;

    pState = static_cast<ReceiveStateType*>(pContext);
    nn::os::LockMutex(&pState->mutex);
    pState->flags |= CecReceiveStateFlagActiveSource;
    nn::os::UnlockMutex(&pState->mutex);
}

void    ReceiverRequestVendorIdCallback(uint8_t opcode, void* pContext) NN_NOEXCEPT
{
    ReceiveStateType*    pState;

    pState = static_cast<ReceiveStateType*>(pContext);
    nn::os::LockMutex(&pState->mutex);
    pState->flags |= CecReceiveStateFlagVid;
    nn::os::UnlockMutex(&pState->mutex);
}

}

void    CecReceiverSendFeatureAbort(uint8_t sourceLogicalAddress, uint8_t destinationLogicalAddress, uint8_t opcode, uint8_t reason) NN_NOEXCEPT
{
    uint8_t     txBuffer[CecReceiveParamMaxMessageLength];

    txBuffer[CecBufferLogicalAddressIndex] = ((sourceLogicalAddress & CecReceiveParamLogicalAddressMask) << 4) +
                                                destinationLogicalAddress;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeFeatureAbort;
    txBuffer[CecBufferPayloadIndex + 0] = opcode;
    txBuffer[CecBufferPayloadIndex + 1] = reason;
    CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFour, true);
}

void    CecReceiverSendFeatureAbortNop(uint8_t sourceLogicalAddress, uint8_t destinationLogicalAddress, uint8_t opcode, uint8_t reason) NN_NOEXCEPT
{
    NN_UNUSED(sourceLogicalAddress);
    NN_UNUSED(destinationLogicalAddress);
    NN_UNUSED(opcode);
    NN_UNUSED(reason);
}

int32_t CecReceiverSendVendorId(bool* pIsSent) NN_NOEXCEPT
{
    int32_t rval=CecReceiverInvalidInitError;
    nn::os::TimerEventType  timerEvent;
    uint32_t    delayUs;
    uint32_t    deltaUs;
    uint8_t     txBuffer[8];

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    *pIsSent = false;
    delayUs = CecBcMin_HoldoffUs;
    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
    do {
        nn::os::StartOneShotTimerEvent(&timerEvent, nn::TimeSpan::FromMicroSeconds(delayUs));
        nn::os::WaitTimerEvent(&timerEvent);
        nn::os::LockMutex(&s_CecReceiveState.mutex);
        *pIsSent = (s_CecReceiveState.flags & CecReceiveStateFlagVid);
        nn::os::UnlockMutex(&s_CecReceiveState.mutex);
        if(*pIsSent == true)
        {
            rval = CecNoError;
            break;
        }
        rval = CecLowLevelInterfaceLastTraffic(&deltaUs);
        if(rval != CecNoError)
        {
            break;
        }
        delayUs = (deltaUs < CecBcMin_BusQuietUs) ? CecBcMin_BusQuietUs : 0;
    } while(delayUs > 0);
    nn::os::FinalizeTimerEvent(&timerEvent);
    CecReceiveBufferClearListener(CecOpcodeGiveDeviceVendorId, ReceiverRequestVendorIdCallback);
    if(*pIsSent == false)
    {
        txBuffer[CecBufferLogicalAddressIndex] = (s_CecReceiveState.pOpState->logicalAddress << 4) | CecReceiveParamBroadcastLogicalAddress;
        txBuffer[CecBufferOpcodeIndex] = CecOpcodeDeviceVendorId;
        txBuffer[CecBufferPayloadIndex + 0] = CecDeviceVendorIdLow;
        txBuffer[CecBufferPayloadIndex + 1] = CecDeviceVendorIdMid;
        txBuffer[CecBufferPayloadIndex + 2] = CecDeviceVendorIdHigh;
        rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFive, false);
    }
    return rval;
}

int32_t CecReceiverSendInactiveSource() NN_NOEXCEPT
{
    int32_t rval=CecReceiverInvalidInitError;
    uint8_t txBuffer[8];
    uint8_t addressValue;

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    addressValue =  (s_CecReceiveState.pOpState->logicalAddress << 4) | CecReceiveParamTvLogicalAddress;
    txBuffer[CecBufferLogicalAddressIndex] = addressValue;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeInactiveSource;
    txBuffer[CecBufferPayloadIndex + 0] = s_CecReceiveState.pOpState->physAddr[0];
    txBuffer[CecBufferPayloadIndex + 1] = s_CecReceiveState.pOpState->physAddr[1];
    rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFour, true);
    if(rval == CecNoError)
    {
        nn::os::LockMutex(&s_CecReceiveState.pOpState->mutex);
        s_CecReceiveState.pOpState->opStatus &= ~CecOpStatusFlagActiveSource;
        nn::os::UnlockMutex(&s_CecReceiveState.pOpState->mutex);
    }
    return rval;
}

int32_t CecReceiverSendActiveSourceCommand(bool trackHpdChange) NN_NOEXCEPT
{
    uint8_t     txBuffer[8];
    uint8_t     addressValue;
    uint32_t    rval;

    if(trackHpdChange)
    {
        CecLowLevelInterfaceInitializeHpdChange();
    }
    nn::os::LockMutex(&s_CecReceiveState.pOpState->mutex);
    addressValue =  (s_CecReceiveState.pOpState->logicalAddress << 4) | CecReceiveParamBroadcastLogicalAddress;
    s_CecReceiveState.pOpState->opStatus |= CecOpStatusFlagActiveSource;
    nn::os::UnlockMutex(&s_CecReceiveState.pOpState->mutex);
    txBuffer[CecBufferLogicalAddressIndex] = addressValue;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeActiveSource;
    txBuffer[CecBufferPayloadIndex + 0] = s_CecReceiveState.pOpState->physAddr[0];
    txBuffer[CecBufferPayloadIndex + 1] = s_CecReceiveState.pOpState->physAddr[1];
    rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFour, false);
    if(rval != CecNoError)
    {
        nn::os::LockMutex(&s_CecReceiveState.pOpState->mutex);
        s_CecReceiveState.pOpState->opStatus &= ~CecOpStatusFlagActiveSource;
        nn::os::UnlockMutex(&s_CecReceiveState.pOpState->mutex);
    }
    else if((trackHpdChange) && (CecLowLevelInterfaceHasHpdChanged()))
    {
        rval = CecReceiverHpdChangedError;
    }
    return rval;
}

int32_t CecReceiverSendImageViewOnCommand() NN_NOEXCEPT
{
    int32_t rval;
    uint8_t txBuffer[8];
    uint8_t logicalAddress;

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    nn::os::LockMutex(&s_CecReceiveState.pOpState->mutex);
    logicalAddress = s_CecReceiveState.pOpState->logicalAddress;
    nn::os::UnlockMutex(&s_CecReceiveState.pOpState->mutex);
    txBuffer[CecBufferLogicalAddressIndex] = (logicalAddress << 4) | CecReceiveParamTvLogicalAddress;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeImageViewOn;
    rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsTwo, true);
    return rval;
}

int32_t CecReceiverSendActiveSource() NN_NOEXCEPT
{
    int32_t rval;
    uint8_t txBuffer[8];
    uint8_t logicalAddress;

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    CecLowLevelInterfaceInitializeHpdChange();
    nn::os::LockMutex(&s_CecReceiveState.pOpState->mutex);
    s_CecReceiveState.pOpState->opStatus |= CecOpStatusFlagActiveSource;
    logicalAddress = s_CecReceiveState.pOpState->logicalAddress;
    nn::os::UnlockMutex(&s_CecReceiveState.pOpState->mutex);
    txBuffer[CecBufferLogicalAddressIndex] = (logicalAddress << 4) | CecReceiveParamTvLogicalAddress;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeImageViewOn;
    rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsTwo, true);
    if(rval == CecNoError)
    {
        uint32_t                delayInMilliseconds;
        nn::os::TimerEventType  timerEvent;

        nn::os::LockMutex(&s_CecReceiveState.pEdidVariation->mutex);
        delayInMilliseconds = (s_CecReceiveState.pEdidVariation->flags & EdidVariationFlagsSlowOtpSequenceBit) ?
                              CecReceiverParamLongQuietPeriodInMilliseconds :
                              CecReceiverParamShortQuietPeriodInMilliseconds;
        nn::os::UnlockMutex(&s_CecReceiveState.pEdidVariation->mutex);
        nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
        nn::os::StartOneShotTimerEvent(&timerEvent, nn::TimeSpan::FromMilliSeconds(delayInMilliseconds));
        nn::os::WaitTimerEvent(&timerEvent);
        nn::os::FinalizeTimerEvent(&timerEvent);
        txBuffer[CecBufferLogicalAddressIndex] = (logicalAddress << 4) | CecReceiveParamBroadcastLogicalAddress;
        txBuffer[CecBufferOpcodeIndex] = CecOpcodeActiveSource;
        txBuffer[CecBufferPayloadIndex + 0] = s_CecReceiveState.pOpState->physAddr[0];
        txBuffer[CecBufferPayloadIndex + 1] = s_CecReceiveState.pOpState->physAddr[1];
        rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFour, false);
    }
    if(rval != CecNoError)
    {
        nn::os::LockMutex(&s_CecReceiveState.pOpState->mutex);
        s_CecReceiveState.pOpState->opStatus &= ~CecOpStatusFlagActiveSource;
        nn::os::UnlockMutex(&s_CecReceiveState.pOpState->mutex);
    }
    else if(CecLowLevelInterfaceHasHpdChanged())
    {
        rval = CecReceiverHpdChangedError;
    }
    return rval;
}

int32_t CecReceiverSendDeviceMenuActivationCommand(bool activate) NN_NOEXCEPT
{
    uint8_t currentState;
    int32_t rval;
    uint8_t txBuffer[CecReceiveParamMaxMessageLength];
    bool    isActiveSource;

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    os::LockMutex(&s_CecReceiveState.pOpState->mutex);
    isActiveSource = !!(s_CecReceiveState.pOpState->opStatus & CecOpStatusFlagActiveSource);
    if(activate)
    {
        s_CecReceiveState.pOpState->opStatus |= CecOpStatusFlagMenuActiveState;
        currentState = CecMenuRequestTypeActivate;
    }
    else
    {
        s_CecReceiveState.pOpState->opStatus &= ~CecOpStatusFlagMenuActiveState;
        currentState = CecMenuRequestTypeDeactivate;
    }
    os::UnlockMutex(&s_CecReceiveState.pOpState->mutex);
    if(isActiveSource)
    {
        txBuffer[CecBufferLogicalAddressIndex] = (s_CecReceiveState.pOpState->logicalAddress << 4) |
                                                 CecReceiveParamTvLogicalAddress;
        txBuffer[CecBufferOpcodeIndex] = CecOpcodeMenuRequest;
        txBuffer[CecBufferPayloadIndex + 0] = currentState;
        rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsThree, true);
    }
    else
    {
        rval = CecReceiverInvalidStateError;
    }
    return rval;
}

int32_t CecReceiverSendRemoteControlCommand(bool start, uint8_t argument) NN_NOEXCEPT
{
    int32_t rval;
    uint8_t length;
    uint8_t txBuffer[CecReceiveParamMaxMessageLength];

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    txBuffer[CecBufferLogicalAddressIndex] = (s_CecReceiveState.pOpState->logicalAddress << 4) |
                                             CecReceiveParamTvLogicalAddress;
    txBuffer[CecBufferOpcodeIndex] = (start) ? CecOpcodeUserControlPressed : CecOpcodeUserControlReleased;
    length = (start) ? CecBufferElementCountIsThree : CecBufferElementCountIsTwo;
    txBuffer[CecBufferPayloadIndex + 0] = argument;
    rval = CecTransmitBestEffort(txBuffer, length, true);
    return rval;
}

int32_t CecReceiverSendOnScreenDisplay(uint8_t* string) NN_NOEXCEPT
{
    int32_t rval;
    uint8_t length;
    uint8_t index = 0;
    uint8_t txBuffer[CecReceiveParamMaxMessageLength];

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    txBuffer[CecBufferLogicalAddressIndex] = (s_CecReceiveState.pOpState->logicalAddress << 4) |
                                             CecReceiveParamTvLogicalAddress;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeSetOsdString;
    txBuffer[CecBufferPayloadIndex + 0] = CecDisplayControlDefaultTime;
    length = CecBufferPayloadIndex + 1;
    while((length < CecReceiveParamMaxMessageLength) && (string[index] != '\0'))
    {
        txBuffer[length] = string[index];
        length++;
        index++;
    }
    rval = CecTransmitBestEffort(txBuffer, length, true);
    return rval;
}

int32_t CecReceiverSendGoStandby(bool tvOnly) NN_NOEXCEPT
{
    int32_t rval;
    uint8_t txBuffer[4];
    uint8_t destinationAddress;

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    destinationAddress = (tvOnly == true) ? CecReceiveParamTvLogicalAddress : CecReceiveParamBroadcastLogicalAddress;
    txBuffer[CecBufferLogicalAddressIndex] = (s_CecReceiveState.pOpState->logicalAddress << 4) | destinationAddress;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeStandby;
    rval = CecTransmitBestEffort(txBuffer, CecBufferElementCountIsTwo, tvOnly);
    return rval;
}

int32_t CecReceiverInit() NN_NOEXCEPT
{
    int32_t rval;

    NN_SDK_ASSERT(!(s_CecReceiveState.flags & CecReceiveStateFlagInit));
    rval = CecReceiveBufferInit(s_CecRxRingbuffer, CecReceiverRingbufferSizeBytes);
    if(rval == CecNoError)
    {
        uint8_t index;

        for(index = 0; index < CecReceiveParamMaxOsdStringLength; index++)
        {
            s_CecReceiveState.osdString[index] = '\0';
        }
        nn::settings::fwdbg::GetSettingsItemValue(s_CecReceiveState.osdString,
                                                  CecReceiveParamMaxOsdStringLength, "productinfo", "cec_osd_name");
        NN_CEC_INFO("%s: osd: %s\n", NN_CURRENT_FUNCTION_NAME, s_CecReceiveState.osdString);
        nn::os::InitializeEvent(&s_CecReceiveState.startEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&s_CecReceiveState.doneEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeMutex(&s_CecReceiveState.mutex, false, 0);
        CecReceiveBufferSetListener(CecOpcodeRequestActiveSource, ReceiverRequestActiveSourceCallback,
                                    static_cast<void*>(&s_CecReceiveState));
        CecReceiveBufferSetListener(CecOpcodeGiveDeviceVendorId, ReceiverRequestVendorIdCallback,
                                    static_cast<void*>(&s_CecReceiveState));
        nn::os::CreateThread(&s_CecReceiveState.thread, ReceiverReceiveThread, &s_CecReceiveState,
                         s_CecReceiverStack, CecReceiverThreadStackSizeBytes,
                         NN_SYSTEM_THREAD_PRIORITY(cec, ReceiveThread));
        nn::os::SetThreadNamePointer(&s_CecReceiveState.thread, NN_SYSTEM_THREAD_NAME(cec, ReceiveThread));
        s_CecReceiveState.flags = CecReceiveStateFlagRunning;
        nn::os::StartThread(&s_CecReceiveState.thread);
        s_CecReceiveState.flags |= CecReceiveStateFlagInit;
        rval = CecNoError;
    }
    return rval;
}

void CecReceiverShutdown() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    CecReceiverStop();
    nn::os::LockMutex(&s_CecReceiveState.mutex);
    s_CecReceiveState.flags &= ~CecReceiveStateFlagRunning;
    nn::os::UnlockMutex(&s_CecReceiveState.mutex);
    nn::os::SignalEvent(&s_CecReceiveState.startEvent);
    nn::os::WaitEvent(&s_CecReceiveState.doneEvent);
    nn::os::SignalEvent(&s_CecReceiveState.startEvent);
    nn::os::WaitEvent(&s_CecReceiveState.doneEvent);
    nn::os::WaitThread(&s_CecReceiveState.thread);
    nn::os::DestroyThread(&s_CecReceiveState.thread);
    nn::os::FinalizeEvent(&s_CecReceiveState.startEvent);
    nn::os::FinalizeEvent(&s_CecReceiveState.doneEvent);
    nn::os::FinalizeMutex(&s_CecReceiveState.mutex);
    s_CecReceiveState.flags = 0;
    CecReceiveBufferShutdown();
}

int32_t CecReceiverStart(OperatingStateType* pOpState, EdidVariationType* pEdidVariation) NN_NOEXCEPT
{
    int32_t rval;

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    nn::os::LockMutex(&s_CecReceiveState.mutex);
    if(s_CecReceiveState.flags & CecReceiveStateFlagReceiving)
    {
        rval = CecReceiverInvalidStateError;
    }
    else
    {
        s_CecReceiveState.flags |= CecReceiveStateFlagReceiving;
        s_CecReceiveState.pOpState = pOpState;
        s_CecReceiveState.pEdidVariation = pEdidVariation;
        rval = CecNoError;
    }
    nn::os::UnlockMutex(&s_CecReceiveState.mutex);
    if(rval == CecNoError)
    {
        nn::os::SignalEvent(&s_CecReceiveState.startEvent);
    }
    return rval;
}

int32_t CecReceiverStop() NN_NOEXCEPT
{
    int32_t rval;

    NN_SDK_ASSERT(s_CecReceiveState.flags & CecReceiveStateFlagInit);
    nn::os::LockMutex(&s_CecReceiveState.mutex);
    if(!(s_CecReceiveState.flags & CecReceiveStateFlagReceiving))
    {
        rval = CecReceiverInvalidStateError;
    }
    else
    {
        s_CecReceiveState.flags &= ~CecReceiveStateFlagReceiving;
        CecReceiveBufferPurgeBuffer(true);
        rval = CecNoError;
    }
    nn::os::UnlockMutex(&s_CecReceiveState.mutex);
    if(rval == CecNoError)
    {
        nn::os::WaitEvent(&s_CecReceiveState.doneEvent);
    }
    return rval;
}

}
}   // namespace cec
}   // namespace nn
