﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/nn_SdkAssert.h>

#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_Receiver.h"
#include "cec_ReceiveBuffer.h"

namespace nn { namespace cec { namespace detail {

typedef struct {
    uint32_t                readIndex;
    uint32_t                writeIndex;
    uint32_t                modulus;
    uint8_t*                pBuffer;
    uint32_t                flags;
    nn::os::MutexType       ringbufferMutex;
    nn::os::TimerEventType  ringbufferEvent;
} RingbufferType;

typedef struct {
    uint16_t    hdrId;
    uint8_t     count;
    uint8_t     spare;
} ReceiveBufferEntryHeaderType;

namespace {

    const uint32_t ReceiveBufferStateFlagInitBit = (1 << 0);

    const int ReceiveBufferParamMaxListeners      = 2;
    const int ReceiveBufferParamRingbufferMinSize = 32;
    const int ReceiveBufferParamRingbufferMargin  = 1;

    const uint16_t ReceiveBufferEntryHeaderId       = 0xCEC0;
    const uint8_t  ReceiveBufferEntryHeaderSpareVal = 0x5A;

    RingbufferType s_RingbufferState;
    ReceiveListenerType s_ReceiveBufferListeners[ReceiveBufferParamMaxListeners];

    uint32_t    CecReceiveBufferGetEntriesInBytes(RingbufferType* pRingBuffer) NN_NOEXCEPT
    {
        uint32_t    rval;

        rval = pRingBuffer->writeIndex - pRingBuffer->readIndex;
        while(static_cast<int32_t>(rval) < 0)
        {
            rval += pRingBuffer->modulus;
        }
        return rval;
    }

}

int32_t CecReceiveBufferSetListener(uint8_t opcode, ListenCallbackFunctionPointerType pCallbackFunction, void* pCallbackContext) NN_NOEXCEPT
{
    int32_t     rval=CecReceiveBufferNoMemoryError;

    NN_SDK_ASSERT(s_RingbufferState.flags & ReceiveBufferStateFlagInitBit);
    if(pCallbackFunction == nullptr)
    {
        rval = CecReceiveBufferInvalidArgError;
    }
    else
    {
        for(uint32_t index=0; index<ReceiveBufferParamMaxListeners; index++)
        {
            if(s_ReceiveBufferListeners[index].pCallbackFunction == nullptr)
            {
                s_ReceiveBufferListeners[index].opcode = opcode;
                s_ReceiveBufferListeners[index].pCallbackFunction = pCallbackFunction;
                s_ReceiveBufferListeners[index].pCallbackContext = pCallbackContext;
                rval = CecNoError;
                break;
            }
        }
    }
    return rval;
}

int32_t CecReceiveBufferClearListener(uint8_t opcode, ListenCallbackFunctionPointerType pCallbackFunction) NN_NOEXCEPT
{
    int32_t     rval=CecReceiveBufferInvalidArgError;

    NN_SDK_ASSERT(s_RingbufferState.flags & ReceiveBufferStateFlagInitBit);
    if(pCallbackFunction != nullptr)
    {
        for(uint32_t index=0; index<ReceiveBufferParamMaxListeners; index++)
        {
            if((s_ReceiveBufferListeners[index].pCallbackFunction == pCallbackFunction) && (s_ReceiveBufferListeners[index].opcode == opcode))
            {
                s_ReceiveBufferListeners[index].opcode = 0;
                s_ReceiveBufferListeners[index].pCallbackFunction = nullptr;
                s_ReceiveBufferListeners[index].pCallbackContext = nullptr;
                rval = CecNoError;
                break;
            }
        }
    }
    return rval;
}

int32_t CecReceiveBufferInit(uint8_t* pBuffer, uint32_t Length) NN_NOEXCEPT
{
    int32_t     rval = CecReceiveBufferInvalidArgError;

    NN_SDK_ASSERT(!(s_RingbufferState.flags & ReceiveBufferStateFlagInitBit));
    if((pBuffer != nullptr) && (Length >= ReceiveBufferParamRingbufferMinSize))
    {
        for(uint32_t index=0; index<ReceiveBufferParamMaxListeners; index++)
        {
            s_ReceiveBufferListeners[index].opcode = 0;
            s_ReceiveBufferListeners[index].pCallbackFunction = nullptr;
            s_ReceiveBufferListeners[index].pCallbackContext = nullptr;
        }
        nn::os::InitializeMutex(&s_RingbufferState.ringbufferMutex, false, 0);
        nn::os::InitializeTimerEvent(&s_RingbufferState.ringbufferEvent, nn::os::EventClearMode_ManualClear);
        s_RingbufferState.readIndex = 0;
        s_RingbufferState.writeIndex = 0;
        s_RingbufferState.flags = ReceiveBufferStateFlagInitBit;
        s_RingbufferState.modulus = Length;
        s_RingbufferState.pBuffer = pBuffer;
        rval = CecNoError;
    }
    return rval;
}

void CecReceiveBufferShutdown() NN_NOEXCEPT
{
    RingbufferType* pRingBuffer;

    pRingBuffer = &s_RingbufferState;
    NN_SDK_ASSERT(pRingBuffer->flags & ReceiveBufferStateFlagInitBit);
    nn::os::FinalizeTimerEvent(&pRingBuffer->ringbufferEvent);
    nn::os::FinalizeMutex(&pRingBuffer->ringbufferMutex);
    pRingBuffer->flags = 0;
}

void CecReceiveBufferPurgeBuffer(bool sendSignalFlag) NN_NOEXCEPT
{
    RingbufferType* pRingBuffer;

    NN_CEC_INFO("%s send signal %d\n", NN_CURRENT_FUNCTION_NAME, sendSignalFlag);
    pRingBuffer = &s_RingbufferState;
    NN_SDK_ASSERT(pRingBuffer->flags & ReceiveBufferStateFlagInitBit);
    nn::os::LockMutex(&pRingBuffer->ringbufferMutex);
    pRingBuffer->readIndex = pRingBuffer->writeIndex;
    if(sendSignalFlag)
    {
        nn::os::SignalTimerEvent(&pRingBuffer->ringbufferEvent);
    }
    nn::os::UnlockMutex(&pRingBuffer->ringbufferMutex);
}

int32_t CecRxBufferEnqueuePacket(uint8_t* pBuffer, uint32_t count) NN_NOEXCEPT
{
    RingbufferType*                 pRingBuffer;
    int32_t                         rval = CecReceiveBufferInvalidArgError;
    uint32_t                        freeCount;
    ReceiveBufferEntryHeaderType    receiveBufferHeader;
    uint8_t*                        pReceiveBufferHeader;

    pRingBuffer = &s_RingbufferState;
    NN_SDK_ASSERT(pRingBuffer->flags & ReceiveBufferStateFlagInitBit);
    if(count <= CecReceiveParamMaxMessageLength)
    {
        if(count > 0)
        {
            pReceiveBufferHeader = reinterpret_cast<uint8_t*>(&receiveBufferHeader);
            receiveBufferHeader.hdrId = ReceiveBufferEntryHeaderId;
            receiveBufferHeader.count = count;
            receiveBufferHeader.spare = ReceiveBufferEntryHeaderSpareVal;
            nn::os::LockMutex(&pRingBuffer->ringbufferMutex);
            freeCount = pRingBuffer->modulus - CecReceiveBufferGetEntriesInBytes(pRingBuffer);
            freeCount -= ReceiveBufferParamRingbufferMargin;
            if(freeCount >= (count + sizeof(ReceiveBufferEntryHeaderType)))
            {
                if(count > 1)
                {
                    for(uint32_t loopIdx=0; loopIdx<ReceiveBufferParamMaxListeners; loopIdx++)
                    {
                        if(s_ReceiveBufferListeners[loopIdx].opcode == pBuffer[CecBufferOpcodeIndex])
                        {
                            if(s_ReceiveBufferListeners[loopIdx].pCallbackFunction != nullptr)
                            {
                                s_ReceiveBufferListeners[loopIdx].pCallbackFunction(pBuffer[CecBufferOpcodeIndex],
                                                               s_ReceiveBufferListeners[loopIdx].pCallbackContext);
                            }
                        }
                    }
                }
                for(uint8_t index=0; index<sizeof(ReceiveBufferEntryHeaderType); index++)
                {
                    pRingBuffer->pBuffer[pRingBuffer->writeIndex] = pReceiveBufferHeader[index];
                    pRingBuffer->writeIndex += 1;
                    if(pRingBuffer->writeIndex >= pRingBuffer->modulus)
                    {
                        pRingBuffer->writeIndex -= pRingBuffer->modulus;
                    }
                }
                for(uint32_t index=0; index<count; index++)
                {
                    pRingBuffer->pBuffer[pRingBuffer->writeIndex] = pBuffer[index];
                    pRingBuffer->writeIndex += 1;
                    if(pRingBuffer->writeIndex >= pRingBuffer->modulus)
                    {
                        pRingBuffer->writeIndex -= pRingBuffer->modulus;
                    }
                }
                rval = static_cast<int32_t>(count) + sizeof(ReceiveBufferEntryHeaderType);
            }
            else
            {
                rval = CecReceiveBufferNoMemoryError;
            }
            nn::os::UnlockMutex(&pRingBuffer->ringbufferMutex);
            if(rval > 0)
            {
                nn::os::SignalTimerEvent(&pRingBuffer->ringbufferEvent);
            }
        }
        else
        {
            rval = CecReceiveBufferInvalidArgError;
        }
    }
    return rval;
}

int32_t CecRxBufferDequeuePacket(uint8_t* pBuffer, uint32_t* pCount, int32_t waitUs) NN_NOEXCEPT
{
    RingbufferType*                 pRingBuffer;
    int32_t                         rval = CecReceiveBufferInvalidArgError;
    uint32_t                        count = 0;
    ReceiveBufferEntryHeaderType    receiveBufferHeader;
    uint8_t*                        pReceiveBufferHeader;

    pRingBuffer = &s_RingbufferState;
    NN_SDK_ASSERT(pRingBuffer->flags & ReceiveBufferStateFlagInitBit);
    if((pBuffer != nullptr) && (pCount != nullptr))
    {
        *pCount = 0;
        if(waitUs > 0)
        {
            nn::os::StartOneShotTimerEvent(&pRingBuffer->ringbufferEvent, nn::TimeSpan::FromMicroSeconds(waitUs));
        }
        nn::os::WaitTimerEvent(&pRingBuffer->ringbufferEvent);
        if(waitUs > 0)
        {
            nn::os::StopTimerEvent(&pRingBuffer->ringbufferEvent);
        }
        nn::os::LockMutex(&pRingBuffer->ringbufferMutex);
        count = CecReceiveBufferGetEntriesInBytes(pRingBuffer);
        if(count > sizeof(ReceiveBufferEntryHeaderType))
        {
            pReceiveBufferHeader = reinterpret_cast<uint8_t*>(&receiveBufferHeader);
            for(uint32_t index = 0; index<sizeof(ReceiveBufferEntryHeaderType); index++)
            {
                pReceiveBufferHeader[index] = pRingBuffer->pBuffer[pRingBuffer->readIndex];
                pRingBuffer->readIndex += 1;
                if(pRingBuffer->readIndex >= pRingBuffer->modulus)
                {
                    pRingBuffer->readIndex -= pRingBuffer->modulus;
                }
            }
            count -= sizeof(ReceiveBufferEntryHeaderType);
            if((receiveBufferHeader.hdrId == ReceiveBufferEntryHeaderId) &&
               (receiveBufferHeader.spare == ReceiveBufferEntryHeaderSpareVal) &&
               (receiveBufferHeader.count > 0) &&
               (static_cast<uint32_t>(receiveBufferHeader.count) <= count))
            {
                for(uint8_t index=0; index<receiveBufferHeader.count; index++)
                {
                    pBuffer[index] = pRingBuffer->pBuffer[pRingBuffer->readIndex];
                    pRingBuffer->readIndex += 1;
                    if(pRingBuffer->readIndex >= pRingBuffer->modulus)
                    {
                        pRingBuffer->readIndex -= pRingBuffer->modulus;
                    }
                }
                *pCount = static_cast<uint32_t>(receiveBufferHeader.count);
                count -= static_cast<uint32_t>(receiveBufferHeader.count);
                rval = CecNoError;
            }
            else
            {
                NN_CEC_INFO("%s: malformed header, purging buffer\n", NN_CURRENT_FUNCTION_NAME);
                pRingBuffer->readIndex = pRingBuffer->writeIndex;
                count = 0;
                rval = CecReceiveBufferInvalidDataError;
            }
        }
        else if(count > 0)
        {
            NN_CEC_INFO("%s: no header, purging buffer\n", NN_CURRENT_FUNCTION_NAME);
            pRingBuffer->readIndex = pRingBuffer->writeIndex;
            count = 0;
            rval = CecReceiveBufferInvalidDataError;
        }
        else
        {
            rval = CecReceiveBufferTimeoutError;
        }
        if(count == 0)
        {
            nn::os::ClearTimerEvent(&pRingBuffer->ringbufferEvent);
        }
        nn::os::UnlockMutex(&pRingBuffer->ringbufferMutex);
    }
    return rval;
}

}
}   // namespace cec
}   // namespace nn
