﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_ReceiveBuffer.h"
#include "cec_LowLevelInterface.h"
#include "cec_DisplayPortInterface.h"
#include "cec_Receiver.h"
#include "cec_TopLevelFiniteStateMachine.h"
#include "cec_DebounceFiniteStateMachine.h"
#include "cec_ManagerImpl.private.h"

namespace nn { namespace cec { namespace detail {

typedef int32_t (* LowLevelInitFunctionPointerType)(void*);
typedef int32_t (* LowLevelShutdownFunctionPointerType)(void);
typedef int32_t (* LowLevelStartFunctionPointerType)(void*);
typedef int32_t (* LowLevelStopFunctionPointerType)(void);
typedef int32_t (* LowLevelEnableFunctionPointerType)(bool);
typedef int32_t (* LowLevelGetPhysicalAddressFunctionPointerType)(uint8_t*);
typedef int32_t (* LowLevelSetLogicalAddressFunctionPointerType)(uint8_t);
typedef int32_t (* LowLevelReadDataFunctionPointerType)(uint8_t*, int*);
typedef int32_t (* LowLevelTransmitDataFunctionPointerType)(TransmitReturnValueType*, uint8_t*, uint8_t);
typedef int32_t (* LowLevelLastTrafficFunctionPointerType)(uint32_t*);
typedef int32_t (* LowLevelGetConnectionStatusFunctionPointerType)(uint32_t*, uint32_t*);
typedef int32_t (* LowLevelPrepareForStopFunctionPointerType)();

typedef struct {
    volatile uint32_t                               flags;
    uint32_t                                        sinkStatus;
    os::ThreadType                                  thread;
    os::EventType                                   pollEvent;
    os::MutexType                                   hpdMutex;
    LowLevelInitFunctionPointerType                 pInitFunction;
    LowLevelShutdownFunctionPointerType             pShutdownFunction;
    LowLevelStartFunctionPointerType                pStartFunction;
    LowLevelStopFunctionPointerType                 pStopFunction;
    LowLevelEnableFunctionPointerType               pEnableFunction;
    LowLevelGetPhysicalAddressFunctionPointerType   pGetPhysicalAddressFunction;
    LowLevelSetLogicalAddressFunctionPointerType    pSetLogicalAddressFunction;
    LowLevelReadDataFunctionPointerType             pReadDataFunction;
    LowLevelTransmitDataFunctionPointerType         pTransmitDataFunction;
    LowLevelLastTrafficFunctionPointerType          pLastTrafficFunction;
    LowLevelGetConnectionStatusFunctionPointerType  pGetConnectionStatusFunction;
    LowLevelPrepareForStopFunctionPointerType       pPrepareForStopFunction;
} LowLevelInterfaceStateType;

namespace {

    const uint32_t LowLevelInterfaceFlagInitBit         = (1 << 0);
    const uint32_t LowLevelInterfaceFlagStartedBit      = (1 << 1);
    const uint32_t LowLevelInterfaceFlagEnabledBit      = (1 << 2);
    const uint32_t LowLevelInterfaceFlagHpdChangedBit   = (1 << 3);
    const uint32_t LowLevelInterfaceFlagHpdStateBit     = (1 << 4);
    const uint32_t LowLevelInterfaceFlagPollEnableBit   = (1 << 5);

    const uint32_t LowLevelInterfaceParamCecReadPollingPeriodInMilliseconds = 50;

    const uint32_t LowLevelInterfaceParamNoSinkDevice = 0;
    const uint32_t LowLevelInterfaceParamSinkDeviceOff = 1;
    const uint32_t LowLevelInterfaceParamSinkDeviceOn = 2;

    const uint32_t LowLevelInterfaceParamHpdLow = 0;
    const uint32_t LowLevelInterfaceParamHpdHigh = 1;

    LowLevelInterfaceStateType s_CecLowLevelState;

    const size_t   LowLevelInterfacePollThreadStackSizeBytes = 8 * 1024;
    NN_ALIGNAS(os::MemoryPageSize) char s_LowLevelInterfacePollThreadStack[LowLevelInterfacePollThreadStackSizeBytes];

    void    CecLowLevelInterfaceReadPollLoop() NN_NOEXCEPT
    {
        int32_t rval;
        int     readByteCount;
        uint8_t readBuffer[CecReceiveParamMaxMessageLength];

        if(s_CecLowLevelState.flags & LowLevelInterfaceFlagPollEnableBit)
        {
            rval = s_CecLowLevelState.pReadDataFunction(readBuffer, &readByteCount);
            if((rval == 0) && (readByteCount > 0))
            {
                CecRxBufferEnqueuePacket(readBuffer, readByteCount);
            }
            os::SleepThread(TimeSpan::FromMilliSeconds(LowLevelInterfaceParamCecReadPollingPeriodInMilliseconds));
        }
    }

    void    CecLowLevelInterfacePollingThread(void* pThreadArgument) NN_NOEXCEPT
    {
        while(s_CecLowLevelState.flags & LowLevelInterfaceFlagPollEnableBit)
        {
            os::WaitEvent(&s_CecLowLevelState.pollEvent);
            CecLowLevelInterfaceReadPollLoop();
        }
    }

    void    CecLowLevelInterfaceTrackHpdChange(uint32_t hpdStatus) NN_NOEXCEPT
    {
        uint32_t    oldFlags;
        uint32_t    newFlags;

        os::LockMutex(&s_CecLowLevelState.hpdMutex);
        oldFlags = s_CecLowLevelState.flags & LowLevelInterfaceFlagHpdStateBit;
        if(hpdStatus == LowLevelInterfaceParamHpdLow)
        {
            s_CecLowLevelState.flags &= ~LowLevelInterfaceFlagHpdStateBit;
        }
        else
        {
            s_CecLowLevelState.flags |= LowLevelInterfaceFlagHpdStateBit;
        }
        newFlags = s_CecLowLevelState.flags & LowLevelInterfaceFlagHpdStateBit;
        if(oldFlags != newFlags)
        {
            s_CecLowLevelState.flags |= LowLevelInterfaceFlagHpdChangedBit;
        }
        os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
    }
}

int32_t CecLowLevelGetHpdState(bool* pOutHpdState) NN_NOEXCEPT
{
    int32_t rval = CecLowLevelInvalidInitError;

    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    NN_SDK_ASSERT(pOutHpdState);
    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit)
    {
        os::LockMutex(&s_CecLowLevelState.hpdMutex);
        *pOutHpdState = !!(s_CecLowLevelState.flags & LowLevelInterfaceFlagHpdStateBit);
        os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
        rval = CecNoError;
    }
    return rval;
}

void    CecLowLevelInterfaceInitializeCecPolling() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit);
    if(!(s_CecLowLevelState.flags & LowLevelInterfaceFlagPollEnableBit))
    {
        os::LockMutex(&s_CecLowLevelState.hpdMutex);
        NN_CEC_INFO("%s hpd %d sink %d\n", NN_CURRENT_FUNCTION_NAME,
                    !!(s_CecLowLevelState.flags & LowLevelInterfaceFlagHpdStateBit), s_CecLowLevelState.sinkStatus);
        if(!(s_CecLowLevelState.flags & LowLevelInterfaceFlagHpdStateBit) &&
           ((s_CecLowLevelState.sinkStatus == LowLevelInterfaceParamSinkDeviceOff) ||
            (s_CecLowLevelState.sinkStatus == LowLevelInterfaceParamSinkDeviceOn)))
        {
            os::SignalEvent(&s_CecLowLevelState.pollEvent);
        }
        else
        {
            os::ClearEvent(&s_CecLowLevelState.pollEvent);
        }
        os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
        s_CecLowLevelState.flags |= LowLevelInterfaceFlagPollEnableBit;
        os::CreateThread(&s_CecLowLevelState.thread, CecLowLevelInterfacePollingThread, &s_CecLowLevelState,
                         s_LowLevelInterfacePollThreadStack, LowLevelInterfacePollThreadStackSizeBytes,
                         NN_SYSTEM_THREAD_PRIORITY(cec, PollThread));
        os::SetThreadNamePointer(&s_CecLowLevelState.thread, NN_SYSTEM_THREAD_NAME(cec, PollThread));
        os::StartThread(&s_CecLowLevelState.thread);
    }
}

void    CecLowLevelInterfaceFinalizeCecPolling() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit);
    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagPollEnableBit)
    {
        s_CecLowLevelState.flags &= ~LowLevelInterfaceFlagPollEnableBit;
        os::SignalEvent(&s_CecLowLevelState.pollEvent);
        os::WaitThread(&s_CecLowLevelState.thread);
        os::DestroyThread(&s_CecLowLevelState.thread);
    }
}

void    CecLowLevelInterfaceCecCallback(void* pContext, uint8_t* buffer, uint32_t length) NN_NOEXCEPT
{
    CecRxBufferEnqueuePacket(buffer, length);
}

void    CecLowLevelInterfaceDcCallback(void* pContext, bool isConnected) NN_NOEXCEPT
{
    uint32_t    eventFlags;
    uint32_t    hpdStatus = LowLevelInterfaceParamHpdLow;
    int32_t     rval;

    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagEnabledBit)
    {
        os::LockMutex(&s_CecLowLevelState.hpdMutex);
        rval = s_CecLowLevelState.pGetConnectionStatusFunction(&hpdStatus, &s_CecLowLevelState.sinkStatus);
        os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
        if(rval != CecDisplayPortParamNoError)
        {
            hpdStatus = static_cast<uint32_t>(isConnected);
        }
        NN_CEC_INFO("%s: HPD %X sink %X rval %d\n", NN_CURRENT_FUNCTION_NAME, hpdStatus, s_CecLowLevelState.sinkStatus, rval);

        CecLowLevelInterfaceTrackHpdChange(hpdStatus);
        eventFlags = (isConnected == false) ? server::CecEventHpdDisconnect : server::CecEventHpdConnect;
        server::CecManagerImplOnEvent(eventFlags, 0, NULL);

        os::LockMutex(&s_CecLowLevelState.hpdMutex);
        if((s_CecLowLevelState.flags & LowLevelInterfaceFlagPollEnableBit) &&
           (hpdStatus == LowLevelInterfaceParamHpdLow) &&
           ((s_CecLowLevelState.sinkStatus == LowLevelInterfaceParamSinkDeviceOff) ||
            (s_CecLowLevelState.sinkStatus == LowLevelInterfaceParamSinkDeviceOn)))
        {
            os::SignalEvent(&s_CecLowLevelState.pollEvent);
        }
        else
        {
            os::ClearEvent(&s_CecLowLevelState.pollEvent);
        }
        os::UnlockMutex(&s_CecLowLevelState.hpdMutex);

        if(hpdStatus == LowLevelInterfaceParamHpdLow)
        {
            CecDebounceOnHpdLow();
            CecTopLevelOnHpd(false);
        }
        else
        {
            CecDebounceOnHpdHigh();
            CecTopLevelOnHpd(true);
        }
    }
}

int32_t CecLowLevelInterfaceInit(uint32_t interfaceType) NN_NOEXCEPT
{
    int32_t rval = CecNoError;

    NN_CEC_INFO("%s: entry\n", NN_CURRENT_FUNCTION_NAME);
    if(!(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit))
    {
        if(interfaceType == LowLevelInterfaceSelectDisplayPort)
        {
            os::InitializeEvent(&s_CecLowLevelState.pollEvent, false, os::EventClearMode_ManualClear);
            os::InitializeMutex(&s_CecLowLevelState.hpdMutex, false, 0);
            s_CecLowLevelState.pInitFunction = CecDisplayPortInit;
            s_CecLowLevelState.pShutdownFunction = CecDisplayPortShutdown;
            s_CecLowLevelState.pStartFunction = CecDisplayPortStart;
            s_CecLowLevelState.pStopFunction = CecDisplayPortStop;
            s_CecLowLevelState.pEnableFunction = CecDisplayPortEnable;
            s_CecLowLevelState.pGetPhysicalAddressFunction = CecDisplayPortGetPhysicalAddress;
            s_CecLowLevelState.pSetLogicalAddressFunction = CecDisplayPortSetLogicalAddress;
            s_CecLowLevelState.pReadDataFunction = CecDisplayPortReadData;
            s_CecLowLevelState.pTransmitDataFunction = CecDisplayPortTransmitData;
            s_CecLowLevelState.pLastTrafficFunction = CecDisplayPortLastTraffic;
            s_CecLowLevelState.pGetConnectionStatusFunction = CecDisplayPortGetConnectionStatus;
            s_CecLowLevelState.pPrepareForStopFunction = CecDisplayPortUnhookCallbacks;
            s_CecLowLevelState.flags = LowLevelInterfaceFlagInitBit;
        }
        else
        {
            rval = CecLowLevelNotSupportedError;
        }
    }
    NN_CEC_INFO("%s: leave --> %X\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
}

int32_t CecLowLevelInterfaceStart() NN_NOEXCEPT
{
    int32_t rval = CecNoError;

    NN_CEC_INFO("%s: entry\n", NN_CURRENT_FUNCTION_NAME);
    if((s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit) &&
       (!(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit)))
    {
        rval = s_CecLowLevelState.pInitFunction(&s_CecLowLevelState);
        if(rval == CecNoError)
        {
            rval = s_CecLowLevelState.pStartFunction(&s_CecLowLevelState);
            if(rval == CecNoError)
            {
                uint32_t    hpdStatus = LowLevelInterfaceParamHpdLow;

                s_CecLowLevelState.sinkStatus = LowLevelInterfaceParamNoSinkDevice;
                rval = s_CecLowLevelState.pGetConnectionStatusFunction(&hpdStatus, &s_CecLowLevelState.sinkStatus);
                NN_CEC_INFO("%s HPD %d sink %d (rval %X)\n", NN_CURRENT_FUNCTION_NAME, hpdStatus, s_CecLowLevelState.sinkStatus, rval);
                if(rval == CecDisplayPortParamNoError)
                {
                    os::LockMutex(&s_CecLowLevelState.hpdMutex);
                    s_CecLowLevelState.flags &= ~LowLevelInterfaceFlagHpdStateBit;
                    if(hpdStatus == LowLevelInterfaceParamHpdHigh)
                    {
                        s_CecLowLevelState.flags |= LowLevelInterfaceFlagHpdStateBit;
                    }
                    os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
                }
                s_CecLowLevelState.flags |= LowLevelInterfaceFlagStartedBit;
            }
            else
            {
                s_CecLowLevelState.pShutdownFunction();
            }
        }
    }
    NN_CEC_INFO("%s: leave --> %X\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
}

int32_t CecLowLevelInterfacePrepareForStop() NN_NOEXCEPT
{
    int32_t rval = CecNoError;

    if((s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit) &&
       (s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit))
    {
        rval = s_CecLowLevelState.pPrepareForStopFunction();
    }
    return rval;
}

int32_t CecLowLevelInterfaceStop() NN_NOEXCEPT
{
    if((s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit) &&
       (s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit))
    {
        s_CecLowLevelState.flags &= ~LowLevelInterfaceFlagStartedBit;
        s_CecLowLevelState.pStopFunction();
        s_CecLowLevelState.pShutdownFunction();
    }
    return CecNoError;
}

int32_t CecLowLevelInterfaceShutdown() NN_NOEXCEPT
{
    int32_t rval = CecNoError;

    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit)
    {
        if(rval == CecNoError)
        {
            os::FinalizeMutex(&s_CecLowLevelState.hpdMutex);
            os::FinalizeEvent(&s_CecLowLevelState.pollEvent);
            s_CecLowLevelState.flags = 0;
        }
    }
    return rval;
}

void    CecLowLevelInterfaceInitializeHpdChange() NN_NOEXCEPT
{
    os::LockMutex(&s_CecLowLevelState.hpdMutex);
    s_CecLowLevelState.flags &= ~LowLevelInterfaceFlagHpdChangedBit;
    os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
}

bool    CecLowLevelInterfaceHasHpdChanged() NN_NOEXCEPT
{
    bool    rval;

    os::LockMutex(&s_CecLowLevelState.hpdMutex);
    rval = !!(s_CecLowLevelState.flags & LowLevelInterfaceFlagHpdChangedBit);
    os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
    return rval;
}

int32_t CecLowLevelInterfaceGetConnectionStatus(uint32_t* pOutHpdStatus, uint32_t* pOutSinkStatus) NN_NOEXCEPT
{
    int32_t rval;

    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    rval = s_CecLowLevelState.pGetConnectionStatusFunction(pOutHpdStatus, pOutSinkStatus);
    if(rval == CecNoError)
    {
        os::LockMutex(&s_CecLowLevelState.hpdMutex);
        s_CecLowLevelState.sinkStatus = *pOutSinkStatus;
        s_CecLowLevelState.flags &= ~LowLevelInterfaceFlagHpdStateBit;
        if(*pOutHpdStatus == LowLevelInterfaceParamHpdHigh)
        {
            s_CecLowLevelState.flags |= LowLevelInterfaceFlagHpdStateBit;
        }
        os::UnlockMutex(&s_CecLowLevelState.hpdMutex);
    }
    return rval;
}

int32_t CecLowLevelInterfaceLastTraffic(uint32_t* pDeltaUs) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit);

    return(s_CecLowLevelState.pLastTrafficFunction(pDeltaUs));
}

int32_t CecLowLevelInterfaceEnable(bool enable) NN_NOEXCEPT
{
    int32_t rval=CecLowLevelInvalidInitError;

    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit)
    {
        rval = s_CecLowLevelState.pEnableFunction(enable);
        if(rval == CecDisplayPortParamNoError)
        {
            if(enable)
            {
                s_CecLowLevelState.flags |= LowLevelInterfaceFlagEnabledBit;
            }
            else
            {
                s_CecLowLevelState.flags &= ~LowLevelInterfaceFlagEnabledBit;
            }
        }
    }
    return rval;
}

int32_t CecLowLevelInterfaceGetPhysicalAddress(uint8_t* pPhysicalAddress) NN_NOEXCEPT
{
    int32_t rval=CecLowLevelInvalidInitError;

    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit)
    {
        rval = s_CecLowLevelState.pGetPhysicalAddressFunction(pPhysicalAddress);
    }
    return rval;
}

int32_t CecLowLevelInterfaceSetLogicalAddress(uint8_t logicalAddress) NN_NOEXCEPT
{
    int32_t rval=CecLowLevelInvalidInitError;

    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit)
    {
        rval = s_CecLowLevelState.pSetLogicalAddressFunction(logicalAddress);
    }
    return rval;
}

int32_t CecLowLevelInterfaceReadData(uint8_t* pReadBuffer, int* pReadByteCount) NN_NOEXCEPT
{
    int32_t rval=CecLowLevelInvalidInitError;

    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit)
    {
        rval = s_CecLowLevelState.pReadDataFunction(pReadBuffer, pReadByteCount);
    }
    return rval;
}

int32_t CecLowLevelInterfaceTransmitData(TransmitReturnValueType* pTransmitReturnValue,
                                         uint8_t* pData, uint8_t count) NN_NOEXCEPT
{
    int32_t rval=CecLowLevelInvalidInitError;

    NN_SDK_ASSERT(s_CecLowLevelState.flags & LowLevelInterfaceFlagInitBit);
    if(s_CecLowLevelState.flags & LowLevelInterfaceFlagStartedBit)
    {
        s_CecLowLevelState.pTransmitDataFunction(pTransmitReturnValue, pData, count);
        rval = CecNoError;
    }
    return rval;
}

}
}   // namespace cec
}   // namespace nn
