﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SdkAssert.h>

#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_FiniteStateMachine.h"
#include "cec_DebounceFiniteStateMachine.h"

namespace nn { namespace cec { namespace detail {

    struct DebounceStateType
    {
        nn::os::MutexType       finiteStateMachineMutex;
        int64_t                 debounceStartTick;
        uint32_t                flags;
        FiniteStateMachineType  fsmState;
    };

    namespace {
        const uint32_t          DebounceStateFlagInit       (1 << 0);
        const uint32_t          DebounceStateFlagInDebounce (1 << 1);

        /* this should be an enum */
        const FiniteStateMachineStateType DebounceStateInit = 0;
        const FiniteStateMachineStateType DebounceStateOnOtp = 1;
        const FiniteStateMachineStateType DebounceStateOnHpdHigh = 2;
        const FiniteStateMachineStateType DebounceStateTvOn = 3;
        const FiniteStateMachineStateType DebounceStateOnStandby = 4;
        const FiniteStateMachineStateType DebounceStateInDebounce = 5;
        const FiniteStateMachineStateType DebounceStateNumber = 6;

        /* this should be an enum */
        const FiniteStateMachineInputType DebounceInputOtpOk = 0;
        const FiniteStateMachineInputType DebounceInputReset = 1;
        const FiniteStateMachineInputType DebounceInputStandbyIndicated = 2;
        const FiniteStateMachineInputType DebounceInputHpdLow = 3;
        const FiniteStateMachineInputType DebounceInputHpdHigh = 4;
        const FiniteStateMachineInputType DebounceInputInDebounce = 5;
        const FiniteStateMachineInputType DebounceInputNotInDebounce = 6;
        const FiniteStateMachineInputType DebounceInputNumber = 7;

        /* this should be an enum */
        const FiniteStateMachineActionValueType DebounceActionStartTimer = 0;
        const FiniteStateMachineActionValueType DebounceActionSetDebounceFlag = 1;
        const FiniteStateMachineActionValueType DebounceActionClearDebounceFlag = 2;
        const FiniteStateMachineActionValueType DebounceActionNumber = 3;

        const FiniteStateMachineStateType s_DebounceStateMatrix[DebounceStateNumber * DebounceInputNumber] =
        {
            DebounceStateOnOtp, DebounceStateInit, DebounceStateInit, DebounceStateInit, DebounceStateOnHpdHigh, DebounceStateInit, DebounceStateInit,
            DebounceStateOnOtp, DebounceStateInit, DebounceStateInit, DebounceStateOnOtp, DebounceStateTvOn, DebounceStateOnOtp, DebounceStateOnOtp,
            DebounceStateTvOn, DebounceStateInit, DebounceStateInit, DebounceStateOnHpdHigh, DebounceStateOnHpdHigh, DebounceStateOnHpdHigh, DebounceStateOnHpdHigh,
            DebounceStateTvOn, DebounceStateInit, DebounceStateOnStandby, DebounceStateTvOn, DebounceStateTvOn, DebounceStateTvOn, DebounceStateTvOn,
            DebounceStateTvOn, DebounceStateInit, DebounceStateOnStandby, DebounceStateInDebounce, DebounceStateOnStandby, DebounceStateOnStandby, DebounceStateOnStandby,
            DebounceStateTvOn, DebounceStateInit, DebounceStateInDebounce, DebounceStateInDebounce, DebounceStateTvOn, DebounceStateInDebounce, DebounceStateTvOn
        };

        const FiniteStateMachineActionType s_DebounceActionMatrix[DebounceStateNumber * DebounceInputNumber] =
        {
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},

            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},

            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},

            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},

            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{DebounceActionStartTimer, DebounceActionClearDebounceFlag, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},

            {{DebounceActionClearDebounceFlag, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{DebounceActionClearDebounceFlag, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{DebounceActionClearDebounceFlag, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{DebounceActionSetDebounceFlag, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
            {{DebounceActionClearDebounceFlag, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}}
        };

        void DebounceActionSetTimer(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
        {
            DebounceStateType*  pState;

            pState = static_cast<DebounceStateType*>(pContext);
            if(pState != nullptr)
            {
                pState->debounceStartTick = nn::os::GetSystemTick().GetInt64Value();
            }
        }

        void DebounceActionSetDebounce(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
        {
            DebounceStateType*  pState;

            pState = static_cast<DebounceStateType*>(pContext);
            if(pState != nullptr)
            {
                pState->flags |= DebounceStateFlagInDebounce;
            }
        }

        void DebounceActionClearDebounce(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
        {
            DebounceStateType*  pState;

            pState = static_cast<DebounceStateType*>(pContext);
            if(pState != nullptr)
            {
                pState->flags &= ~DebounceStateFlagInDebounce;
            }
        }

        const FiniteStateMachineActionHandlerFunctionPointerType s_DebounceActionHandlers[DebounceActionNumber] =
        {
            DebounceActionSetTimer,
            DebounceActionSetDebounce,
            DebounceActionClearDebounce
        };

        DebounceStateType   s_DebounceState;
        uint64_t            DebounceParamDebouncePeriodInMilliseconds = 1500;

        void InitializeDebounceFiniteStateMachine() NN_NOEXCEPT
        {
            nn::os::InitializeMutex(&s_DebounceState.finiteStateMachineMutex, false, 0);
            CecFiniteStateMachineInit(&s_DebounceState.fsmState,
                                      DebounceStateNumber,
                                      DebounceActionNumber,
                                      DebounceInputNumber,
                                      s_DebounceStateMatrix,
                                      s_DebounceActionMatrix,
                                      s_DebounceActionHandlers,
                                      static_cast<void*>(&s_DebounceState));
        }

        void StepDebounceFiniteStateMachine(FiniteStateMachineInputType fsmInput, void* pExtraArgs) NN_NOEXCEPT
        {
            nn::os::LockMutex(&s_DebounceState.finiteStateMachineMutex);
            NN_CEC_INFO("%s On Entry: state %d input %d\n",
                        NN_CURRENT_FUNCTION_NAME, s_DebounceState.fsmState.fsmState, fsmInput);
            CecFiniteStateMachineStep(&s_DebounceState.fsmState, fsmInput, pExtraArgs);
            NN_CEC_INFO("%s Exit: state %d in debounce %d\n",
                        NN_CURRENT_FUNCTION_NAME, s_DebounceState.fsmState.fsmState,
                        !!(s_DebounceState.flags & DebounceStateFlagInDebounce));
            nn::os::UnlockMutex(&s_DebounceState.finiteStateMachineMutex);
        }

        void FinalizeDebounceFiniteStateMachine() NN_NOEXCEPT
        {
            StepDebounceFiniteStateMachine(DebounceInputReset, nullptr);
            nn::os::FinalizeMutex(&s_DebounceState.finiteStateMachineMutex);
        }
    }

    void CecDebounceInit() NN_NOEXCEPT
    {
        if(!(s_DebounceState.flags & DebounceStateFlagInit))
        {
            InitializeDebounceFiniteStateMachine();
            s_DebounceState.flags |= DebounceStateFlagInit;
        }
    }

    void CecDebounceFinalize() NN_NOEXCEPT
    {
        if(s_DebounceState.flags & DebounceStateFlagInit)
        {
            FinalizeDebounceFiniteStateMachine();
            s_DebounceState.flags &= ~DebounceStateFlagInit;
        }
    }

    void CecDebounceOnOtp() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(s_DebounceState.flags & DebounceStateFlagInit);
        StepDebounceFiniteStateMachine(DebounceInputOtpOk, nullptr);
    }

    void CecDebounceOnReset() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(s_DebounceState.flags & DebounceStateFlagInit);
        StepDebounceFiniteStateMachine(DebounceInputReset, nullptr);
    }

    void CecDebounceOnStandby() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(s_DebounceState.flags & DebounceStateFlagInit);
        StepDebounceFiniteStateMachine(DebounceInputStandbyIndicated, nullptr);
    }

    void CecDebounceOnHpdLow() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(s_DebounceState.flags & DebounceStateFlagInit);
        StepDebounceFiniteStateMachine(DebounceInputHpdLow, nullptr);
    }

    void CecDebounceOnHpdHigh() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(s_DebounceState.flags & DebounceStateFlagInit);
        StepDebounceFiniteStateMachine(DebounceInputHpdHigh, nullptr);
    }

    void CecDebounceOnRoutingChange(bool isThisAddress) NN_NOEXCEPT
    {
        FiniteStateMachineInputType fsmInput;

        NN_SDK_ASSERT(s_DebounceState.flags & DebounceStateFlagInit);
        fsmInput = (isThisAddress) ? DebounceInputNotInDebounce : DebounceInputStandbyIndicated;
        StepDebounceFiniteStateMachine(fsmInput, nullptr);
    }

    bool CecDebounceIsInDebouncePeriod() NN_NOEXCEPT
    {
        int64_t                     deltaTicks;
        uint64_t                    deltaInMilliseconds;
        bool                        rval;
        FiniteStateMachineInputType fsmInput;

        NN_SDK_ASSERT(s_DebounceState.flags & DebounceStateFlagInit);
        nn::os::LockMutex(&s_DebounceState.finiteStateMachineMutex);
        deltaTicks = nn::os::GetSystemTick().GetInt64Value() - s_DebounceState.debounceStartTick;
        deltaInMilliseconds = nn::os::ConvertToTimeSpan(nn::os::Tick(deltaTicks)).GetMilliSeconds();
        fsmInput = (deltaInMilliseconds > DebounceParamDebouncePeriodInMilliseconds) ?
                   DebounceInputNotInDebounce : DebounceInputInDebounce;
        NN_CEC_INFO("%s On Entry: state %d input %d delta t %d\n",
                    NN_CURRENT_FUNCTION_NAME, s_DebounceState.fsmState.fsmState, fsmInput, deltaInMilliseconds);
        CecFiniteStateMachineStep(&s_DebounceState.fsmState, fsmInput, nullptr);
        rval = !!(s_DebounceState.flags & DebounceStateFlagInDebounce);
        NN_CEC_INFO("%s Exit: state %d in debounce %d\n",
                    NN_CURRENT_FUNCTION_NAME, s_DebounceState.fsmState.fsmState, rval);
        nn::os::UnlockMutex(&s_DebounceState.finiteStateMachineMutex);
        return rval;
    }

}}}
