﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief
 *
 * @details
 */

#include "cdmsc_PrivateIncludes.h"

#include "cdmsc_Bot.h"
#include "cdmsc_LogicalUnit.h"

namespace nn {
namespace cdmsc {
namespace driver {

// Forward declaration
class Driver;

class Device
{
public:
    Device(nn::usb::Host&                  usbHost,
           nn::usb::InterfaceQueryOutput  *pProfile,
           Driver&                         driver,
           uint32_t                        sequence);
    ~Device();

    Result Initialize() NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;

    Result Detach() NN_NOEXCEPT;

    Result BotRead(uint8_t lun, void *pCmd, uint8_t cmdLength,
                   void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT;
    Result BotWrite(uint8_t lun, void *pCmd, uint8_t cmdLength,
                    const void *buffer, uint32_t size, uint32_t *pOutXferred) NN_NOEXCEPT;

    void   RegisterKeepAliveTimer(nn::os::MultiWaitHolderType& timer) NN_NOEXCEPT;
    void   UnregisterKeepAliveTimer(nn::os::MultiWaitHolderType& timer) NN_NOEXCEPT;

    void   Lock() NN_NOEXCEPT;
    void   Unlock() NN_NOEXCEPT;

    void   ArmWatchdogTimer(uint32_t timeoutInMs) NN_NOEXCEPT;
    void   DisarmWatchdogTimer() NN_NOEXCEPT;
    void   FireWatchdogTimer() NN_NOEXCEPT;

    void* operator new(size_t size) NN_NOEXCEPT;
    void  operator delete(void* p, size_t size) NN_NOEXCEPT;

private:
    void Thread();
    static void ThreadEntry(void* pThis)
    {
        ((Device*)pThis)->Thread();
    }

    Result CreateLogicalUnits() NN_NOEXCEPT;
    void   DestroyLogicalUnits() NN_NOEXCEPT;

private:
    nn::usb::Host&                         m_UsbHost;
    nn::usb::HostInterface                 m_UsbInterface;
    nn::usb::InterfaceQueryOutput          m_Profile;

    Driver&                                m_Driver;
    Bot                                    m_Bot;

    const uint32_t                         m_Sequence;
    uint8_t                                m_MaxLun;
    LogicalUnit                           *m_pLogicalUnit[MaxLun + 1];

    nn::os::SystemEventType               *m_pStateChangeEvent;
    nn::os::MultiWaitHolderType            m_StateChangeEventHolder;

    // Transaction Watchdog
    nn::os::TimerEvent                     m_WatchdogTimer;
    nn::os::MultiWaitHolderType            m_WatchdogTimerHolder;

    nn::os::Mutex                          m_Mutex;

    // device thread
    static const size_t                    StackSize = 1024 * 16;
    bool                                   m_IsThreadRunning;
    nn::os::ThreadType                     m_Thread;
    void                                  *m_pThreadStack;
    nn::os::MultiWaitType                  m_MultiWait;
    nn::os::Event                          m_BreakEvent;
    nn::os::MultiWaitHolderType            m_BreakEventHolder;
};

} // driver
} // cdmsc
} // nn
