﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "cdmsc_PrivateIncludes.h"

/**
 * @file
 * @brief
 *
 * @details
 */

namespace nn {
namespace cdmsc {
namespace driver {

/*
 * Ref [BOT r1.0] 5.1
 *
 * Command Block Wrapper (CBW)
 */
struct Cbw
{
    uint32_t dCBWSignature;
    uint32_t dCBWTag;
    uint32_t dCBWDataTransferLength;
    uint8_t  bmCBWFlags;
    uint8_t  bCBWLUN;
    uint8_t  bCBWCBLength;
    uint8_t  CBWCB[16];
} NN_CDMSC_PACKED;

/*
 * Ref [BOT r1.0] 5.2
 *
 * Command Status Wrapper (CSW)
 */
struct Csw
{
    uint32_t dCSWSignature;
    uint32_t dCSWTag;
    uint32_t dCSWDataResidue;
    uint8_t  bCSWStatus;
} NN_CDMSC_PACKED;

const uint8_t  MaxLun       = 15;
const uint32_t CbwSignature = 0x43425355;
const uint32_t CswSignature = 0x53425355;

enum CbwFlags
{
    CbwFlags_DataOut = 0 << 7,
    CbwFlags_DataIn  = 1 << 7
};

/*
 * Ref [BOT r1.0] Table 5.3
 */
enum CswStatus
{
    CswStatus_CommandPassed = 0,
    CswStatus_CommandFailed = 1,
    CswStatus_PhaseError    = 2
};

class Bot
{
public:
    Bot()
        : m_pInterface(nullptr)
        , m_CbwTag(0)
        , m_pBuffer(nullptr)
    {
    }

    ~Bot()
    {
    }

    Result Initialize(nn::usb::HostInterface *pInterface);
    Result Finalize();

    Result Read(uint8_t lun, void *pCmd, uint8_t cmdLength,
                void *buffer, uint32_t size, uint32_t *pOutXferred);
    Result Write(uint8_t lun, void *pCmd, uint8_t cmdLength,
                 const void *buffer, uint32_t size, uint32_t *pOutXferred);
    Result GetMaxLun(uint8_t *pMaxLun);

private:
    Result DoCommand(Cbw& cbw);
    Result DoDataIn(void *buffer, uint32_t size);
    Result DoDataOut(void *buffer, uint32_t size);
    Result DoStatus(Cbw& cbw, Csw& csw);
    Result Execute(Cbw& cbw, void *buffer, Csw& csw);

    Result StorageReset();
    Result ClearBulkIn();
    Result ClearBulkOut();
    Result ResetRecovery();

private:
    // Max of dCBWDataTransferLength
    const uint32_t               MaxXferSize = nn::usb::HsLimitMaxUrbTransferSize;

    nn::usb::HostInterface      *m_pInterface;
    nn::usb::HostEndpoint        m_BulkIn;
    nn::usb::HostEndpoint        m_BulkOut;

    nn::usb::InterfaceProfile    m_IfProfile;

    uint32_t                     m_CbwTag;

    static const size_t          DmaBufferSize = 4096;
    uint8_t                     *m_pBuffer;
};

} // driver
} // cdmsc
} // nn
