﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdmsc_Bot.h"

namespace nn {
namespace cdmsc {
namespace driver {

Result
Bot::Initialize(nn::usb::HostInterface *pInterface)
{
    Result result = ResultSuccess();

    nn::usb::UsbEndpointDescriptor *pBulkInDesc  = nullptr;
    nn::usb::UsbEndpointDescriptor *pBulkOutDesc = nullptr;

    m_pInterface = pInterface;

    result = m_pInterface->GetInterface(&m_IfProfile);
    if (result.IsFailure())
    {
        goto bail1;
    }

    for (auto& desc : m_IfProfile.epInDesc)
    {
        if (nn::usb::UsbEndpointIsBulk(&desc))
        {
            pBulkInDesc = &desc;
            break;
        }
    }

    for (auto& desc : m_IfProfile.epOutDesc)
    {
        if (nn::usb::UsbEndpointIsBulk(&desc))
        {
            pBulkOutDesc = &desc;
            break;
        }
    }

    if (pBulkInDesc == nullptr || pBulkOutDesc == nullptr)
    {
        result = ResultInvalidInterface();
        goto bail1;
    }

    result = m_BulkIn.Initialize(m_pInterface, pBulkInDesc, 1, MaxXferSize);
    if (result.IsFailure())
    {
        goto bail1;
    }

    result = m_BulkOut.Initialize(m_pInterface, pBulkOutDesc, 1, MaxXferSize);
    if (result.IsFailure())
    {
        goto bail2;
    }

    m_pBuffer = reinterpret_cast<uint8_t*>(
        detail::Allocate(nn::usb::HwLimitDmaBufferAlignmentSize, DmaBufferSize)
    );
    if (m_pBuffer == nullptr)
    {
        result = ResultMemoryAllocError();
        goto bail3;
    }

    return ResultSuccess();

bail3:
    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_BulkOut.Finalize());
bail2:
    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_BulkIn.Finalize());
bail1:
    return result;
}

Result
Bot::Finalize()
{
    Result result = ResultSuccess();

    detail::Deallocate(m_pBuffer, DmaBufferSize);
    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_BulkOut.Finalize());
    NN_CDMSC_ABORT_UNLESS_SUCCESS(m_BulkIn.Finalize());

    std::memset(&m_IfProfile, 0, sizeof(m_IfProfile));

    m_pInterface = nullptr;

    return result;
}

/*
 * Execute one BOT Read transaction with retry
 *
 * ResultSuccess
 *   Everything is good.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *   BOT transaction error, and we are failed to bring it back to sync. Device
 *   reset is inevitable.
 *
 * ResultBotFailed
 *   BOT transaction failed, but we are still in sync. BOT retry is Okay.
 *
 * ResultCommandProtocolError
 *   Command Protocol error.
 */
Result
Bot::Read(uint8_t lun, void *pCmd, uint8_t cmdLength,
          void *buffer, uint32_t size, uint32_t *pOutXferred)
{
    Result result = ResultSuccess();

    Cbw cbw = {
        .dCBWSignature          = CbwSignature,
        .dCBWTag                = m_CbwTag,
        .dCBWDataTransferLength = size,
        .bmCBWFlags             = CbwFlags_DataIn,
        .bCBWLUN                = lun,
        .bCBWCBLength           = cmdLength,
    };
    Csw csw = { 0, 0, 0, 0 };

    NN_CDMSC_ABORT_UNLESS(cmdLength >= 1 && cmdLength <= 16);

    std::memcpy(&cbw.CBWCB, pCmd, cbw.bCBWCBLength);

    for (int i = 0; i < BotTransactionRetryMax; i++)
    {
        NN_CDMSC_DO(Execute(cbw, buffer, csw));

        if (ResultBotFailed::Includes(result))
        {
            // Retry
            continue;
        }

        break;
    }

    if (result.IsSuccess())
    {
        if (size >= csw.dCSWDataResidue)
        {
            *pOutXferred = size - csw.dCSWDataResidue;
        }
        else
        {
            *pOutXferred = 0;
        }

        if (csw.bCSWStatus == CswStatus_CommandFailed)
        {
            result = ResultCommandProtocolError();
        }
    }

    return result;
}

/*
 * Execute one BOT Write transaction with retry
 *
 * ResultSuccess
 *   Everything is good.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *   BOT transaction error, and we are failed to bring it back to sync. Device
 *   reset is inevitable.
 *
 * ResultBotFailed
 *   BOT transaction failed, but we are still in sync. BOT retry is Okay.
 *
 * ResultCommandProtocolError
 *   Command Protocol error.
 */
Result
Bot::Write(uint8_t lun, void *pCmd, uint8_t cmdLength,
           const void *buffer, uint32_t size, uint32_t *pOutXferred)
{
    Result result = ResultSuccess();

    Cbw cbw = {
        .dCBWSignature          = CbwSignature,
        .dCBWTag                = m_CbwTag,
        .dCBWDataTransferLength = size,
        .bmCBWFlags             = CbwFlags_DataOut,
        .bCBWLUN                = lun,
        .bCBWCBLength           = cmdLength,
    };
    Csw csw = { 0, 0, 0, 0 };

    NN_CDMSC_ABORT_UNLESS(cmdLength >= 1 && cmdLength <= 16);

    std::memcpy(&cbw.CBWCB, pCmd, cbw.bCBWCBLength);

    for (int i = 0; i < BotTransactionRetryMax; i++)
    {
        NN_CDMSC_DO(Execute(cbw, const_cast<void*>(buffer), csw));

        if (ResultBotFailed::Includes(result))
        {
            // Retry
            continue;
        }

        break;
    }

    if (result.IsSuccess())
    {
        if (size >= csw.dCSWDataResidue)
        {
            *pOutXferred = size - csw.dCSWDataResidue;
        }
        else
        {
            *pOutXferred = 0;
        }

        if (csw.bCSWStatus == CswStatus_CommandFailed)
        {
            result = ResultCommandProtocolError();
        }
    }

    return result;
}

/*
 * BOT Command Phase
 *
 * ResultSuccess
 *   Everything is good.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *   BOT transaction error, and we are failed to bring it back to sync. Device
 *   reset is inevitable.
 *
 * ResultUsbTransactionError
 *   USB transaction failed, we can retry the command phase.
 *
 * ResultBotFailed
 *   BOT transaction failed, but we are still in sync. BOT retry is Okay.
 */
Result
Bot::DoCommand(Cbw& cbw)
{
    Result result = ResultSuccess();
    size_t xferredSize = 0;

    std::memcpy(m_pBuffer, &cbw, sizeof(cbw));

    NN_CDMSC_DO(m_BulkOut.PostBuffer(&xferredSize, m_pBuffer, sizeof(cbw)));

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (nn::usb::ResultStalled::Includes(result))
    {
        /*
         * Ref [BOT r1.0] 5.3.1
         */
        NN_CDMSC_WARN("BOT Command Phase Stall!\n");
        NN_CDMSC_DO(ResetRecovery());
        if (result.IsSuccess())
        {
            // it's still an error
            result = ResultBotFailed();
        }
    }
    else if (result.IsFailure()) // USB layer error
    {
        NN_CDMSC_WARN("BOT Command Phase Error %d:%d\n",
                      result.GetModule(), result.GetDescription());
        result = ResultUsbTransactionError();
    }

    return result;
}

/*
 * BOT Data Phase (In)
 *
 * ResultSuccess
 *   Everything is good.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *   BOT transaction error, and we are failed to bring it back to sync. Device
 *   reset is inevitable.
 *
 * ResultUsbTransactionError
 *   USB transaction failed, we can retry the status phase.
 */
Result
Bot::DoDataIn(void *buffer, uint32_t size)
{
    Result result = ResultSuccess();
    size_t xferredSize = 0;

    NN_CDMSC_DO(
        m_BulkIn.PostBuffer(&xferredSize, buffer, size)
    );

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (nn::usb::ResultStalled::Includes(result))
    {
        /*
         * Ref [BOT r1.0] 6.7.2 Host expects to receive data from the device
         *
         * Clear the Bulk-In pipe and proceed
         */
        NN_CDMSC_WARN("BOT Data Phase Stall!\n");
        NN_CDMSC_DO(ClearBulkIn());
    }
    else if (result.IsFailure()) // USB layer error
    {
        NN_CDMSC_WARN("BOT Data Phase Error %d:%d\n",
                      result.GetModule(), result.GetDescription());
        result = ResultUsbTransactionError();
    }

    return result;
}

/*
 * BOT Data Phase (Out)
 *
 * ResultSuccess
 *   Everything is good.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *   BOT transaction error, and we are failed to bring it back to sync. Device
 *   reset is inevitable.
 *
 * ResultUsbTransactionError
 *   USB transaction failed, we can retry the status phase.
 */
Result
Bot::DoDataOut(void *buffer, uint32_t size)
{
    Result result = ResultSuccess();
    size_t xferredSize = 0;

    NN_CDMSC_DO(
        m_BulkOut.PostBuffer(&xferredSize, buffer, size)
    );

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (nn::usb::ResultStalled::Includes(result))
    {
        /*
         * Ref [BOT r1.0] 6.7.3
         *
         * Clear the Bulk-Out pipe and proceed
         */
        NN_CDMSC_WARN("BOT Data Phase Stall!\n");
        NN_CDMSC_DO(ClearBulkOut());
    }
    else if (result.IsFailure()) // USB layer error
    {
        NN_CDMSC_WARN("BOT Data Phase Error %d:%d\n",
                      result.GetModule(), result.GetDescription());
        result = ResultUsbTransactionError();
    }

    return result;
}

/*
 * BOT Status Phase
 *
 * ResultSuccess
 *   Everything is good.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *   BOT transaction error, and we are failed to bring it back to sync. Device
 *   reset is inevitable.
 *
 * ResultUsbTransactionError
 *   USB transaction failed, we can retry the status phase.
 *
 * ResultBotFailed
 *   BOT transaction failed, but we are still in sync. BOT retry is Okay.
 *
 * TODO: Review
 *   - [BOT r1.0] Figure 2 specifies 1 STALL retry
 *   - [BOT r1.0] 6.7 is ambiguous on STALL retry (infinite retry?)
 *   - Cafe does multiple STALL retries
 *   - BSD does 1 retry, regardless STALL or not
 * For now, we choose to do 1 STALL retry.
 */
Result
Bot::DoStatus(Cbw& cbw, Csw& csw)
{
    Result result = ResultSuccess();
    size_t xferredSize = 0;

    int stallRetry = 1;

    while (true)
    {
        NN_CDMSC_DO(m_BulkIn.PostBuffer(&xferredSize, m_pBuffer, sizeof(Csw)));

        if (nn::usb::ResultInterfaceInvalid::Includes(result))
        {
            result = ResultDeviceNotAvailable();
        }
        else if (nn::usb::ResultStalled::Includes(result))
        {
            NN_CDMSC_WARN("BOT Status Phase Stall!\n");
            if (stallRetry--)
            {
                NN_CDMSC_DO(ClearBulkIn());
                if (result.IsSuccess())
                {
                    // Retry
                    NN_CDMSC_WARN("Retry BOT Status Phase...\n");
                    continue;
                }
            }
            else
            {
                NN_CDMSC_DO(ResetRecovery());
                if (result.IsSuccess())
                {
                    result = ResultBotFailed();
                }
            }
        }
        else if (result.IsFailure()) // USB layer error
        {
            result = ResultUsbTransactionError();
        }

        break;
    }

    if (result.IsSuccess())
    {
        /*
         * Ref [BOT r1.0] 6.3 Valid and Meaningful CSW
         */
        bool  isCswValid      = false;
        bool  isCswMeaningful = false;
        Csw  *pCsw = reinterpret_cast<Csw*>(m_pBuffer);

        if (xferredSize         == sizeof(Csw)  &&
            pCsw->dCSWSignature == CswSignature &&
            pCsw->dCSWTag       == cbw.dCBWTag   )
        {
            isCswValid = true;

            switch (pCsw->bCSWStatus)
            {
            case CswStatus_CommandPassed:
            case CswStatus_CommandFailed:
                if (pCsw->dCSWDataResidue <= cbw.dCBWDataTransferLength)
                {
                    isCswMeaningful = true;
                }
                break;

            case CswStatus_PhaseError:
                isCswMeaningful = true;
                break;

            default:
                isCswMeaningful = false;
                break;
            }
        }

        /*
         * Ref [BOT r1.0] 6.5 Host Error Handling
         */
        if (isCswValid && isCswMeaningful)
        {
            std::memcpy(&csw, pCsw, xferredSize);
        }
        else
        {
            NN_CDMSC_DO(ResetRecovery());
            if (result.IsSuccess())
            {
                result = ResultBotFailed();
            }
        }
    }

    return result;
}

/*
 * Execute one single BOT transaction, with phase retry
 *
 * ResultSuccess
 *   Wire Protocol (i.e. BOT) succeed, but Command Protocol (e.g. SCSI, UFI,
 *   etc) could fail. csw.bCSWStatus is 0 or 1.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *   BOT transaction error, and we are failed to bring it back to sync. Device
 *   reset is inevitable.
 *
 * ResultBotFailed
 *   BOT transaction failed, but we are still in sync. BOT retry is Okay.
 */
Result
Bot::Execute(Cbw& cbw, void *buffer, Csw& csw)
{
    Result result = ResultSuccess();

    m_CbwTag++;

    // Command Phase
    for (int i = 0; i < BotPhaseRetryMax; i++)
    {
        NN_CDMSC_DO(DoCommand(cbw));

        if (ResultUsbTransactionError::Includes(result))
        {
            // Retry
            continue;
        }

        break;
    }

    if (result.IsFailure())  // Abort
    {
        if (ResultUsbTransactionError::Includes(result))
        {
            // Force reset
            result = ResultBotOutOfSync();
        }
        return result;
    }

    // Data Phase
    if (cbw.dCBWDataTransferLength != 0)
    {
        for (int i = 0; i < BotPhaseRetryMax; i++)
        {
            if (cbw.bmCBWFlags == CbwFlags_DataIn)
            {
                NN_CDMSC_DO(DoDataIn(buffer, cbw.dCBWDataTransferLength));
            }
            else
            {
                NN_CDMSC_DO(DoDataOut(buffer, cbw.dCBWDataTransferLength));
            }

            if (ResultUsbTransactionError::Includes(result))
            {
                // Retry
                continue;
            }

            break;
        }

        if (result.IsFailure()) // Abort
        {
            if (ResultUsbTransactionError::Includes(result))
            {
                // Force reset
                result = ResultBotOutOfSync();
            }
            return result;
        }
    }

    // Status Phase
    for (int i = 0; i < BotPhaseRetryMax; i++)
    {
        NN_CDMSC_DO(DoStatus(cbw, csw));

        if (ResultUsbTransactionError::Includes(result))
        {
            // Retry
            continue;
        }

        break;
    }

    if (result.IsFailure())  // Abort
    {
        if (ResultUsbTransactionError::Includes(result))
        {
            // Force reset
            result = ResultBotOutOfSync();
        }
        return result;
    }

    /*
     * Ref [BOT r1.0] 6.7 The Thirteen Cases
     */
    if (csw.bCSWStatus == CswStatus_CommandPassed ||
        csw.bCSWStatus == CswStatus_CommandFailed  )
    {
        if (cbw.dCBWDataTransferLength == 0)
        {
            // This should always be true because CSW must be meaningful
            NN_CDMSC_ABORT_UNLESS(csw.dCSWDataResidue == 0);

            // Case (1), good, do nothing
            result = ResultSuccess();
        }
        else
        {
            // This should always be true because CSW must be meaningful
            NN_CDMSC_ABORT_UNLESS(csw.dCSWDataResidue <= cbw.dCBWDataTransferLength);

            // Case (4), (5), (6), (9), (11), (12)
            //
            // Caller determines the amount of data that was processed from
            // the difference of dCBWDataTransferLength and the value of
            // dCSWDataResidue.
            result = ResultSuccess();
        }
    }
    else // csw.bCSWStatus == CswStatus_CommandPhaseError
    {
        // Case (2), (3), (7), (8), (10), (13)
        NN_CDMSC_DO(ResetRecovery());
        if (result.IsSuccess())
        {
            result = ResultBotFailed();
        }
    }

    return result;
}

/*
 * Ref [BOT r1.0] 3.1
 */
Result
Bot::StorageReset()
{
    Result result = ResultSuccess();

    size_t xferredSize = 0;

    result = m_pInterface->ControlRequest(
        &xferredSize,
        m_pBuffer,
        BmRequestType(Class, Interface, HostToDevice),
        255,
        0,
        m_IfProfile.ifDesc.bInterfaceNumber,
        0
    );

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        result = ResultBotOutOfSync();
    }

    return result;
}

/*
 * Ref [BOT r1.0] 3.2
 */
Result
Bot::GetMaxLun(uint8_t *pMaxLun)
{
    Result result = ResultSuccess();

    size_t xferredSize = 0;

    result = m_pInterface->ControlRequest(
        &xferredSize,
        m_pBuffer,
        BmRequestType(Class, Interface, DeviceToHost),
        254,
        0,
        m_IfProfile.ifDesc.bInterfaceNumber,
        1
    );

    if (result.IsSuccess() && xferredSize == 1)
    {
        /*
         * Ref [BOT r1.0] 3.2
         *   "Logical Unit Numbers on the device shall be numbered contiguously
         *    starting from LUN 0 to a maximum LUN of 15 (Fh)."
         */
        if (m_pBuffer[0] <= MaxLun)
        {
            *pMaxLun = m_pBuffer[0];
        }
        else
        {
            result = ResultInvalidLun();
        }
    }
    else if (nn::usb::ResultStalled::Includes(result))
    {
        /*
         * Ref [Bot r1.0] 3.2
         *   "Devices that do not support multiple LUNs may STALL this command."
         */
        *pMaxLun = 0;
        result = ResultSuccess();
    }
    else if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }

    return result;
}

/*
 * Clear Feature HALT to the Bulk-In endpoint
 */
Result
Bot::ClearBulkIn()
{
    Result result = ResultSuccess();

    result = m_BulkIn.ClearEndpointHalt();
    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        result = ResultBotOutOfSync();
    }

    return result;
}

/*
 * Clear Feature HALT to the Bulk-Out endpoint
 */
Result
Bot::ClearBulkOut()
{
    Result result = ResultSuccess();

    result = m_BulkOut.ClearEndpointHalt();
    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        result = ResultBotOutOfSync();
    }

    return result;
}

/*
 * Ref [BOT r1.0] 5.3.4
 *
 * ResultSuccess
 *    Succeed to bring the device back to sync.
 *
 * ResultDeviceNotAvailable
 *    Device is detached, nothing we can do.
 *
 * ResultBotOutOfSync
 *    Failed to perform UMS class specific request. Device reset is needed.
 */
Result
Bot::ResetRecovery()
{
    Result result = ResultSuccess();

    NN_CDMSC_RETURN_UPON_ERROR(StorageReset());
    NN_CDMSC_RETURN_UPON_ERROR(ClearBulkIn());
    NN_CDMSC_RETURN_UPON_ERROR(ClearBulkOut());

    return result;
}

} // driver
} // cdmsc
} // nn
