﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidUsb_Private.h"

namespace nn {
namespace cdhid {
namespace usb {

//////////////////////////////////////////////////////////////////////////////
CtrlSession::CtrlSession(UsbHidInterface *pUsbHidInterface, uint32_t deviceHandle) NN_NOEXCEPT
: m_StateChangeEvent(nn::os::EventClearMode_AutoClear, true)
{
    m_pUsbHidInterface  = pUsbHidInterface;
    m_DeviceHandle      = deviceHandle;
}


//////////////////////////////////////////////////////////////////////////////
CtrlSession::~CtrlSession() NN_NOEXCEPT
{
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::GetStateChangeEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle) NN_NOEXCEPT
{
    eventHandle.Set(nn::sf::NativeHandle(m_StateChangeEvent.GetReadableHandle(), false));
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::SignalStateChangeEvent() NN_NOEXCEPT
{
    m_StateChangeEvent.Signal();
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::GetString(uint32_t stringIndex, nn::sf::OutBuffer outBuffer) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidGetString(
                                        reinterpret_cast<uint8_t *>(outBuffer.GetPointerUnsafe()),
                                        outBuffer.GetSize(),
                                        stringIndex,
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::GetReport(uint32_t reportType, uint32_t reportId, nn::sf::OutBuffer outBuffer) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidGetReport(
                                        reportType,
                                        reportId,
                                        reinterpret_cast<uint8_t *>(outBuffer.GetPointerUnsafe()),
                                        outBuffer.GetSize(),
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::SetReport(uint32_t reportType, uint32_t reportId, nn::sf::InBuffer inBuffer) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidSetReport(
                                        reportType,
                                        reportId,
                                        reinterpret_cast<uint8_t *>(const_cast<char *>(inBuffer.GetPointerUnsafe())),
                                        inBuffer.GetSize(),
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::GetIdle(nn::sf::OutBuffer outBuffer, uint8_t reportId) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidGetIdle(
                                        reinterpret_cast<uint8_t *>(outBuffer.GetPointerUnsafe()),
                                        reportId,
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::SetIdle(uint8_t idle, uint8_t reportId) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidSetIdle(
                                        idle,
                                        reportId,
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::GetProtocol(nn::sf::OutBuffer outBuffer) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidGetProtocol(
                                        reinterpret_cast<uint8_t *>(outBuffer.GetPointerUnsafe()),
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::SetProtocol(uint8_t protocol) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidSetProtocol(
                                        protocol,
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::GetDescriptor(nn::sf::OutBuffer outBuffer, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidGetDescriptor(
                                        reinterpret_cast<uint8_t *>(outBuffer.GetPointerUnsafe()),
                                        bmRequestType,
                                        wValue,
                                        wIndex,
                                        outBuffer.GetSize(),
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::SetDescriptor(nn::sf::InBuffer inBuffer, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidSetDescriptor(
                                        reinterpret_cast<uint8_t *>(const_cast<char *>(inBuffer.GetPointerUnsafe())),
                                        bmRequestType,
                                        wValue,
                                        wIndex,
                                        inBuffer.GetSize(),
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::GetCodeBook(nn::sf::OutBuffer outBuffer) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidGetCodeBook(
                                        reinterpret_cast<uint8_t *>(outBuffer.GetPointerUnsafe()),
                                        outBuffer.GetSize(),
                                        m_DeviceHandle
                                        );
}


//////////////////////////////////////////////////////////////////////////////
Result CtrlSession::Write(nn::sf::InBuffer inBuffer, nn::sf::Out<std::uint32_t> pBytesWritten, nn::TimeSpan timeout) NN_NOEXCEPT
{
    return m_pUsbHidInterface->AhidWrite(
                                        pBytesWritten.GetPointer(),
                                        reinterpret_cast<uint8_t *>(const_cast<char *>(inBuffer.GetPointerUnsafe())),
                                        inBuffer.GetSize(),
                                        m_DeviceHandle,
                                        timeout
                                        );
}

} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
