﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidUsb_Private.h"

namespace nn {
namespace cdhid {
namespace usb {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result UsbHidCodeBook::Initialize(uint8_t* pHidReportDescriptor, uint16_t length) NN_NOEXCEPT
{
    nn::ahid::CodeBookHeader* pCodeBookHeader = reinterpret_cast<nn::ahid::CodeBookHeader*>(m_CodeBook);

    memset(m_CodeBook, 0, sizeof(m_CodeBook));

    m_pCodeBookWrite            = &m_CodeBook[sizeof(nn::ahid::CodeBookHeader)];
    m_CodeBookBytes             = sizeof(nn::ahid::CodeBookHeader);

    // - make 3 passes to encode the CodeBook
    // - input, output, feature, can appear in any order in the HID report descriptor
    // - the CodeBook needs to have the items in order of input, output, feature

    // Only generate report ID table on first pass :0
    m_GenerateReportIds = true;

    //NN_SDK_LOG("Input\n");
    ParseHidReportDescriptor(pHidReportDescriptor, length, MainItemTag::Input);
    pCodeBookHeader->inputSize = m_StartingBit[0] / 8;
    //NN_SDK_LOG("Size %d\n", pCodeBookHeader->inputSize);

    m_GenerateReportIds = false;

    //NN_SDK_LOG("Output\n");
    ParseHidReportDescriptor(pHidReportDescriptor, length, MainItemTag::Output);
    pCodeBookHeader->outputSize = m_StartingBit[0] / 8;
    //NN_SDK_LOG("Size %d\n", pCodeBookHeader->outputSize);

    //NN_SDK_LOG("Feature\n");
    ParseHidReportDescriptor(pHidReportDescriptor, length, MainItemTag::Feature);
    pCodeBookHeader->featureSize = m_StartingBit[0] / 8;
    //NN_SDK_LOG("Size %d\n", pCodeBookHeader->featureSize);

    // If there were no report IDs then encode the one and only report
    if (pCodeBookHeader->reportCount == 0)
    {
        pCodeBookHeader->reportCount    = 1;
        pCodeBookHeader->reportId[0]    = 0;
        pCodeBookHeader->usagePage[0]   = pHidReportDescriptor[1];
        pCodeBookHeader->usage[0]       = pHidReportDescriptor[3];
    }

    if (m_CodeBookBytes > nn::ahid::CodeBook::CodeBook_Size)
    {
        NN_SDK_LOG("CodeBook size requires %d bytes, is truncated at %d\n", m_CodeBookBytes, nn::ahid::CodeBook::CodeBook_Size);
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidCodeBook::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result UsbHidCodeBook::GetCodeBook(uint8_t* pBuffer, uint32_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize >= nn::ahid::CodeBook::CodeBook_Size)
    {
        memcpy(pBuffer, m_CodeBook, nn::ahid::CodeBook::CodeBook_Size);

        return ResultSuccess();
    }

    return nn::ahid::ResultBufferSize();
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::ParseHidReportDescriptor(uint8_t* pHidReportDescriptor, uint16_t length, uint8_t tag) NN_NOEXCEPT
{
    int bytesRemaining = length;
    uint8_t* pRead = pHidReportDescriptor;

    m_GlobalItemsStackDepth = 0;

    ClearGlobalItems();
    ClearLocalItems();

    // initialize starting bit for all report IDs
    m_StartingBit[0] = 0;

    for (uint8_t i = 1; i < 255; i++)
    {
        m_StartingBit[i] = 8;
    }

    while (bytesRemaining > 0)
    {
        uint8_t bSize   = *pRead & 0x03;
        uint8_t bType   = (*pRead & 0x0c) >> 2;
        uint8_t bTag    = *pRead >> 4;

        //uint8_t skipBytes;

        if ((bSize == 0x02) && (bTag == 0xf0))  // "long item"
        {
            bSize   = pRead[1];
            bTag    = pRead[2];

            NN_SDK_LOG("long item bTag %02x bType %02x bSize %02x\n", bTag, bType, bSize);

            //skipBytes = bSize + 2;
        }
        else // "short item"
        {
            switch (bType)
            {
            case ItemType::Main:

                ParseMainItemShort(&pRead[1], bTag, bSize, tag);

                break;

            case ItemType::Global:

                ParseGlobalItemShort(&pRead[1], bTag, bSize);

                break;

            case ItemType::Local:

                ParseLocalItemShort(&pRead[1], bTag, bSize);

                break;

            default:

                NN_SDK_LOG("Unknown Item Type %d\n", bType);

                break;
            }
        }

        pRead           += SkipBytes[bSize];
        bytesRemaining  -= SkipBytes[bSize];
    }

    // get correct asking size. Use m_StartingBit[0] for storage of the largest
    for (uint8_t i = 0; i < 255;i++)
    {
        if (m_StartingBit[0] < m_StartingBit[i])
        {
            m_StartingBit[0] = m_StartingBit[i];
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::ParseMainItemShort(uint8_t* pDataParams, uint8_t bTag, uint8_t bLength, uint8_t tag) NN_NOEXCEPT
{
    if (bTag == tag)
    {
        switch (bTag)
        {
        case MainItemTag::Input:
        case MainItemTag::Output:
        case MainItemTag::Feature:

            CodeBookEntry(bTag, *pDataParams);

            break;

        case MainItemTag::Collection:
        case MainItemTag::EndCollection:

            break;

        default:

            NN_SDK_LOG("Unknown Main Item %d\n", bTag);

            break;
        }
    }

    ClearLocalItems();
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::ParseGlobalItemShort(uint8_t* pData, uint8_t bTag, uint8_t bLength) NN_NOEXCEPT
{
    int32_t data = GetData(pData, bLength);

    switch (bTag)
    {
    case GlobalItemTag::UsagePage:
    case GlobalItemTag::LogicalMinimum:
    case GlobalItemTag::LogicalMaximum:
    case GlobalItemTag::PhysicalMinimum:
    case GlobalItemTag::PhysicalMaximum:
    case GlobalItemTag::UnitExponent:
    case GlobalItemTag::Unit:
    case GlobalItemTag::ReportSize:
    case GlobalItemTag::ReportCount:

        m_GlobalItems[m_GlobalItemsStackDepth][bTag] = data;

        break;

    case GlobalItemTag::ReportId:

        m_GlobalItems[m_GlobalItemsStackDepth][bTag] = data;

        if (m_GenerateReportIds)
        {
            // Store params per report id in table
            nn::ahid::CodeBookHeader* pCodeBookHeader = reinterpret_cast<nn::ahid::CodeBookHeader*>(m_CodeBook);
            pCodeBookHeader->reportId[pCodeBookHeader->reportCount]     = data;
            pCodeBookHeader->usagePage[pCodeBookHeader->reportCount]    = m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::UsagePage];
            pCodeBookHeader->usage[pCodeBookHeader->reportCount]        = m_UsageReport;
            pCodeBookHeader->reportCount++;
        }

        break;

    case GlobalItemTag::Push:

        if (m_GlobalItemsStackDepth < (GlobalItems::StackDepth - 2))
        {
            for (int i = 0; i < GlobalItems::ParamsCount; i++)
            {
                m_GlobalItems[m_GlobalItemsStackDepth + 1][i] = GlobalItem(i);
            }

            m_GlobalItemsStackDepth++;
        }
        else
        {
            NN_SDK_LOG("Push exceeds GlobalItems::StackDepth\n");
        }

        break;

    case GlobalItemTag::Pop:

        if (m_GlobalItemsStackDepth)
        {
            m_GlobalItemsStackDepth--;
        }
        else
        {
            NN_SDK_LOG("Pop less than 0\n");
        }

        break;

    default:

        NN_SDK_LOG("Unknown Global Item %d\n", bTag);

        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::ParseLocalItemShort(uint8_t* pData, uint8_t bTag, uint8_t bLength) NN_NOEXCEPT
{
    int32_t data = GetData(pData, bLength);

    switch (bTag)
    {
    case LocalItemTag::Usage:

        m_Usage[m_UsageIndex++] = data;

        break;

    case LocalItemTag::UsageMinimum:

        m_UsageMinimum = data;

        break;

    case LocalItemTag::UsageMaximum:

        m_UsageMaximum = data;

        break;

    case LocalItemTag::DesignatorIndex:

        m_DesignatorIndex = data;

        break;

    case LocalItemTag::DesignatorMinimum:

        m_DesignatorMinimum = data;

        break;

    case LocalItemTag::DesignatorMaximum:

        m_DesignatorMaximum = data;

        break;

    case LocalItemTag::StringIndex:

        m_StringIndex = data;

        break;

    case LocalItemTag::StringMinimum:

        m_StringMinimum = data;

        break;

    case LocalItemTag::StringMaximum:

        m_StringMaximum = data;

        break;

    case LocalItemTag::Delimiter:

        break;

    default:

        NN_SDK_LOG("Unknown Local Item %d\n", bTag);

        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
int32_t UsbHidCodeBook::GetData(uint8_t* pData, uint8_t bSize) NN_NOEXCEPT
{
    int32_t data = 0;

    switch (bSize)
    {
    case ItemSize::Zero:

        break;

    case ItemSize::One:

        data = *pData;

        break;

    case ItemSize::Two:

        data = *pData++;
        data |= (*pData << 8);

        break;

    case ItemSize::Four:

        data = *pData++;
        data |= (*pData++ << 8);
        data |= (*pData++ << 16);
        data |= (*pData << 24);

        break;

    default:

        NN_SDK_LOG("unknown bSize %d\n", bSize);

        break;
    }

    return data;
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::CodeBookEntry(uint8_t bTag, uint8_t dataParams) NN_NOEXCEPT
{
    for (int32_t i = 0; i < GlobalItem(GlobalItemTag::ReportCount); i++)
    {
        uint16_t usage;

        // If there are Usage data stored then use the data
        if (m_UsageIndex)
        {
            usage = (m_UsageIndex > i) ? m_Usage[i] : m_Usage[m_UsageIndex - 1];
        }
        else
        {
            usage = 0;
        }

        if (usage || m_UsageMinimum || m_UsageMaximum)
        {
            uint8_t     index = 0;
            uint16_t    usageMin, usageMax;

            if ((dataParams & 0x02) == 0) // is array
            {
                index = i;

                if (usage == 0)
                {
                    usageMin = m_UsageMinimum;
                    usageMax = m_UsageMaximum;
                }
                else
                {
                    usageMin = usage;
                    usageMax = usage;
                }
            }
            else
            {
                if (usage == 0)
                {
                    usageMin = m_UsageMinimum + i;
                    usageMax = usageMin;
                }
                else
                {
                    usageMin = usage;
                    usageMax = usage;
                }
            }

            //NN_SDK_LOG("report id %d usage %04x %04x starting bit %d bits %d index %d\n", GlobalItem(GlobalItemTag::ReportId), usageMin, usageMax, StartingBit(), GlobalItem(GlobalItemTag::ReportSize), index);

            if (GlobalItem(GlobalItemTag::ReportSize) == 1) // 1 bit data
            {
                AddItemDataBit(usageMin, usageMax, bTag, dataParams, index);
            }
            else if (GlobalItem(GlobalItemTag::ReportSize) <= 8)
            {
                AddItemData8(usageMin, usageMax, bTag, dataParams, index);
            }
            else if (GlobalItem(GlobalItemTag::ReportSize) <= 16)
            {
                AddItemData16(usageMin, usageMax, bTag, dataParams, index);
            }
            else
            {
                NN_SDK_LOG("unhanded ReportSize %d\n", GlobalItem(GlobalItemTag::ReportSize));
            }
        }
        else
        {
            //NN_SDK_LOG("skip %d\n", m_ReportSize);
        }

        m_StartingBit[GlobalItem(GlobalItemTag::ReportId)] += GlobalItem(GlobalItemTag::ReportSize);
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::AddEntry(void* pData, uint8_t bytes, uint8_t bTag)
{
    // Only print up to nn::ahid::CodeBook::CodeBook_Size
    switch (bTag)
    {
    // Only print items we care about
    case MainItemTag::Input:
    case MainItemTag::Output:
    case MainItemTag::Feature:

        if (m_CodeBookBytes + bytes <= nn::ahid::CodeBook::CodeBook_Size)
        {
            memcpy(m_pCodeBookWrite, pData, bytes);
            m_pCodeBookWrite += bytes;

            AddEntryToHeader(bTag);
        }

        // Always keep in track of number of bytes to print warning
        m_CodeBookBytes += bytes;

        break;

    default:

        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::AddItemDataBit(uint16_t usageMin, uint16_t usageMax, uint8_t bTag, uint8_t dataParams, uint8_t index)
{
    nn::ahid::ItemBitmask8 itemBitmask8;

    int32_t startingBit = StartingBit();

    itemBitmask8.item.type      = nn::ahid::DataType::DataType_Bitmask8;
    itemBitmask8.item.length    = sizeof(itemBitmask8);
    itemBitmask8.item.reportId  = GlobalItem(GlobalItemTag::ReportId);
    itemBitmask8.item.usagePage = GlobalItem(GlobalItemTag::UsagePage);
    itemBitmask8.item.index     = index;
    itemBitmask8.item.usageMin  = usageMin;
    itemBitmask8.item.usageMax  = usageMax;

    itemBitmask8.dataOffset     = startingBit / 8;
    itemBitmask8.dataBits       = 1 << (startingBit % 8);
    itemBitmask8.dataShift      = startingBit % 8;

    AddEntry(&itemBitmask8, sizeof(itemBitmask8), bTag);
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::AddItemData8(uint16_t usageMin, uint16_t usageMax, uint8_t bTag, uint8_t dataParams, uint8_t index)
{
    int32_t startingBit = StartingBit();

    if (m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::Unit]) // logical->physical->unit+exponent
    {
        if (m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::ReportSize] < 8)   // use mask
        {
            nn::ahid::ItemPhysical8Mask itemPhysical8Mask;

            itemPhysical8Mask.item.type         = nn::ahid::DataType::DataType_Physical8Mask;
            itemPhysical8Mask.item.length       = sizeof(itemPhysical8Mask);
            itemPhysical8Mask.item.reportId     = GlobalItem(GlobalItemTag::ReportId);
            itemPhysical8Mask.item.usagePage    = GlobalItem(GlobalItemTag::UsagePage);
            itemPhysical8Mask.item.index        = index;
            itemPhysical8Mask.item.usageMin     = usageMin;
            itemPhysical8Mask.item.usageMax     = usageMax;

            itemPhysical8Mask.dataOffset        = startingBit / 8;
            itemPhysical8Mask.dataBits          = DataBits8[GlobalItem(GlobalItemTag::ReportSize)] << (startingBit % 8);
            itemPhysical8Mask.dataShift         = startingBit % 8;
            itemPhysical8Mask.logicalMinimum    = GlobalItem(GlobalItemTag::LogicalMinimum);
            itemPhysical8Mask.logicalMaximum    = GlobalItem(GlobalItemTag::LogicalMaximum);
            itemPhysical8Mask.physicalMinimum   = GlobalItem(GlobalItemTag::PhysicalMinimum);
            itemPhysical8Mask.physicalMaximum   = GlobalItem(GlobalItemTag::PhysicalMaximum);
            itemPhysical8Mask.unitExponont      = GlobalItem(GlobalItemTag::UnitExponent);
            itemPhysical8Mask.unit              = GlobalItem(GlobalItemTag::Unit);

            AddEntry(&itemPhysical8Mask, sizeof(itemPhysical8Mask), bTag);
        }
        else
        {
            nn::ahid::ItemPhysical8 itemPhysical8;

            itemPhysical8.item.type             = nn::ahid::DataType::DataType_Physical8;
            itemPhysical8.item.length           = sizeof(itemPhysical8);
            itemPhysical8.item.reportId         = GlobalItem(GlobalItemTag::ReportId);
            itemPhysical8.item.usagePage        = GlobalItem(GlobalItemTag::UsagePage);
            itemPhysical8.item.index            = index;
            itemPhysical8.item.usageMin         = usageMin;
            itemPhysical8.item.usageMax         = usageMax;

            itemPhysical8.dataOffset            = startingBit / 8;
            itemPhysical8.logicalMinimum        = GlobalItem(GlobalItemTag::LogicalMinimum);
            itemPhysical8.logicalMaximum        = GlobalItem(GlobalItemTag::LogicalMaximum);
            itemPhysical8.physicalMinimum       = GlobalItem(GlobalItemTag::PhysicalMinimum);
            itemPhysical8.physicalMaximum       = GlobalItem(GlobalItemTag::PhysicalMaximum);
            itemPhysical8.unitExponont          = GlobalItem(GlobalItemTag::UnitExponent);
            itemPhysical8.unit                  = GlobalItem(GlobalItemTag::Unit);

            AddEntry(&itemPhysical8, sizeof(itemPhysical8), bTag);
        }
    }
    else // logical
    {
        if (m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::ReportSize] < 8)   // use mask
        {
            nn::ahid::ItemLogical8Mask itemLogical8Mask;

            itemLogical8Mask.item.type          = nn::ahid::DataType::DataType_Logical8Mask;
            itemLogical8Mask.item.length        = sizeof(itemLogical8Mask);
            itemLogical8Mask.item.reportId      = GlobalItem(GlobalItemTag::ReportId);
            itemLogical8Mask.item.usagePage     = GlobalItem(GlobalItemTag::UsagePage);
            itemLogical8Mask.item.index         = index;
            itemLogical8Mask.item.usageMin      = usageMin;
            itemLogical8Mask.item.usageMax      = usageMax;

            itemLogical8Mask.dataOffset         = startingBit / 8;
            itemLogical8Mask.dataBits           = DataBits8[GlobalItem(GlobalItemTag::ReportSize)] << (startingBit % 8);
            itemLogical8Mask.dataShift          = startingBit % 8;
            itemLogical8Mask.logicalMinimum     = GlobalItem(GlobalItemTag::LogicalMinimum);
            itemLogical8Mask.logicalMaximum     = GlobalItem(GlobalItemTag::LogicalMaximum);

            AddEntry(&itemLogical8Mask, sizeof(itemLogical8Mask), bTag);
        }
        else
        {
            nn::ahid::ItemLogical8 itemLogical8;

            itemLogical8.item.type              = nn::ahid::DataType::DataType_Logical8;
            itemLogical8.item.length            = sizeof(itemLogical8);
            itemLogical8.item.reportId          = GlobalItem(GlobalItemTag::ReportId);
            itemLogical8.item.usagePage         = GlobalItem(GlobalItemTag::UsagePage);
            itemLogical8.item.index             = index;
            itemLogical8.item.usageMin          = usageMin;
            itemLogical8.item.usageMax          = usageMax;

            itemLogical8.dataOffset             = startingBit / 8;
            itemLogical8.logicalMinimum         = GlobalItem(GlobalItemTag::LogicalMinimum);
            itemLogical8.logicalMaximum         = GlobalItem(GlobalItemTag::LogicalMaximum);

            AddEntry(&itemLogical8, sizeof(itemLogical8), bTag);
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::AddItemData16(uint16_t usageMin, uint16_t usageMax, uint8_t bTag, uint8_t dataParams, uint8_t index)
{
    int32_t startingBit = StartingBit();

    if (m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::Unit]) // logical->physical->unit+exponent
    {
NN_SDK_LOG("not handeled m_Unit, need to add new item type to AHID\n");
        if (GlobalItem(GlobalItemTag::ReportSize) < 16)   // use mask
        {
        }
        else
        {

        }
    }
    else // logical
    {
        if (GlobalItem(GlobalItemTag::ReportSize) < 16)   // use mask
        {
            nn::ahid::ItemLogical16Mask itemLogical16Mask;

            itemLogical16Mask.item.type         = nn::ahid::DataType::DataType_Logical16Mask;
            itemLogical16Mask.item.length       = sizeof(itemLogical16Mask);
            itemLogical16Mask.item.reportId     = GlobalItem(GlobalItemTag::ReportId);
            itemLogical16Mask.item.usagePage    = GlobalItem(GlobalItemTag::UsagePage);
            itemLogical16Mask.item.index        = index;
            itemLogical16Mask.item.usageMin     = usageMin;
            itemLogical16Mask.item.usageMax     = usageMax;

            itemLogical16Mask.dataOffset        = startingBit / 8;

            itemLogical16Mask.dataBits          = DataBits16[GlobalItem(GlobalItemTag::ReportSize)] << (startingBit % 8);
            itemLogical16Mask.dataShift         = startingBit % 8;
            itemLogical16Mask.signBit           = 1 << (GlobalItem(GlobalItemTag::ReportSize) - 1);
            itemLogical16Mask.logicalMinimum    = GlobalItem(GlobalItemTag::LogicalMinimum);
            itemLogical16Mask.logicalMaximum    = GlobalItem(GlobalItemTag::LogicalMaximum);

            AddEntry(&itemLogical16Mask, sizeof(itemLogical16Mask), bTag);
        }
        else
        {
            nn::ahid::ItemLogical16 itemLogical16;

            itemLogical16.item.type             = nn::ahid::DataType::DataType_Logical16;
            itemLogical16.item.length           = sizeof(itemLogical16);
            itemLogical16.item.reportId         = GlobalItem(GlobalItemTag::ReportId);
            itemLogical16.item.usagePage        = GlobalItem(GlobalItemTag::UsagePage);
            itemLogical16.item.index            = 0;
            itemLogical16.item.usageMin         = usageMin;
            itemLogical16.item.usageMax         = usageMax;

            itemLogical16.dataOffset            = startingBit / 8;
            itemLogical16.logicalMinimum        = GlobalItem(GlobalItemTag::LogicalMinimum);
            itemLogical16.logicalMaximum        = GlobalItem(GlobalItemTag::LogicalMaximum);

            AddEntry(&itemLogical16, sizeof(itemLogical16), bTag);
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::AddEntryToHeader(uint8_t bTag)
{
    nn::ahid::CodeBookHeader* pCodeBookHeader = reinterpret_cast<nn::ahid::CodeBookHeader*>(m_CodeBook);

    switch(bTag)
    {
    case MainItemTag::Input:

        pCodeBookHeader->inputItems++;

        break;

    case MainItemTag::Output:

        pCodeBookHeader->outputItems++;

        break;

    case MainItemTag::Feature:

        pCodeBookHeader->featureItems++;

        break;

    default:

        NN_SDK_LOG("unexpected bTag %d %s\n", bTag, __FUNCTION__);

        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
int32_t UsbHidCodeBook::GlobalItem(int index) NN_NOEXCEPT
{
    if (index < ParamsCount)
    {
        return m_GlobalItems[m_GlobalItemsStackDepth][index];
    }

    NN_SDK_LOG("unexpected index %d %s\n", index, __FUNCTION__);

    return 0;
}


//////////////////////////////////////////////////////////////////////////////
int32_t UsbHidCodeBook::StartingBit() NN_NOEXCEPT
{
    return m_StartingBit[GlobalItem(GlobalItemTag::ReportId)];
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::ClearGlobalItems() NN_NOEXCEPT
{
    for (int i = 0; i < GlobalItems::ParamsCount; i++)
    {
        m_GlobalItems[m_GlobalItemsStackDepth][i] = 0;
    }
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::ClearLocalItems() NN_NOEXCEPT
{
    // Store base usage for report ID support
    m_UsageReport = m_Usage[0];

    m_Usage[0]          = 0;
    m_UsageMinimum      = 0;
    m_UsageMaximum      = 0;
    m_DesignatorIndex   = 0;
    m_DesignatorMinimum = 0;
    m_DesignatorMaximum = 0;
    m_StringIndex       = 0;
    m_StringMinimum     = 0;
    m_StringMaximum     = 0;

    m_UsageIndex        = 0;
}


//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::PrintGlobalItems() NN_NOEXCEPT
{
    NN_SDK_LOG("UsagePage           %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::UsagePage]);
    NN_SDK_LOG("LogicalMinimum      %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::LogicalMinimum]);
    NN_SDK_LOG("LogicalMaximum      %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::LogicalMaximum]);
    NN_SDK_LOG("PhysicalMinimum     %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::PhysicalMinimum]);
    NN_SDK_LOG("PhysicalMaximum     %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::PhysicalMaximum]);
    NN_SDK_LOG("UnitExponent        %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::UnitExponent]);
    NN_SDK_LOG("Unit                %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::Unit]);
    NN_SDK_LOG("ReportSize          %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::ReportSize]);
    NN_SDK_LOG("ReportId            %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::ReportId]);
    NN_SDK_LOG("ReportCount         %d\n", m_GlobalItems[m_GlobalItemsStackDepth][GlobalItemTag::ReportCount]);
}

//////////////////////////////////////////////////////////////////////////////
void UsbHidCodeBook::PrintLocalItems() NN_NOEXCEPT
{
    NN_SDK_LOG("m_Usage             %d index %d\n", m_Usage[m_UsageIndex], m_UsageIndex);
    NN_SDK_LOG("m_UsageMinimum      %d\n", m_UsageMinimum);
    NN_SDK_LOG("m_UsageMaximum      %d\n", m_UsageMaximum);
    NN_SDK_LOG("m_DesignatorIndex   %d\n", m_DesignatorIndex);
    NN_SDK_LOG("m_DesignatorMinimum %d\n", m_DesignatorMinimum);
    NN_SDK_LOG("m_DesignatorMaximum %d\n", m_DesignatorMaximum);
    NN_SDK_LOG("m_StringIndex       %d\n", m_StringIndex);
    NN_SDK_LOG("m_StringMinimum     %d\n", m_StringMinimum);
    NN_SDK_LOG("m_StringMaximum     %d\n", m_StringMaximum);
}

} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
